/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "singlecontainer.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"

wftk::SingleContainer::SingleContainer(unsigned top_pad,
	unsigned bottom_pad, unsigned left_pad, unsigned right_pad)
	: top_pad_(top_pad), bottom_pad_(bottom_pad),
	  left_pad_(left_pad), right_pad_(right_pad),
	  child_(0)
{
  setPackingInfo();
}

void wftk::SingleContainer::setPadding(unsigned top_pad, unsigned bottom_pad,
	unsigned left_pad, unsigned right_pad)
{
  top_pad_ = top_pad;
  bottom_pad_ = bottom_pad;
  left_pad_ = left_pad;
  right_pad_ = right_pad;

  packingUpdate();
}

void wftk::SingleContainer::setPackingInfo()
{
  // uses background size
  Widget::setPackingInfo();

  PackingInfo child_info;

  if(child_)
    child_info = child_->getPackingInfo();

  child_info.x.min += left_pad_ + right_pad_;
  child_info.x.pref += left_pad_ + right_pad_;
  child_info.y.min += top_pad_ + bottom_pad_;
  child_info.y.pref += top_pad_ + bottom_pad_;

  packing_info_.x.contain(child_info.x);
  packing_info_.y.contain(child_info.y);
}

void wftk::SingleContainer::handleResize(Uint16 w, Uint16 h)
{
  Widget::handleResize(w, h);

  if(child_) {
    Rect copy(0, 0, w, h);

    copy.x += left_pad_;
    copy.w -= left_pad_ + right_pad_;
    copy.y += top_pad_;
    copy.h -= top_pad_ + bottom_pad_;

    PackingInfo child_info = child_->getPackingInfo();

    if(!child_info.x.expand && copy.w > child_info.x.pref) {
      copy.x += (copy.w - child_info.x.pref) / 2;
      copy.w = child_info.x.pref;
    }
    if(!child_info.y.expand && copy.h > child_info.y.pref) {
      copy.y += (copy.h - child_info.y.pref) / 2;
      copy.h = child_info.y.pref;
    }

    Debug::channel(Debug::PACKING) << "Resizing child of SingleContainer sized "
	<< getRect() << " to " << copy << Debug::endl;

    child_->resize(copy);
  }
}

void wftk::SingleContainer::pack(ScreenArea* child)
{
  if(child_)
    child_->setParent(0);

  child_ = child;
  child->setParent(this);
}

void wftk::SingleContainer::clear()
{
  if(child_)
    child_->setParent(0);

  child_ = 0;
}
