/*
   RAW routines
   --------------------------------------------------------------------
   VCHE - Virtual Console Hex Editor

   Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

/*
 * Draws a box at 'x', 'y' with 'xlen' width and 'ylen' height
 */
void raw_box( int x, int y, int xlen, int ylen, int attr)
{
 char *buffer;
 int j;

 buffer = ( char *) allocate_mem( vc_cols);

 term_color( attr);

 term_enable_graf_map();

 memset( buffer, RAW_LINE_HORIZ, xlen - 1);
 buffer[        0] = RAW_LINE_TOPLEFT;
 buffer[ xlen - 1] = RAW_LINE_TOPRIGHT;

 term_goto( x, y);
 write( fileno( stdout), buffer, xlen);

 buffer[        0] = RAW_LINE_BOTTOMLEFT;
 buffer[ xlen - 1] = RAW_LINE_BOTTOMRIGHT;

 term_goto( x, y + ylen - 1);
 write( fileno( stdout), buffer, xlen);

 memset( buffer, RAW_LINE_NOTHING, xlen - 1);
 buffer[ 0] = buffer[ xlen - 1] = RAW_LINE_VERT;

 for( j = 1; j < ylen - 1; j++) {
       term_goto( x, y + j);
       write( fileno( stdout), buffer, xlen); 
 }

 free( buffer);
 term_disable_graf_map();
}

/*
 * Write a string to the output buffer
 */
void raw_buffer_puts( int x, int y, int size, char *str, int attr)
{
 int co2 = vc_cols * 2;
 int pos = 2 *( y * vc_cols + x);
 int i   = 0;

 while( i < size * 2) {
      obuffer[ pos + i] = restore_buffer[ pos + i + co2] = *str++;
      i++;
      obuffer[ pos + i] = restore_buffer[ pos + i + co2] = attr;
      i++;
 }
}

/* 
 * Write a string
 */
void raw_puts( int x, int y, int size, char *str)
{
 int pos = 2 *( y * vc_cols + x);
 int i   = 0;

 term_goto( x, y);
 write( fileno( stdout), str, size);

 while( i < size * 2 && !stop_restore) {
        restore_buffer[ pos + i++] = *str++;
        restore_buffer[ pos + i++] = current_color;
 }

}

void raw_putc( int x, int y, char byte)
{
 int pos = 2 *( y * vc_cols + x);

 term_goto( x, y);
 write( fileno( stdout), &byte, 1);

 if( !stop_restore) {
     restore_buffer[ pos    ] = byte;
     restore_buffer[ pos + 1] = current_color;
 }
}

void raw_color_putc( int x, int y, char byte, int attr)
{
 term_color( attr);
 raw_putc( x, y, byte);
}

void raw_color_puts( int x, int y, int size, char *str, int attr)
{
 term_color( attr);
 raw_puts( x, y, size, str);
}

void raw_save_screen( void)
{
 stop_restore = 1;
}

void raw_load_screen( void)
{
 char *buffer;
 int i, j;
 int fcolor;
 int buffer_size         = vc_cols * vc_lines;
 int restore_buffer_size = vc_cols * 2 * vc_lines;

 i = 0;

 stop_restore = 0;

 buffer = ( char *) allocate_mem( buffer_size);

 term_goto( 0, 0);

 fcolor = restore_buffer[ 1];

 while( i < restore_buffer_size) {

        j = 0;

        term_color( fcolor);

        while( restore_buffer[ i + 1] == fcolor) {
            buffer[ j] = restore_buffer[ i];
            i += 2;
            j++;
        }

        write( fileno( stdout), buffer, j);

        fcolor = restore_buffer[ i + 1];
 }


 /* I prefer to do this, it's ugly but fast, the right way
    will be too slow, well... I dunno... I am open minded, if someone
    wants to send me a patch on this I will accept it :) */

 term_enable_graf_map();
 raw_putc( vc_cols - 26, 0, RAW_LINE_VERT);
 raw_putc( vc_cols - 19, 0, RAW_LINE_VERT);
 raw_putc( vc_cols - 5 , 0, RAW_LINE_VERT);
 term_disable_graf_map();

 fflush( stdout);
 free( buffer);
}

/*
 * Draws a dialog box.
 */
void raw_dialog_box( char *titlestr, char *valuestr, int size, int tattr, int vattr)
{
 char buf[ 2048];
 int titlelen, len;
 int x, y;

 titlelen = strlen( titlestr);
 len = titlelen > size ? titlelen : size;

 y = vc_lines / 2;

 sprintf( buf, "%-250.250s", valuestr);

 x = ( vc_cols - len - 4) / 2;
 raw_box( x, y - 2, len + 4, 4, tattr);

 x = ( vc_cols - len    ) / 2;
 raw_puts( x, y - 1, titlelen, titlestr);

 term_color( vattr);
 x = ( vc_cols - size   ) / 2;
 raw_puts( x, y, size, buf);
}

/*
 * Fill the screen with attribute 'attr'
 */
void raw_fill_screen( int attr)
{
 int restore_buffer_size = vc_cols * 2 * vc_lines;
 char *buffer;
 int size;
 int i;

 i = 0;
 size = (vc_lines * vc_cols);

 buffer = ( char *) allocate_mem( (size + 1) * sizeof( char));

 memset( buffer, ' ', size);

 term_goto( 0, 0);
 term_color( attr);

 write( fileno( stdout), buffer, size);

 while( i < restore_buffer_size && !stop_restore) {
        restore_buffer[ i++] = ' ';
        restore_buffer[ i++] = attr;
 }

 free( buffer);
}

/*
 * Shows an error (or attention) message box and wait for a key.
 */
void raw_show_alert_box( char *str, int addy)
{
 char *buf;
 int len;
 int x, y;

 len = strlen( str);
 buf = ( char *) allocate_mem( len + 64);

 strcpy( buf, str);

 raw_save_screen();

 if( len < 16) len = 16;
 if( len > vc_cols - 4) {
     len = vc_cols - 4;
     buf[ vc_cols - 5] = '>';
 }

 y = vc_lines / 2;
 y+= addy;

 x = ( vc_cols - len - 4) / 2;

 term_beep();

 raw_box(  x, y - 2, len + 4, 4, VC_COLOR_ATTENTION);

 x = ( vc_cols - len) / 2;
 raw_puts( x, y - 1, len, buf               );

 x = ( vc_cols -  16) / 2;
 raw_puts( x, y    , 16 , "Press any key...");

 key_wait_for_any();
 raw_load_screen();
 free( buf);
}

/*
 * Do a box, fill it with '\n' separated lines of text and wait for a key
 */
void raw_show_screentext( char *text, char *title)
{
 char *buffer;
 char *p = text;
 int y = 2, i = 0;

 buffer = ( char *) allocate_mem( vc_cols * 2);

 raw_save_screen();

 raw_box( 0, 0, vc_cols, vc_lines, VC_COLOR_BACKCYAN);

 raw_do_title( title);

 term_color( VC_COLOR_BACKCYAN);

 while( *p && y < (vc_lines - 3)) {

        /* 
         * I love C language :) 
         * Print the line if we reach an '\n' or the end of the screen
         */

        if( ((*p == '\n') ? *p++: 0) || i == vc_cols - 4) {
            raw_puts( 2, y++, i, buffer);
            i = 0;
            continue;
        }
        buffer[ i++] = *p++;
 }

 term_color( VC_COLOR_BACKGRAY);
 raw_puts( ( vc_cols - 18) / 2, vc_lines - 2, 18, " Press any key... ");

 key_wait_for_any();

 raw_load_screen();

 free( buffer);
}

/*
 * Write a title
 */
void raw_do_title( char *title)
{
 int len;
 int x;

 len = strlen( title);
 x = ( vc_cols - len) / 2;

 term_enable_graf_map();
 raw_putc( x -   1, 0, RAW_RTEE);
 raw_putc( x + len, 0, RAW_LTEE);
 term_disable_graf_map();

 term_color( VC_COLOR_BACKGRAY);
 raw_puts( x, 0, len, title);
}

/*
 * Write a centered bottom line
 */
void raw_put_bottom_line( char *text)
{
 int len;
 int x;
 int y;

 len = strlen( text);
 x = ( vc_cols - len) / 2;
 y = vc_lines - 1;

 term_color( VC_COLOR_BACKGRAY);
 raw_puts( 0, y, vc_cols, space_buffer);
 raw_puts( x, y, len, text);
}

/*
 * Flush the output buffer
 */
void raw_flush( void)
{
 char *buffer;
 int i, j;
 int fcolor;
 int obuffer_size = vc_cols * 2 * ( vc_lines - 2);
 int buffer_size  = vc_cols *     ( vc_lines - 2);

 i = 0;

 buffer = ( char *) allocate_mem( buffer_size);

 term_goto( 0, 1);

 fcolor = obuffer[ 1];

 while( i < obuffer_size) {

        j = 0;

        term_color( fcolor);

        while( obuffer[ i + 1] == fcolor) {
            buffer[ j] = obuffer[ i];
            i += 2;
            j++;
        }
        
        write( fileno( stdout), buffer, j);

        fcolor = obuffer[ i + 1];
 }

 fflush( stdout);
 free( buffer);
}

/*
 * Shows a question message, and let navigate options.
 */
int raw_ask( char *msg, char *q1, char *q2, char *q3, int posit)
{
 int lenmsg;
 int lenqs;
 int lq1,lq2,lq3;
 int len;
 int xq1, xq2, xq3;
 int x, y;
 int pos = posit;
 int c;

 raw_save_screen();
 term_hide_cursor();

 lq1 = strlen( q1);
 lq2 = strlen( q2);
 lq3 = strlen( q3);

 lenmsg = strlen( msg);
 lenqs  = lq1 + lq2 + lq3 + 2;

 len = lenmsg > lenqs ? lenmsg : lenqs;
 y = vc_lines / 2;

 x = ( vc_cols - len - 4) / 2;
 raw_box(  x, y - 2, len + 4, 4, VC_COLOR_ATTENTION);

 x = ( vc_cols - len) / 2;
 raw_puts( x, y - 1, lenmsg, msg);

 xq1 = ( vc_cols - lenqs) / 2;
 xq2 = xq1 + lq1 + 1;
 xq3 = xq2 + lq2 + 1;

 while( 1) {
        if( pos > 2) pos = 0;
        if( pos < 0) pos = 2;

        switch( pos) {
                case 0: term_color( VC_COLOR_BACKGRAY); 
                        raw_puts( xq1, y, lq1, q1);
                        term_color( VC_COLOR_ATTENTION);
                        raw_puts( xq2, y, lq2, q2);
                        raw_puts( xq3, y, lq3, q3);
                        break;
                case 1: term_color( VC_COLOR_ATTENTION);
                        raw_puts( xq1, y, lq1, q1);
                        term_color( VC_COLOR_BACKGRAY);
                        raw_puts( xq2, y, lq2, q2);
                        term_color( VC_COLOR_ATTENTION);
                        raw_puts( xq3, y, lq3, q3);
                        break;
                case 2: term_color( VC_COLOR_ATTENTION);
                        raw_puts( xq1, y, lq1, q1);
                        raw_puts( xq2, y, lq2, q2);
                        term_color( VC_COLOR_BACKGRAY);
                        raw_puts( xq3, y, lq3, q3);
                        break;
        }

        c = key_get();

        switch( c) {
                case '\n': raw_load_screen();
                           term_unhide_cursor();
                           return pos;
                case VC_KEY_RIGHT : pos++;
                                    break;
                case VC_KEY_LEFT  : pos--;
                                    break;
                case VC_KEY_HOME  : pos = 0;
                                    break;
                case VC_KEY_END   : pos = 2;
                                    break;
        } /* end switch( c) */
 } /* end while( 1) */

}

/*
 * Shows a message without waiting for a key
 */
void raw_show_message( char *msg, int addy)
{
 int len;
 int x, y;

 len = strlen( msg);

 y = ( vc_lines / 2) - 1;
 y += addy;

 x = ( vc_cols - len - 4) / 2;
 raw_box(  x, y - 1, len + 4, 3, VC_COLOR_ATTENTION);

 x = ( vc_cols - len) / 2;
 raw_puts( x, y   , len, msg);
}

void raw_fill_obuffer( void)
{
 int obuffer_size = vc_cols * 2 * ( vc_lines - 2);
 int co2          = vc_cols * 2;
 int i = 0;

 while( i < obuffer_size) {
        obuffer[ i] = restore_buffer[ i + co2] = ' ';
        i++;
        obuffer[ i] = restore_buffer[ i + co2] = VC_COLOR_BACKBLUE;
        i++;
 }
}

