/*
 * Copyright (c) 1991-1993 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the Computer Systems
 *	Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
static const char rcsid[] =
    "@(#) $Header: audio-voxware.cc,v 1.10 96/04/26 05:22:05 van Exp $ (LBL)";

#include <string.h>
#include <sys/fcntl.h>
#include <errno.h>
#if defined(sco) || defined(__bsdi__)
#include <sys/socket.h>
#endif
#if defined(__FreeBSD__)
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <machine/soundcard.h>
#else
#include <sys/soundcard.h>
#endif
#include "audio.h"
#include "Tcl.h"

#define ULAW_ZERO 0x7f
#define ABUFLOG2 8
#define ABUFLEN (1 << ABUFLOG2)
#define NFRAG 5

class VoxWareAudio : public Audio {
    public:
	VoxWareAudio();
	virtual int FrameReady();
	virtual u_char* Read();
	virtual	void Write(u_char *);
	virtual void SetRGain(int);
	virtual void SetPGain(int);
	virtual void OutputPort(int);
	virtual void InputPort(int);
	virtual void Obtain();
	virtual void RMute();
	virtual void RUnmute();
	virtual int HalfDuplex() const;
    protected:

	u_char* readptr;
	u_char* readbufend;
	u_char* readbuf;

	u_char* ubufptr;
	u_char* ubufend;
	u_char* ubuf;

	u_char* writeptr;
	u_char* writebufend;
	u_char* writebuf;
};

static class VoxWareAudioMatcher : public Matcher {
    public:
	VoxWareAudioMatcher() : Matcher("audio") {}
	TclObject* match(const char* fmt) {
		if (strcmp(fmt, "voxware") == 0)
			return (new VoxWareAudio);
		else
			return (0);
	}
} voxware_audio_matcher;

VoxWareAudio::VoxWareAudio()
{
	readbuf = new u_char[ABUFLEN];
	readptr = readbufend = readbuf + ABUFLEN;

	writeptr = writebuf = new u_char[ABUFLEN];
	writebufend = writebuf + ABUFLEN;

	ubufptr = ubuf = new u_char[blksize];
	ubufend = ubuf + blksize;
	memset(ubuf, ULAW_ZERO, blksize);
}

int VoxWareAudio::HalfDuplex() const
{
	/*XXX change this if full duplex audio device available*/
	return 1;
}

void VoxWareAudio::Obtain()
{
	if (HaveAudio())
		abort();

	fd = open("/dev/audio", O_RDWR|O_NDELAY);
	if (fd >= 0) {
		int on = 1;
		ioctl(fd, FIONBIO, &on);

		int frag = (NFRAG << 16) | ABUFLOG2;
		ioctl(fd, SNDCTL_DSP_SETFRAGMENT, &frag);
#ifdef fullduplex
		Audio::Obtain();
#else
		notify();
#endif
	}
}

void VoxWareAudio::Write(u_char *cp)
{
	if (HaveAudio() && (rmute & 1) != 0) {
		register u_char *cpend = cp + blksize;
		register u_char *wbuf = writeptr;
		register u_char *wend = writebufend;
		for ( ; cp < cpend; cp += 4) {
			wbuf[0] = cp[0];
			wbuf[1] = cp[1];
			wbuf[2] = cp[2];
			wbuf[3] = cp[3];
			wbuf += 4;
			if (wbuf >= wend) {
				wbuf = writebuf;
				if (write(fd, (char*)wbuf, ABUFLEN) != ABUFLEN)
					perror("aud write");
			}
		}
		writeptr = wbuf;
	}
}

int VoxWareAudio::FrameReady()
{
	if ((rmute & 1) == 0) {
		register u_char* cp = ubufptr;
		register u_char* cpend = ubufend;
		register u_char* rbuf = readptr;
		register u_char* rend = readbufend;

		for ( ; cp < cpend; cp += 4) {
			if (rbuf >= rend) {
				rbuf = readbuf;
				int cc = read(fd, (char*)rbuf, ABUFLEN);
				if (cc <= 0) {
					ubufptr = cp;
					readbufend = rbuf;
					if (cc == -1 && errno != EAGAIN) {
						Release();
						Obtain();
					}
					return (0);
				}
				readbufend = rend = rbuf + cc;
			}
			cp[0] = rbuf[0];
			cp[1] = rbuf[1];
			cp[2] = rbuf[2];
			cp[3] = rbuf[3];
			rbuf += 4;
		}
		readptr = rbuf;
	}
	return (1);
}

u_char* VoxWareAudio::Read()
{
	u_char* cp = ubuf;
	ubufptr = cp;
	return (cp);
}

void VoxWareAudio::SetRGain(int level)
{
	rgain = level;
}

void VoxWareAudio::SetPGain(int level)
{
	pgain = level;
}

void VoxWareAudio::OutputPort(int p)
{
	oport = p;
}

void VoxWareAudio::InputPort(int p)
{
	iport = p;
}

void VoxWareAudio::RMute()
{
	rmute |= 1;
}

void VoxWareAudio::RUnmute()
{
	rmute &=~ 1;
}
