/* ==================================================== ======== ======= *
 *
 *  uctrl.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uctrl_hpp_
#define	_uctrl_hpp_
//pragma ident	"@(#)uctrl.hpp	ubit:03.06.00"
#include <ubit/umode.hpp>


/** generic interaction controller.
 *
 *  this class controls object's interaction. in the current implementation
 *  it is a superclass of UGroup, so that all container objects have 
 *  the same basic interactive capabilities (in other words, a UBox object
 *  can behave in the same way as a UButton, a UCheckbox or a textual object 
 *  if its modes are set in an appropriate way).
 *
 * Note: this class is a base class for UGroup (the logical "grouping" 
 * container) and UBox (the graphical "viewable" container)
 */
class UCtrl: public UBrick {	// abstract class
public:
  UCtrl();
  ///< constructor (see: ~UBrick and ~UGroup for more info).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void enable(bool state, bool call_callbacks = true);
  void setEnabled(bool b) {enable(b);}
  /**< 
   *  enables this object if 'state' is true; disables it if false.
   *  calls the associated object's callbacks if second arg is true
   *  <br>see also: isEnabled()
    */

  bool isEnabled() const;
  /**<
   *  is this object currently enabled ?.
   *  'enabled' means: sensible to high-level events such as: UOn::action,
   *  UOn::arm,UOn::disarm
   *  - see also: enable(), setEnabled()
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void select(bool state, bool call_callbacks = true);
  void setSelected(bool b) {select(b);}
  /**< 
   * selects this object if 'state' is true; unselects it if false.
   * calls the associated object's callbacks if second arg is true
   * <br>see also: isSelected(), isSelectable()
    */

  bool isSelected() const {return (cmodes & UMode::SELECTED) != 0;}
  /**<
   * is this object currently selected ?.
   * an object can be selected if isSelectable() returns true.
   * - see: select(), isSelectable()
    */

  bool isSelectable() const {return (cmodes & UMode::CAN_SELECT) != 0;}
  /**< 
   * can this object be selected interactively (such as a UCheckbox) ?.
   * this behavior depends on the CAN_SELECT mode. It can be changed 
   * by setting this mode to true or by adding the UMode::canSelect brick.
   * - see also: isSelected(), select()
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual bool isShown() const;
  /**< 
   * is this object currently shown ?.
   * true if the object is "showable" (see isShowable()) and all its parents 
   * are also 'showable'.
   * - see: isShowable() and UGroup::show()
   * - Note: "shown" objects are not necessarily "visible" on the screen as they
   *   can be hidden or clipped by other objects (such as scroll panes, etc.)
    */

  bool isShowable() const {return (cmodes & UMode::CAN_SHOW) != 0;}
  /**< 
   * can this object be shown ?.
   * if true, means that this object will be shown if its parents can also
   * be shown. (NB: This fct. is similar to isVisible() in Jaja, hoping this 
   * name will be slightly less unclear... :-)
   * - See also: isShown() and UGroup::show()
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void update() = 0;
  virtual void update(UUpdate) = 0;
  /**< updates object's graphics.
   * Note: update() == update(UUpdate::layout)
   * See also: class UUpdate.
    */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  bool isCmode(u_modes _cmodes) const {return ((cmodes & _cmodes) != 0);}
  ///< [impl] is ONE of these Container modes verified (see class UMode for details) ?.

  u_modes getCmodes() const {return cmodes;}
  ///< [impl] returns current Container mode bitmask.

  void setCmodes(u_modes _cmodes, bool state);
  ///< [impl] sets Container modes according to state.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  bool isDef(UMode &m) const 
  {return ((cmodes & m.cmodes) != 0 || (bmodes & m.bmodes) != 0);}
  ///< [impl] is ONE of the fields of this UMode object verified ? (see class UMode for details).

  bool isDef(u_modes some_Bmodes, u_modes some_Cmodes) const 
  {return ((cmodes & some_Cmodes) != 0 || (bmodes & some_Bmodes) != 0);}
  ///< [impl] is ONE of these Bmodes OR ONE of these Cmodes verified ? (see class UMode for details).

  bool isAllDef(u_modes all_Bmodes, u_modes all_Cmodes) const 
  {return ((cmodes&all_Cmodes)==all_Cmodes && (bmodes&all_Bmodes)==all_Bmodes);}
  ///< [impl] are ALL these bmodes AND ALL these Cmodes verified ? (see class UMode for details).

  void setState(u_state istate);
  u_state getState()  const {return istate;}
  ///< [impl].

  // inherited: virtual void fire(UEvent*, const UOn&);   // fires callbacks

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

#ifndef NO_DOC
protected:
  friend class UMode;
  friend class UOn;
  friend class UAppli;
  friend class UFlow;

  enum {STANDARD, BROWSING, TYPING};
  u_modes cmodes;
  u_state istate;

  virtual void updateView(UEvent&, class UView*, const class UUpdate&) = 0;
 
  virtual void enterBehavior(class UFlow&, UEvent&, int bstyle);
  virtual void leaveBehavior(class UFlow&, UEvent&);
  virtual void armBehavior(class UFlow&, UEvent&, int bstyle);
  virtual bool disarmBehavior(class UFlow&, UEvent&, int bstyle);
  virtual bool actionBehavior(class UFlow&, UEvent&);
  virtual void keyPressBehavior(class UFlow&, UEvent&);
  virtual void keyReleaseBehavior(class UFlow&, UEvent&);
};
#endif

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
#endif

