/* settingsMode.cc
   Allows the user to change game settings

   Copyright (C) 2003  Mathias Broxvall

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "general.h"
#include "gameMode.h"
#include "menuMode.h"
#include "glHelp.h"
#include "SDL/SDL_image.h"
#include "settingsMode.h"
#include "game.h"
#include "settings.h"
#include "menuMode.h"
#include "sound.h"
#include "font.h"
#include "calibrateJoystickMode.h"
#include "menusystem.h"

using namespace std;

SDL_Surface *SettingsMode::title, *SettingsMode::return2menu,*SettingsMode::return2menuHigh;
SettingsMode *SettingsMode::settingsMode;

extern void changeScreenResolution();

void SettingsMode::init() {
  char str[256];
  int i;

  sprintf(str,"%s/images/gamesettings.png",SHARE_DIR);
  title=IMG_Load(str);
  if(!title) { printf("Error: Failed to load %s\n",str); exit(0); } 

  sprintf(str,"%s/images/return2menu.png",SHARE_DIR);
  return2menu=IMG_Load(str);
  if(!return2menu) { printf("Error: Failed to load %s\n",str); exit(0); } 

  sprintf(str,"%s/images/return2menuHigh.png",SHARE_DIR);
  return2menuHigh=IMG_Load(str);
  if(!return2menuHigh) { printf("Error: Failed to load %s\n",str); exit(0); } 

  settingsMode = new SettingsMode();
}
SettingsMode::SettingsMode() {}
void SettingsMode::activated() { subscreen=SUBSCREEN_GRAPHICS; }
void SettingsMode::deactivated() {Settings::settings->save();}
void SettingsMode::display() {
  int w,h,i;

  glPushAttrib(GL_ENABLE_BIT);
  Enter2DMode();
  glColor4f(1.0,1.0,1.0,1.0);
  glClearColor(0.0,0.0,0.0,0.0);
  glClear(GL_COLOR_BUFFER_BIT); // | GL_DEPTH_BUFFER_BIT);
  drawSurface(title,(screenWidth-title->w)/2,64,title->w,title->h);

  char str[256];
  int opt;
  glColor3f(1.0,1.0,1.0);

  clearSelectionAreas();
  switch(subscreen) {
  case SUBSCREEN_GRAPHICS:
	// Choose subscreen
	menuItem_Left(MENU_SUBSCREEN,0,"Graphics");

	// Resolution
	sprintf(str,"%dx%dx%d", screenWidth, screenHeight,Settings::settings->colorDepth);
	menuItem_LeftRight(MENU_RESOLUTION,1,"  Resolution",str);

	// Windowed
	menuItem_LeftRight(MENU_WINDOWED,2,"  Fullscreen",(char*)(Settings::settings->is_windowed?"No":"Yes"));

	// gfx details
	switch(Settings::settings->gfx_details) {
	case 0: sprintf(str,"None"); break;
	case 1: sprintf(str,"Minimalistic"); break;
	case 2: sprintf(str,"Simple"); break;
	case 3: sprintf(str,"Standard"); break;
	case 4: sprintf(str,"Extra"); break;
	case 5: sprintf(str,"Everything"); break;
	}
	menuItem_LeftRight(MENU_GFX_DETAILS,3,"  Details",str);

	// show FPS
	menuItem_LeftRight(MENU_SHOW_FPS,4,"  Show FPS",(char*)(Settings::settings->showFPS?"Yes":"No"));

	break;

  case SUBSCREEN_CONTROLS:
	// Choose subscreen
	menuItem_Left(MENU_SUBSCREEN,0,"Controls");

	/* Mouse sensitivity */
	sprintf(str,"%1.2f", Settings::settings->mouseSensitivity);
	menuItem_LeftRight(MENU_SENSITIVITY,1,"  Sensitivity",str);

	/* Steering */
	if(Settings::settings->rotateSteering == 0) sprintf(str,"normal");
	else if(Settings::settings->rotateSteering > 0) sprintf(str,"+%d degrees",45*Settings::settings->rotateSteering);
	else if(Settings::settings->rotateSteering < 0) sprintf(str,"-%d degrees",-45*Settings::settings->rotateSteering);
	menuItem_LeftRight(MENU_STEERING,2,"  Steering",str);

	/* Joystick */
	i=Settings::settings->joystickIndex;
	if(i) snprintf(str,255,"%s",SDL_JoystickName(i-1));
	else snprintf(str,255,"no joystick");
	menuItem_Left(0,4,"  Joystick:");
	if(strlen(str) < 15)
	  menuItem_LeftRight(MENU_JOYSTICK,4,"",str);
	else
	  menuItem_Center(MENU_JOYSTICK,5,str);

	break;

  case SUBSCREEN_OTHER:
	// Choose subscreen
	menuItem_Left(MENU_SUBSCREEN,0,"Other");

	/* Music and Sfx volumes */
	sprintf(str,"%d%%",(int)(Settings::settings->musicVolume*100.0));
	menuItem_LeftRight(MENU_MUSIC_VOLUME,1,"  Music volume",str);
	sprintf(str,"%d%%",(int)(Settings::settings->sfxVolume*100.0));
	menuItem_LeftRight(MENU_SFX_VOLUME,2,"  Effects volume",str);

	break;
  }
  menuItem_Left(MENU_RETURN,10,"Done");

#ifdef FOOBARS
  if(selected == SETTING_RETURN)
	drawSurface(return2menuHigh,screenWidth/2-return2menuHigh->w/2,screenHeight-20-Y_IMG_DIFF,return2menuHigh->w,return2menuHigh->h);
  else
	drawSurface(return2menu,screenWidth/2-return2menu->w/2,screenHeight-20-Y_IMG_DIFF,return2menu->w,return2menu->h);
#endif

  glEnable(GL_BLEND);
  drawMousePointer();
  displayFrameRate();
  Leave2DMode();
  glPopAttrib();
}
void SettingsMode::key(int key) {
  if(key == SDLK_SPACE) { int x,y; SDL_GetMouseState(&x,&y); mouseDown(1,x,y); }
  if(key == SDLK_RETURN) { int x,y; SDL_GetMouseState(&x,&y); mouseDown(3,x,y); }
  if(key == SDLK_ESCAPE) GameMode::activate(MenuMode::menuMode);
}
void SettingsMode::idle(Real td) {
  int x,y;

  tickMouse(td);

  SDL_GetMouseState(&x,&y);
  Uint8 *keystate = SDL_GetKeyState(NULL);
  if(keystate[SDLK_LEFT]) { x-=(int)(150/fps); SDL_WarpMouse(x,y); }
  if(keystate[SDLK_RIGHT]) { x+=(int)(150/fps); SDL_WarpMouse(x,y); }
  if(keystate[SDLK_UP]) { y-=(int)(150/fps); SDL_WarpMouse(x,y); }
  if(keystate[SDLK_DOWN]) { y+=(int)(150/fps); SDL_WarpMouse(x,y); }
}

extern int screenResolutions[4][2],nScreenResolutions;

void SettingsMode::mouseDown(int button,int x,int y) {
  int up;
  if(button == 1) up=1;
  else up=0;
  Settings *settings=Settings::settings;

  switch(getSelectedArea()) {
  case MENU_SUBSCREEN: subscreen=(SettingsMode::eSubScreen)mymod(subscreen+(up?1:-1),NUM_SUBSCREENS); break;
  case MENU_RETURN: GameMode::activate(MenuMode::menuMode); break;

  case MENU_RESOLUTION: 
	// Change screen resolution until we got a valid screen
	do {
	  settings->resolution = settings->resolution + (up?1:-1);
	  if(settings->resolution == nScreenResolutions) {
		settings->resolution=0;
		if(settings->colorDepth == 16) settings->colorDepth=32;
		else settings->colorDepth=16;
	  } else if(settings->resolution == -1) {
		settings->resolution=nScreenResolutions-1;
		if(settings->colorDepth == 16) settings->colorDepth=32;
		else settings->colorDepth=16;
	  }
	  changeScreenResolution();
	} while(!screen);
	break;
  case MENU_WINDOWED:
	Settings::settings->is_windowed = Settings::settings->is_windowed ? 0 : 1;
	screen = SDL_SetVideoMode(screenWidth,screenHeight, 16, SDL_SWSURFACE | SDL_OPENGL | 
							  (Settings::settings->is_windowed?0:SDL_FULLSCREEN));
	break;
  case MENU_GFX_DETAILS:
	// Change level of graphic details */
	settings->gfx_details = mymod((settings->gfx_details + (up?1:-1)),6);	
	break;
  case MENU_SHOW_FPS: Settings::settings->showFPS = Settings::settings->showFPS ? 0 : 1; break;

  case MENU_SENSITIVITY:
	Settings::settings->mouseSensitivity += up?0.25:-0.25; 
	if(Settings::settings->mouseSensitivity > 5.001) Settings::settings->mouseSensitivity = 0.0;
	if(Settings::settings->mouseSensitivity < 0.000) Settings::settings->mouseSensitivity = 5.00;
	break;
  case MENU_STEERING: settings->rotateSteering=mymod(settings->rotateSteering+3+(up?1:-1),8)-3; break;
  case MENU_JOYSTICK:
	settings->joystickIndex = mymod(settings->joystickIndex+(up?1:-1),SDL_NumJoysticks()+1);
	if(settings->joystickIndex > 0) {
	  GameMode::activate(CalibrateJoystickMode::calibrateJoystickMode);
	}
	break;

  case MENU_MUSIC_VOLUME:
	settings->musicVolume = mymod(((int) (settings->musicVolume*10.0))+(up?1:-1),11)*0.1;
	volumeChanged();
	break;
  case MENU_SFX_VOLUME:
	settings->sfxVolume = mymod(((int) (settings->sfxVolume*10.0))+(up?1:-1),11)*0.1;
	volumeChanged();
	playEffect(SFX_GOT_FLAG);
	break;

  }
}
