/*
 * misc.c
 *
 *     -- miscellaneous utility functions
 *
 * Copyright  1999 Anselm Lingnau <lingnau@tm.informatik.uni-frankfurt.de>
 * See file COPYING for conditions on use and distribution.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

#include "dvi.h"
#include "dviInt.h"

#ifndef lint
static char rcsid[] VAR_UNUSED = "$Id: misc.c,v 1.2 2000/02/15 11:23:41 lingnau Exp $";
#endif /* lint */


/*
 * ------------------------------------------------------------------------
 *
 * DviSaveStr --
 *
 *     Given a string, save a copy of the string in dynamic memory.
 *
 * Results:
 *     A pointer to the saved copy of the string. If there isn't enough
 *     dynamic memory left, the program is aborted.
 *
 * ------------------------------------------------------------------------
 */

char *
DviSaveStr (string)
    const char *string;
{
    char *copy = ckalloc(strlen(string) + 1);

    if (copy == 0) {
	fprintf(stderr, "Not enough memory\n");
	exit(1);
    }
    return strcpy(copy, string);
}


/*
 * ------------------------------------------------------------------------
 *
 * DviSaveStrN --
 *
 *     Given a string, save a copy of the string in dynamic memory
 *     up to a given length.
 *
 * Results:
 *     A pointer to the saved copy of the string. If there isn't enough
 *     dynamic memory left, the program is aborted.
 *
 * ------------------------------------------------------------------------
 */

char *
DviSaveStrN (string, length)
    const char *string;
    const size_t length;
{
    char *copy = ckalloc(length + 1);

    if (copy == 0) {
	fprintf(stderr, "Not enough memory\n");
	exit(1);
    }
    strncpy(copy, string, length);
    copy[length] = '\0';
    return copy;
}

/*
 * ------------------------------------------------------------------------
 *
 * Dvi_LoadFileBinary --
 *
 *     Load a named file into dynamically allocated memory.
 *
 * Returns:
 *     A pointer to an area of memory holding the contents of the file
 *     specified by `name' or a null pointer if loading the file fails.
 *
 * Side effects:
 *     The procedure tries to open the file specified by `name', allocate
 *     sufficient memory to hold its contents and read the file into
 *     the memory area.
 *
 *     This is used for loading font files.
 *
 * ------------------------------------------------------------------------
 */

U8 *
Dvi_LoadFileBinary (const char *name)
{
    int fileDesc;               /* File descriptor for reading the file */
    struct stat statBuf;        /* To find out the file size */
    U8 *buffer;                 /* Pointer to buffer for reading the file */

    fileDesc = open(name, O_RDONLY);
    if (fileDesc < 0) {
        return (U8 *)0;
    }
    if (fstat(fileDesc, &statBuf) < 0) {
        close(fileDesc);
        return (U8 *)0;
    }
    if ((buffer = ckalloc((size_t)statBuf.st_size)) == 0) {
        close(fileDesc);
        return (U8 *)0;
    }
    if (read(fileDesc, (char *)buffer, (size_t)statBuf.st_size)
        != statBuf.st_size) {
        ckfree((char *)buffer);
        close(fileDesc);
        return (U8 *)0;
    }
    close(fileDesc);
    return buffer;
}
