/*
   SwingWT
   Copyright(c)2003-2004, R. Rawson-Tetley

   For more information on distributing and using this program, please
   see the accompanying "COPYING" file.

   Contact me by electronic mail: bobintetley@users.sourceforge.net

   $Log: Container.java,v $
   Revision 1.54  2004/11/02 11:07:30  bobintetley
   Fixed a number of small compatibility bugs

   Revision 1.53  2004/10/30 20:11:54  bobintetley
   Code cleanup

   Revision 1.52  2004/06/10 07:57:57  dannaab
   Added AWTSwingWrapper to map AWT components to the corresponding Swing version.  Implemted using it: TextComponent, TextArea, TextField; used similar technique for Menu components

   Revision 1.51  2004/06/08 12:53:38  bobintetley
   Container.getComponentCount() is now thread safe

   Revision 1.50  2004/06/08 09:24:21  dannaab
   Rename Component.getPeer() -> getSWTPeer().  added ComponentPeer and stubbed out support classes.

   Revision 1.49  2004/05/11 09:41:39  bobintetley
   Fixes to JInternalFrame ordering

   Revision 1.48  2004/05/10 14:33:12  bobintetley
   MDI support and JPanel border fixes

   Revision 1.47  2004/05/07 12:11:15  bobintetley
   Default layout fixes and correct behaviour for null layout

   Revision 1.46  2004/05/06 16:45:44  laurentmartelli
   Fixed getMaximumSize() compil bug

   Revision 1.45  2004/05/06 16:16:08  laurentmartelli
   Use layout manager for getMinimumSize(), getMaximumSize() and getPreferredSize()

   Revision 1.44  2004/05/05 23:32:29  laurentmartelli
   Fixed add(String,Component) to call add(Component,Object)

   Revision 1.43  2004/05/05 13:24:30  bobintetley
   Bugfixes and Laurent's patch for binary compatibility on Container.add()

   Revision 1.42  2004/05/05 12:43:19  bobintetley
   Patches/new files from Laurent Martell

   Revision 1.41  2004/05/04 00:57:03  dannaab
   Sun java and javax package naming improvements.  Fix ambiguous class names by hardcoding SWT full class name

   Revision 1.40  2004/05/03 20:53:43  dannaab
   small tweak to layoutmanager instance objects to make easier to read

   Revision 1.39  2004/04/28 07:19:29  bobintetley
   Fixes to destruction code

   Revision 1.38  2004/04/27 06:37:42  bobintetley
   Corrected hierarchy with JComponent descending Container

   Revision 1.37  2004/04/23 00:52:31  dannaab
   Handle borders in a Swing-like way. Implement EmptyBorder & TitledBorder

   Revision 1.36  2004/03/31 08:07:34  bobintetley
   JTable bug fixed that prevented it redrawing after being dropped from
   a container. Protected component method allows catching of when a component
   is removed from a container.

   Revision 1.35  2004/03/22 15:10:21  bobintetley
   JRootPane and JLayeredPane implementation

   Revision 1.34  2004/03/12 19:59:47  bobintetley
   Fix to container remove() that allows recreation of components

   Revision 1.33  2004/03/12 11:05:23  bobintetley
   Fixed memory leak in container destruction

   Revision 1.32  2004/02/03 09:21:20  bobintetley
   Fixed threading bug in invokeIn, and JComboBox hotkeys

   Revision 1.31  2004/02/02 12:36:36  bobintetley
   Proper JScrollPane/ScrollBar implementation

   Revision 1.30  2004/01/30 09:21:55  bobintetley
   Container bug fixed that caused problems with layout

   Revision 1.29  2004/01/27 11:06:09  bobintetley
   Fixed bugs in dropping/re-adding the same components

   Revision 1.28  2004/01/26 14:26:29  bobintetley
   Container/Component Listener/Event support

   Revision 1.27  2004/01/26 08:10:59  bobintetley
   Many bugfixes and addition of SwingSet

   Revision 1.26  2004/01/16 15:53:32  bobintetley
   Many compatibility methods added to Container, Component, JInternalFrame,
      UIManager, SwingUtilities, JTabbedPane, JPasswordField, JCheckBox
      and JRadioButton.

   Revision 1.25  2004/01/16 09:35:46  bobintetley
   Full event dispatch thread support!

   Revision 1.24  2004/01/15 15:20:29  bobintetley
   Java2D work

   Revision 1.23  2004/01/09 11:47:27  bobintetley
   Automatic JButton mapping!

   Revision 1.22  2004/01/09 10:33:56  bobintetley
   Changes for JToolBar to allow platform ToolBars, mixed with other components

   Revision 1.21  2004/01/07 09:47:13  bobintetley
   Fixed class cast exception retrieving component lists

   Revision 1.20  2004/01/05 12:31:13  bobintetley
   Table/List/Toolbar fixes for correct layout

   Revision 1.19  2004/01/05 11:31:46  bobintetley
   More layout fixes

   Revision 1.18  2004/01/05 10:09:52  bobintetley
   Various fixes for new layouts

   Revision 1.17  2004/01/05 08:48:09  bobintetley
   Manually merged Daniel's changes

   Revision 1.15  2003/12/18 09:37:33  bobintetley
   Enabled support for JTabbedPane and JClosableTabbedPane (SWT doesn't
   support, so a bit of magic in here)

   Revision 1.14  2003/12/17 16:30:35  bobintetley
   Flowlayout fix, vertical toolbar support and cleaned up text alignment
   hierarchy.

   Revision 1.13  2003/12/17 11:37:46  bobintetley
   Better GridBag support

   Revision 1.12  2003/12/16 17:46:17  bobintetley
   Additional thread safety methods

   Revision 1.11  2003/12/16 14:51:16  bobintetley
   Fixed hang when a window close event closes itself again

   Revision 1.10  2003/12/16 13:14:33  bobintetley
   Use of SwingWTUtils.isSWTControlAvailable instead of null test

   Revision 1.9  2003/12/16 12:36:07  bobintetley
   Fix to closing of internal frames

   Revision 1.8  2003/12/16 09:19:02  bobintetley
   Various small fixes to match Swing more closely

   Revision 1.7  2003/12/15 19:11:09  bobintetley
   Containers default to FlowLayout unless overriden (as real Swing does)

   Revision 1.6  2003/12/15 18:29:56  bobintetley
   Changed setParent() method to setSwingWTParent() to avoid conflicts with applications

   Revision 1.5  2003/12/15 17:51:56  bobintetley
   Revalidate support

   Revision 1.4  2003/12/15 15:54:25  bobintetley
   Additional core methods

   Revision 1.3  2003/12/14 09:13:38  bobintetley
   Added CVS log to source headers

*/


package swingwt.awt;

import org.eclipse.swt.widgets.*;
import org.eclipse.swt.*;

import swingwtx.swing.*;

import java.util.*;

/**
 * Any component that can contain other components.
 * This class maps the Swing layouts/placements to SWT
 */
public class Container extends Component {

    /** The SWT peer this container represents */
    public org.eclipse.swt.widgets.Composite composite = null;

    /** The layout manager being used to position child components */
    protected LayoutManager layout = null;

    /** Cache of components waiting to be laid out */
    protected Vector comps = new Vector();

    protected Vector containerListeners = new Vector();
    
    public Dimension getMinimumSize() { 
        return (layout==null) ? super.getMinimumSize() : layout.minimumLayoutSize(this);
    }

    public Dimension minimumSize() { return getMinimumSize(); }
    public Dimension preferredSize() { return getPreferredSize(); }

    public Dimension getPreferredSize() { 
        return (layout==null) ? super.getPreferredSize() : layout.preferredLayoutSize(this);
    }

    public Dimension getMaximumSize() { 
        return (layout instanceof LayoutManager2) ? 
            ((LayoutManager2)layout).maximumLayoutSize(this) :
            super.getMaximumSize();
    }

    public void addContainerListener(swingwt.awt.event.ContainerListener l) {
        containerListeners.add(l);
    }

    public void removeContainerListener(swingwt.awt.event.ContainerListener l) {
        containerListeners.remove(l);
    }

    public Component add(Component c) {
        return doAdd(c);
    }
    
    public Component add(Component c, int index) {
        return add(c);    
    }

    /** Actually does the hard work of adding something to a container */
    public Component doAdd(final Component c) {

        if (c == null) return null;
        final Container me = this;

        // Don't add it to the cache again if this component is already cached
        // - this stops us adding the same component twice.
        if (comps.indexOf(c) == -1) {
            comps.add(c);
        }
        
        // Cache us with the child component - this is just so that
        // the hierarchy is visible before the peers are realised
        c.parent = this;        
        
        if (!SwingWTUtils.isSWTControlAvailable(composite)) return c;

        // Register the comonent with the layout if needed
        addComponentToLayout(c);

        // Ensure that the setSwingWTParent calls are on the dispatch
        // thread as they go down - object creation should always be
        // on this thread
        SwingUtilities.invokeSync(new Runnable() {
            public void run() {

                try {
                    c.setSwingWTParent(me);
                    c.setCachedProperties();
                    c.registerEvents();
                    processContainerEvent(new swingwt.awt.event.ContainerEvent(me, swingwt.awt.event.ContainerEvent.COMPONENT_ADDED, c));
                }
                catch (Exception e) {
                    e.printStackTrace();
                }

                queuedValidate();

            }
        });
        return c;
    }

    protected void processEvent(AWTEvent e) {
        // Can't think of any reason for this to be here other than compile compat
    }
    
    protected void processContainerEvent(swingwt.awt.event.ContainerEvent e) {
        if (containerListeners.size() == 0) return;
        for (int i = 0; i < containerListeners.size(); i++) {
            if (e.getID() == swingwt.awt.event.ContainerEvent.COMPONENT_ADDED)
                ((swingwt.awt.event.ContainerListener) containerListeners.get(i)).componentAdded(e);
            else
                ((swingwt.awt.event.ContainerListener) containerListeners.get(i)).componentRemoved(e);
        }
    }

    /**
     * Adds an existing component to the layout manager
     */
    public void addComponentToLayout(Component c) {
        if (layout != null) {
            if (layout instanceof LayoutManager2)  {
                ((LayoutManager2)layout).addLayoutComponent(c, c.layoutModifier);
            }
            else
                layout.addLayoutComponent(c.getName(), c);
        }
    }

    /** Forces laying out of this container's child components again. */
    public void invalidate() {
        final Container me = this;
        if (SwingWTUtils.isSWTControlAvailable(composite))
            if (layout != null)
                SwingUtilities.invokeSync(new Runnable() {
                    public void run() {
                        layout.layoutContainer(me);
                    }
                });
    }

    /** Forces laying out of this container's child components again. */
    public void doLayout() { invalidate(); }
    /** Forces laying out of this container's child components again. */
    public void validate() { invalidate(); }
    /** Forces laying out of this container's child components again. */
    public void revalidate() { invalidate(); }

    /**
     * Attempts to ensure that a request to layout a container is only
     * done once whilst adding many components to a container. What we do is
     * basically set a flag to say we need to relayout the container, and if further
     * requests come in the next 50ms, we ignore them. This should really take the heat
     * off the CPU whilst building containers with many objects.
     *
     * Since this generally gets called as we work down the tree doing setSwingWTParent
     * calls, that thread will basically block with no invalidation until all
     * components have been added (what we want) - this is just horribly complex to
     * get your head round, but it works :-)
     *
     * @author Robin Rawson-Tetley
     */
    public void queuedValidate() {
        if (queuedValidateRequest) return; // Already waiting - don't do anything
         queuedValidateRequest = true;
         SwingUtilities.invokeIn(new Runnable() {
             public void run() {
                 invalidate();
                 queuedValidateRequest = false;
             }
         }, 1);
    }
    protected boolean queuedValidateRequest = false;

    /**
     * Addition for layouts requiring object based modifiers
     */
    public void add(swingwt.awt.Component c, Object layoutModifier) {
        if (c == null) return;
        c.layoutModifier = layoutModifier;

        if (layoutModifier instanceof String)
            layout.addLayoutComponent((String)layoutModifier, c);

        doAdd(c);
    }
    
    public void add(swingwt.awt.Component c, Object layoutModifier, int index) {
        add(c, layoutModifier);    
    }

    public Component add(String name, swingwt.awt.Component c) {
        add(c,name);
        return c;
    }

    public void dispose() {

        // Dispose of all child components (and their components)
        if (comps != null) {
            for (int i = 0; i < comps.size(); i++) {
                try {
                    ((Component) comps.get(i)).dispose();
                }
                catch (Exception e) {}
            }
            // Ditch the cache
            comps.removeAllElements();
        }

        if (peer != null) super.dispose();
        
	composite = null;
	peer = null;
    }

    /**
     * Removes a component from the the container
     * by destroying the peer.
     */
    public void remove(swingwt.awt.Component c) {
        comps.remove(c);
        if (layout != null) layout.removeLayoutComponent(c);
        c.componentOnlyDispose();
        c.setComponentRemoved();
        processContainerEvent(new swingwt.awt.event.ContainerEvent(this, swingwt.awt.event.ContainerEvent.COMPONENT_REMOVED, c));
        invalidate();
    }

    public void remove(int index) {
        Component c = (Component) comps.get(index);
        remove(c);
    }

    public void removeAll() {
        for (int i = 0; i < comps.size(); i++) {
            Component c = (Component) comps.get(i);
	    remove(c);
        }
        comps.removeAllElements();
        invalidate();
    }

    /** Removes a cached component only */
    public void removeComponentFromCache(Component c) {
        comps.remove(c);
    }

    /** Returns the layout used by this container */
    public LayoutManager getLayout() {
            return layout;
    }

    /**
     * I'm totally re-writing this method.  No longer
     * will it only mirror the layouts.  From now on,
     * it will actually use the layout managers to layout
     * the component, not the mirrored SWT layouts.
     * Two big advantages with this approach.  (a) It
     * allows custom layout managers.  (b) It means
     * less work since we don't have to port anything.
     * (yea!)
     *
     * @author Daniel Spiewak
     */
    public void setLayout(LayoutManager l) {
        setLayoutImpl(l);
    }

    /**
     * The actual code that handles assignment of the
     * layout manager.
     *
     * Separated out as setLayout will often get
     * subclassed and my head will explode if I don't
     * do something like this :-)
     */
    protected void setLayoutImpl(LayoutManager l) {
        layout = l;

        if (composite == null)
                return;

        SwingUtilities.invokeSync(new Runnable() {
            public void run() {

                // we don't want an SWT layout since we will control the layout ourselves
                composite.setLayout(null);

                // add a resize listener to let us know when to invalidate the layout
                composite.addListener(SWT.Resize, new Listener() {
                        public void handleEvent(org.eclipse.swt.widgets.Event e) {
                                invalidate();
                        }
                });

            }
        });
    }

    /** Returns the SWT peer being used for this container */
    public org.eclipse.swt.widgets.Composite getComposite() {
        return composite;
    }

    public Container getParent() {
        return parent;
    }

    /** Called when this gets added. We can set layouts and things
     *  then if necessary. Note that subclasses are responsible
     *  for creating the actual peer and setting the protected
     *  "composite" variable */
    public void setSwingWTParent(swingwt.awt.Container parent) throws Exception {

        if (layout != null)
            setLayoutImpl(layout);

        if (comps.size() > 0) {
            Object[] obs = comps.toArray();
            for (int i = 0; i < obs.length; i++) {
                Component c = (Component) obs[i];
                doAdd(c);
            }
        }
    }

    public ComponentOrientation getComponentOrientation() {
            return ComponentOrientation.getOrientation(Locale.getDefault());
    }

    public Insets getInsets() {
        return new Insets(0,0,0,0);
    }

    public int getComponentCount() {
        return comps.size();
    }

    public Component getComponent(int i) {
            return (Component) comps.get(i);
    }

    public Component[] getComponents() {
        Object[] comp = comps.toArray();
        Component[] cps = new Component[comp.length];
        for (int i = 0; i < comp.length; i++) {
            cps[i] = (Component) comp[i];
        }
        return cps;
    }

    /**
     * Useful for debugging purposes. Shows this container and
     * all it's children.
     */
    public void debug_showContainmentTree() {
        System.out.println("Containment Tree: ====================");
        System.out.println(getClass().getName());
        xdebug_showChildren(this, 1);
    }
    private void xdebug_showChildren(Container c, int level) {
        final String SPACE = "                   ";
        for (int i = 0; i < c.comps.size(); i++) {
            Component d =  (Component) c.comps.get(i);
            System.out.println(SPACE.substring(0, level) + d.getClass().getName());
            if (d instanceof Container)
                xdebug_showChildren((Container) d, level+1);
        }
    }

}
