/*
 * Smart Gnome Control - file functions file
 * Copyright (C) 2004 by Mark J. Fine <fineware@users.sourceforge.net>
 *
 * This program provides a graphical user interface to multiple
 * communications receivers using the Hamlib library.
 *
 * $Id: logfile.c,v 1.2 2004/12/29 21:05:48 fineware Exp $
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 *      Free Software Foundation, Inc.
 *      59 Temple Place - Suite 330
 *      Boston, MA  02111-1307
 *      USA
 *
 */

#include "logfile.h"
#include "extra.h"
#include "support.h"

const gchar* basestr = ".sgcontrol";

/* routine to create user data directory string */
gchar*
get_user_base()
{
	gchar* pathstr;
	
	pathstr = g_concat_dir_and_file(g_get_home_dir(),basestr);
	return pathstr;
}


/* routine to see if user directory exists, if not create it and copy dbs into it */
void
prep_user_dir()
{
	int		i , f_count;
	
	gchar*	f_name = NULL;
	gchar*	f_name2 = NULL;
	gchar*	old_path = NULL;
	GList*	from_file_list = NULL;
	GnomeVFSFileInfo*			f_info;
	GnomeVFSDirectoryFilter*	filter;
	GnomeVFSResult				result;
	GnomeVFSURI*				uri1;
	GnomeVFSURI*				uri2;
	
	basepath = get_user_base();
	basedbpath = g_strdup_printf("%s/%s",basepath,"/SWBCGIDE.*");
	baselogpath = g_strdup_printf("%s/%s",basepath,"/*.log");
	if (!gnome_vfs_initialized())
		gnome_vfs_init();	
	if (!g_file_exists(basepath)) {
		if (gnome_vfs_make_directory(basepath, 0744) == GNOME_VFS_OK) {
			old_path = g_strdup_printf("%s",PACKAGE_DATA_DIR);
			filter = gnome_vfs_directory_filter_new(GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
													GNOME_VFS_DIRECTORY_FILTER_NODIRS |
													GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
													GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
													GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR,
													"*.*");
			if (gnome_vfs_directory_list_load(&from_file_list, old_path,
												GNOME_VFS_FILE_INFO_DEFAULT, filter) == GNOME_VFS_OK) {
				f_count = g_list_length(from_file_list);
				for (i = 0; i < f_count; i++) {
					f_info = (GnomeVFSFileInfo *)g_list_nth_data(from_file_list,i);
					f_name = g_strdup_printf("%s/%s",old_path,f_info->name);
					uri1 = gnome_vfs_uri_new (f_name);
					f_name2 = g_strdup_printf("%s/%s",basepath,f_info->name);
					uri2 = gnome_vfs_uri_new (f_name2);
					result = gnome_vfs_xfer_uri(uri1, uri2,
												GNOME_VFS_XFER_DEFAULT,
												GNOME_VFS_XFER_ERROR_MODE_ABORT,
												GNOME_VFS_XFER_OVERWRITE_MODE_ABORT, NULL, NULL);
					if (result != GNOME_VFS_OK)
						show_message("User Prep Initialization Error", (gchar*)gnome_vfs_result_to_string(result), "error");
					g_free(f_name);
					g_free(f_name2);
				}
				g_list_free(from_file_list);
			}
			g_free(old_path);
		}
	}
}


gboolean
file_exists                            (gchar* in_file)
{
	gboolean		result;
	GnomeVFSURI*	uri;
	
	if (!gnome_vfs_initialized())
		gnome_vfs_init();	
	uri = gnome_vfs_uri_new (in_file);
	result = gnome_vfs_uri_exists(uri);
	return result;
}


gchar*
strchrins                              (gchar* in_ins,
                                        gchar* in_str,
                                        int    in_pos)
{
	gchar*	temp;
	gchar*	sect1;
	gchar*	sect2;
	
	temp = in_str;
	sect1 = g_strndup(temp,in_pos);
	sect2 = g_strdup(temp+in_pos);
	g_free(temp);
	temp = g_strdup_printf("%s%s%s",sect1,in_ins,sect2);
	g_free(sect1);
	g_free(sect2);
	return temp;
}


gchar*
strchrdel                              (gchar* in_str,
                                        int    in_pos,
                                        int    in_len)
{
	gchar*	temp;
	gchar*	sect1 = NULL;
	gchar*	sect2;
	
	temp = in_str;
	if (in_pos != 0)
		sect1 = g_strndup(temp,in_pos);
	sect2 = g_strdup(temp+in_pos+in_len);
	g_free(temp);
	if (in_pos != 0)
		temp = g_strdup_printf("%s%s",sect1,sect2);
	else
		temp = g_strdup(sect2);
	if (in_pos != 0)
		g_free(sect1);
	g_free(sect2);
	return temp;
}


void
select_from_log_list                   (GtkWidget* inWidget,
                                        GdkEventButton  *inEvent,
                                        gint in_row)
{
	gint	in_col;
	guint	len;
	gchar*	text;
	gchar*	freq_text;
	double	r_freq;

	if (inEvent == NULL)
		in_col = 0;
	else {
		if (inEvent->button == 1)
			get_list_clicked_row_col(inWidget, inEvent, &in_row, &in_col);
		else {
			in_row = -1;
			in_col = -1;
		}
	}
	if ((log_name) && (in_row > -1)) {
		set_log_data(in_row);
		if ((ispower) && (in_col == 0)) {
			gtk_clist_get_text(GTK_CLIST(inWidget),in_row,0,&text);
			if (text) {
				len = strlen(text);
				freq_text = g_strndup(text,len-4);
				g_strup(freq_text);
				r_freq = g_strtod(freq_text,NULL);
				if (r_freq != 0.0) {
					switch (text[len-3]) {
						case 'K' : r_freq = r_freq * 1000.0; break;
						case 'M' : r_freq = r_freq * 1000000.0; break;
						case 'G' : r_freq = r_freq * 1000000000.0; break;
					}
					frequency = r_freq;
					set_freq(TRUE);
				}
				g_free(freq_text);
			}
		}
	}
}


void
set_log_data                           (int in_row)
{
	gint			len, pos;
	gchar*			temp;
	GtkWidget*		gen_item;

	if ((log_details) && (in_row > -1)) {
		temp = g_list_nth_data(log_details, in_row);
		gen_item = lookup_widget(log1,"LogText");
		len = gtk_text_get_length(GTK_TEXT(gen_item));
		if (len > 0)
			gtk_editable_delete_text(GTK_EDITABLE(gen_item),0,len);
		if (temp) {
			pos = 0;
			len = strlen(temp);
			if (len > 0)
				gtk_editable_insert_text(GTK_EDITABLE(gen_item),temp,len,&pos);
		}
	}
}


void
create_log_file()
{
	log_selector = gtk_file_selection_new("Create New Log File");
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(log_selector),baselogpath);
	gtk_file_selection_complete(GTK_FILE_SELECTION(log_selector),"*.log");
	gtk_window_set_position(GTK_WINDOW(log_selector), GTK_WIN_POS_CENTER);
	gtk_window_set_modal(GTK_WINDOW(log_selector), TRUE);
	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC(create_logname), NULL);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
						(gpointer) log_selector);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->cancel_button),
						"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
						(gpointer) log_selector);
	gtk_widget_show(log_selector);
}


void
create_logname						   (GtkFileSelection *selector,
                                        gpointer         user_data)
{
	gboolean	isok;
	gchar*		temp;
	gchar*		log_filename;
	gchar*		log_extension;
	gchar*		log_message;
	
	isok = FALSE;
	temp = gtk_file_selection_get_filename(GTK_FILE_SELECTION(log_selector));
	log_extension = get_extension_from_full_path(temp);
	if (strcmp(log_extension,"log") == 0)
		isok = TRUE;
	g_free(log_extension);

	if (isok) {
		if (log_name) {
			g_free(log_name);
			log_name = NULL;
		}
		log_name = g_strdup(temp);
		if (file_exists(log_name)) {
			log_filename = get_filename_from_full_path(log_name);
			log_message = g_strdup_printf("%s already exists.\n Overwrite?",log_filename);
			show_yes_no_message("Create New Log File",log_message,"warning",new_logfile,NULL);
			g_free(log_message);
			g_free(log_filename);	
		}
		else
			new_logfile();
	}
	else {
		log_filename = get_filename_from_full_path(temp);
		log_message = g_strdup_printf("Illegal log filename.\n%s not created.",log_filename);
		show_message("Create New Log File",log_message,"error");
		g_free(log_message);
		g_free(log_filename);
	}
}


void
new_logfile()
{
	int				len;
	FILE			*fp;
	gchar*			temp;
	gchar*			log_title_str;
	gchar*			log_file_name = NULL;
	GtkWidget*		gen_item;

	fp = fopen(log_name,"w");
	if (!fp) {
		temp = g_strdup_printf("Can't create %s.",log_name);
		show_message("Create New Log File",temp,"warning");
		g_free(temp);
	}
	else {
		log_file_name = get_filename_from_full_path(log_name);
		log_title_str = g_strdup_printf("Log Display - [%s]",log_file_name);
		gtk_window_set_title(GTK_WINDOW(log1),log_title_str);
		g_free(log_file_name);
		g_free(log_title_str);

		gen_item = lookup_widget(log1,"LogText");
		len = gtk_text_get_length(GTK_TEXT(gen_item));
		if (len > 0)
			gtk_editable_delete_text(GTK_EDITABLE(gen_item),0,len);	

		gen_item = lookup_widget(log1,"LogList");
		if (gen_item)
			gtk_clist_clear(GTK_CLIST(gen_item));
		if (log_details) {
			g_list_free(log_details);
			log_details = NULL;
		}
		fclose(fp);	
		gconf_client_set_string(confclient,"/apps/sgcontrol/last_log",log_name,NULL);
	}
}


void
get_log_file()
{
	log_selector = gtk_file_selection_new("Open Log File");
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(log_selector),baselogpath);
	gtk_file_selection_complete(GTK_FILE_SELECTION(log_selector),"*.log");
	gtk_window_set_position(GTK_WINDOW(log_selector), GTK_WIN_POS_CENTER);
	gtk_window_set_modal(GTK_WINDOW(log_selector), TRUE);
	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC(store_logname), NULL);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->ok_button),
						"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
						(gpointer) log_selector);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->cancel_button),
						"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
						(gpointer) log_selector);
	gtk_widget_show(log_selector);
}


void
store_logname						   (GtkFileSelection *selector,
                                        gpointer         user_data)
{
	gboolean	isok;
	gchar*		temp;
	gchar*		log_filename;
	gchar*		log_extension;
	gchar*		log_message;
	
	isok = FALSE;
	temp = gtk_file_selection_get_filename(GTK_FILE_SELECTION(log_selector));
	log_extension = get_extension_from_full_path(temp);
	if (strcmp(log_extension,"log") == 0)
		isok = TRUE;
	g_free(log_extension);

	if (isok) {
		if (log_name) {
			g_free(log_name);
			log_name = NULL;
		}
		log_name = g_strdup(temp);
		open_logfile();
	}
	else {
		log_filename = get_filename_from_full_path(temp);
		log_message = g_strdup_printf("Illegal log filename.\n%s not opened.",log_filename);
		show_message("Open Log File",log_message,"error");
		g_free(log_message);
		g_free(log_filename);
	}	
}


void
open_logfile()
{	
	int				i, j, len, pos, tab_no, color_toggle, tab5, tab6;
	FILE			*fp;
	char            buf[10000];
	gchar*          tab;
	gchar*			tabtab;
	gchar*			temp;
	gchar*			temp2;
	gchar*			nul_pos;
	gchar*			log_line;
	gchar*			log_title_str;
	gchar*			log_file_name = NULL;
	gchar           **text;
	GtkWidget*		gen_item;
	
	fp = fopen(log_name,"r");
	if (!fp) {
		temp = g_strdup_printf("%s not found.",log_name);
		show_message("Open Log File",temp,"warning");
		g_free(temp);
	}
	else {
		log_file_name = get_filename_from_full_path(log_name);
		log_title_str = g_strdup_printf("Log Display - [%s]",log_file_name);
		gtk_window_set_title(GTK_WINDOW(log1),log_title_str);
		g_free(log_file_name);
		g_free(log_title_str);

		gen_item = lookup_widget(log1,"LogText");
		len = gtk_text_get_length(GTK_TEXT(gen_item));
		if (len > 0)
			gtk_editable_delete_text(GTK_EDITABLE(gen_item),0,len);

		gen_item = lookup_widget(log1,"LogList");
		gtk_clist_clear(GTK_CLIST(gen_item));
		if (log_details) {
			g_list_free(log_details);
			log_details = NULL;
		}
			
		j = 0;
		color_toggle = -1;
		gtk_clist_freeze(GTK_CLIST(gen_item));
		while (!feof(fp)) {
			if (fgets(buf,10000,fp) != NULL) {
				log_line = g_strdup(buf);
			
				/* strip off crs and lfs */		
				if (strlen(log_line) != 0) {
					nul_pos = strchr(log_line, 13);
					if (nul_pos) {
						temp = g_strndup(log_line, (nul_pos - (gchar*)log_line));
						g_free(log_line);
						log_line = g_strdup(temp);
						g_free(temp);
					}
				}
			
				if (strlen(log_line) != 0) {
					nul_pos = strchr(log_line, 255);
					if (nul_pos)
						temp = g_strndup(log_line, (nul_pos - (gchar*)log_line));
					else
						temp = g_strdup(log_line);
					
					/* count how many tabs */
					tab_no = 0;
					len = 0;
					tab5 = 0;
					tab6 = 0;
					temp2 = g_strdup(temp);
					nul_pos = strchr(temp2, 0x09);
					do {
						if (nul_pos) {
							pos = nul_pos-temp2+1;
							temp2 = strchrdel(temp2,0,pos);
							len = len + pos;
							tab_no++;
							if (tab_no == 5)
								tab5 = len;
							if (tab_no == 6)
								tab6 = len;
							nul_pos = strchr(temp2, 0x09);
						}
					} while (nul_pos != NULL); 
					g_free(temp2);
					tab = g_strdup_printf("%c",0x09);
					if (tab_no == 6) {
						temp = strchrins(tab,temp,tab5);
						temp = strchrins(tab,temp,tab6+1);
					}
					
					/* kill first tab in log between freq and kHz */
					for (i = 0; i < strlen(temp); i++) {
						if (temp[i] == 0x09) {
							temp[i] = 0x20;
							break;
						}
					}				
				
					/* kill null segments by adding a space */
					tabtab = g_strdup_printf("%c%c",0x09,0x09);
					nul_pos = strstr(temp,tabtab);
					do {
						if (nul_pos) {
							pos = nul_pos-temp+1;
							temp = strchrins(" ",temp,pos);
							nul_pos = strstr(temp,tabtab);
						}
					} while (nul_pos != NULL);
					g_free(tabtab);
							
					text = g_strsplit(temp, tab, 8);
					g_free(temp);
					temp = text[4];
					text[4] = text[5];
					text[5] = text[6];
					text[6] = temp;
					gtk_clist_append(GTK_CLIST(gen_item),text);
	
					if (color_toggle < 0)
						color_toggle +=2;
					else
						color_toggle -=2;
					if (color_toggle > 0)
						set_clist_row_background(log1,"LogList",bg_color_str,j);
					else
						set_clist_row_background(log1,"LogList",bg_color_str2,j);
					
					gtk_clist_set_row_data(GTK_CLIST(gen_item), j, GUINT_TO_POINTER(j));
					j++;
					g_strfreev(text);
					g_free(tab);
				
					/* now add details to the log_details list */
					nul_pos = strchr(log_line, 255);
					if (nul_pos != NULL)
						temp = g_strdup(nul_pos+3);
					else
						temp = g_strdup(" ");

					/* kill breaking segments by adding a space */
					if (temp) {
						tabtab = g_strdup_printf("%c%c%c",0xff,0x09,0x09);
						nul_pos = strstr(temp,tabtab);
						do {
							if (nul_pos) {
								pos = nul_pos-temp;
								temp = strchrdel(temp,pos,3);
								temp = strchrins(" ",temp,pos);
								nul_pos = strstr(temp,tabtab);
							}
						} while (nul_pos != NULL);
						g_free(tabtab);
					}
					log_details = g_list_append(log_details, temp);
				}
				g_free(log_line);
				for (i = 0; i < 10000; i++)
					buf[i] = 0x00;
			}
		}
		fclose(fp);
		gtk_clist_thaw(GTK_CLIST(gen_item));
		j--;
		if (j > -1) {
			set_list_selected_row(log1,"LogList",j);
			set_log_data(j);
		}
		else {
			temp = g_strdup_printf("%s has no records!",log_name);
			show_message("Open Log File",temp,"info");
			g_free(temp);
		}
		gconf_client_set_string(confclient,"/apps/sgcontrol/last_log",log_name,NULL);
	}
}


void
put_log_file()
{
	if (log_name == NULL) {
		show_message("Save Log Error","No log is currently loaded.\nPlease load a log file before doing a log function.","warning");
	}
	else {	
		log_selector = gtk_file_selection_new("Save Log File As...");
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(log_selector),baselogpath);
		gtk_file_selection_complete(GTK_FILE_SELECTION(log_selector),"*.log");
		gtk_window_set_position(GTK_WINDOW(log_selector), GTK_WIN_POS_CENTER);
		gtk_window_set_modal(GTK_WINDOW(log_selector), TRUE);
		gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->ok_button),
							"clicked", GTK_SIGNAL_FUNC(store_logname_as), NULL);
		gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->ok_button),
							"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
							(gpointer) log_selector);
		gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(log_selector)->cancel_button),
							"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
							(gpointer) log_selector);
		gtk_widget_show(log_selector);
	}
}


void
store_logname_as					   (GtkFileSelection *selector,
                                        gpointer         user_data)
{
	gboolean	isok;
	gchar*		temp;
	gchar*		log_filename;
	gchar*		log_extension;
	gchar*		log_message;
	
	isok = FALSE;
	temp = gtk_file_selection_get_filename(GTK_FILE_SELECTION(log_selector));
	log_extension = get_extension_from_full_path(temp);
	if (strcmp(log_extension,"log") == 0)
		isok = TRUE;
	g_free(log_extension);

	if (isok) {
		if (log_name) {
			g_free(log_name);
			log_name = NULL;
		}
		log_name = g_strdup(temp);
		if (file_exists(log_name)) {
			log_filename = get_filename_from_full_path(log_name);
			log_message = g_strdup_printf("%s already exists.\n Overwrite?",log_filename);
			show_yes_no_message("Save Log File As...",log_message,"warning",save_logfile,NULL);
			g_free(log_message);
			g_free(log_filename);	
		}
		else
			save_logfile();
	}
	else {
		log_filename = get_filename_from_full_path(temp);
		log_message = g_strdup_printf("Illegal log filename.\n%s not saved.",log_filename);
		show_message("Save Log File As...",log_message,"error");
		g_free(log_message);
		g_free(log_filename);
	}
}


void
save_logfile()
{
	int				i, j, len, num_logs;
	FILE			*fp;
	gchar*			buf;
	gchar*          temp;
	gchar*          crlf;
	gchar*          tab;
	gchar*			tabtab;
	gchar*			log_title_str;
	gchar*			log_file_name = NULL;
	gchar*          text[9];
	GtkWidget*		gen_item;
	
	fp = fopen(log_name,"w");
	if (!fp) {
		temp = g_strdup_printf("Can't save %s.",log_name);
		show_message("Save Log File",temp,"warning");
		g_free(temp);
	}
	else {
		log_file_name = get_filename_from_full_path(log_name);
		log_title_str = g_strdup_printf("Log Display - [%s]",log_file_name);
		gtk_window_set_title(GTK_WINDOW(log1),log_title_str);
		g_free(log_file_name);
		g_free(log_title_str);

		crlf = g_strdup_printf("%c%c",0x0d,0x0a);
		tab = g_strdup_printf("%c",0x09);
		tabtab = g_strdup_printf("%c%c%c",0xff,0x09,0x09);

		num_logs = get_list_num_rows(log1,"LogList");
		gen_item = lookup_widget(log1,"LogList");
		for (j = 0; j < num_logs; j++) {
			for (i = 0; i < 8; i++)
				gtk_clist_get_text(GTK_CLIST(gen_item), j, i, &text[i]);
			text[8] = g_list_nth_data(log_details,j);
			text[8] = g_strdelimit(text[8],crlf,0x20);
			g_strstrip(text[8]);
			temp = g_strdup(text[0]);
			len = strlen(temp);
			temp[len-4] = 0x09;			
			buf = g_strdup_printf("%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s\n", temp,tab,
																			text[1],tab,
																			text[2],tab,
																			text[3],tab,
																			text[6],tab,
																			text[4],tab,
																			text[5],tab,
																			text[7],tabtab,
																			text[8]);
			fputs(buf,fp);
			g_free(temp);
		}
		fclose(fp);
		gconf_client_set_string(confclient,"/apps/sgcontrol/last_log",log_name,NULL);
		g_free(crlf);
		g_free(tab);
		g_free(tabtab);
	}	
}


void
change_log_selection                   (gboolean is_next)
{
	gint	cur_row;
	gint	max_row;
	GtkWidget*	gen_item;
	
	cur_row = get_list_selected_row(log1,"LogList");
	max_row = get_list_num_rows(log1,"LogList");
	if (is_next)
		cur_row++;
	else
		cur_row--;
	if ((cur_row > -1) && (cur_row < max_row)) {
		set_list_selected_row(log1,"LogList",cur_row);
		gen_item = lookup_widget(log1,"LogList");
		select_from_log_list(gen_item,NULL,cur_row);
	}
}


/* log entry uses modes to determine how to display/act:
   1 - Add
   2 - Modify
   3 - Delete
*/
gint		log_mode;
char		days[7] = {"smtwhfa"};
char		months[12][4] = {"Jan\x00", "Feb\x00", "Mar\x00", "Apr\x00", "May\x00", "Jun\x00", 
			                 "Jul\x00", "Aug\x00", "Sep\x00", "Oct\x00", "Nov\x00", "Dec\x00"};

void
show_log_entry                         (gint     in_mode,
										gboolean use_faceplate,
							 			gint	 in_row)
{
	int			i, j;
	gint		len, pos;
	gchar*		log_mode_str;
	gchar*		log_file_name;
	gchar*		log_title_str;
	gchar*		freq_str;
	gchar*		mode_str;
	gchar*		time_str = NULL;
	gchar*		date_str = NULL;
	gchar*		temp;
	gchar*		text[9];
	GtkWidget*	gen_item;
	time_t    	s_time;
	struct tm 	g_time;
	
	log_mode = in_mode;
	if (log_name == NULL) {
		show_message("Log Entry Error","No log is currently loaded.\nPlease load a log file before doing a log function.","warning");
	}
	else {
		switch (log_mode) {
			case 1 : log_mode_str = g_strdup("Add"); break;
			case 2 : log_mode_str = g_strdup("Modify"); break; 
			case 3 : log_mode_str = g_strdup("Delete"); break;
			default : log_mode_str = g_strdup("");
		}
		log_file_name = get_filename_from_full_path(log_name);
		log_title_str = g_strdup_printf("%s Log Entry - [%s]",log_mode_str,log_file_name);
		gtk_window_set_title(GTK_WINDOW(lentry),log_title_str);
		g_free(log_file_name);
		g_free(log_title_str);

		set_enabled(lentry,"LEDateLbl",(log_mode != 3));
		set_enabled(lentry,"LEDateEntry",(log_mode != 3));
		set_enabled(lentry,"LETimeLbl",(log_mode != 3));
		set_enabled(lentry,"LETimeEntry",(log_mode != 3));
		set_enabled(lentry,"LEFreqLbl",(log_mode != 3));
		set_enabled(lentry,"LEFreqEntry",(log_mode != 3));
		set_enabled(lentry,"LEkHzRBtn",(log_mode != 3));
		set_enabled(lentry,"LEMHzRBtn",(log_mode != 3));
		set_enabled(lentry,"LECallLbl",(log_mode != 3));
		set_enabled(lentry,"LECallEntry",(log_mode != 3));
		set_enabled(lentry,"LEStationLbl",(log_mode != 3));
		set_enabled(lentry,"LEStationBox",(log_mode != 3));
		set_enabled(lentry,"LESINPOLbl",(log_mode != 3));
		set_enabled(lentry,"LESINPOEntry",(log_mode != 3));
		set_enabled(lentry,"LEModeLbl",(log_mode != 3));
		set_enabled(lentry,"LEModeEntry",(log_mode != 3));
		set_enabled(lentry,"LECountryLbl",(log_mode != 3));
		set_enabled(lentry,"LECountryBox",(log_mode != 3));
		set_enabled(lentry,"LogEntryText",(log_mode != 3));
		
		gen_item = lookup_widget(lentry,"LogEntryText");
		if (gen_item) {
			len = gtk_text_get_length(GTK_TEXT(gen_item));
			if (len > 0)
				gtk_editable_delete_text(GTK_EDITABLE(gen_item),0,len);
		}

		if (log_mode == 1) {
			if (ispower) {
				freq_str = g_strdup_printf("%8.2f",frequency);
				set_text(lentry,"LEFreqEntry",freq_str);
				set_down(lentry,"LEkHzRBtn",TRUE);
				set_down(lentry,"LEMHzRBtn",FALSE);
				mode_str = get_mode_str(mode);
				set_text(lentry,"LEModeEntry",mode_str);
				g_free(freq_str);
				g_free(mode_str);
			}
			else {
				set_text(lentry,"LEFreqEntry"," ");
				set_down(lentry,"LEkHzRBtn",FALSE);
				set_down(lentry,"LEMHzRBtn",FALSE);
				set_text(lentry,"LEModeEntry"," ");
			}
			
  			time(&s_time);
  			gmtime_r(&s_time,&g_time);
			time_str = g_strdup_printf("%02d%02d%c", g_time.tm_hour, g_time.tm_min, (char)days[g_time.tm_wday]);
			date_str = g_strdup_printf("%02d %s %04d", g_time.tm_mday, months[g_time.tm_mon], g_time.tm_year);
			date_str = strchrdel(date_str,7,2);
			set_text(lentry,"LETimeEntry",time_str);
			set_text(lentry,"LEDateEntry",date_str);
			g_free(time_str);
			g_free(date_str);

			set_text(lentry,"LESINPOEntry"," ");
			set_text(lentry,"LECallEntry"," ");
			
			if (use_faceplate) {
				gen_item = lookup_widget(app1,"clist1");
				if (gen_item) {
					gtk_clist_get_text(GTK_CLIST(gen_item), in_row, 2, &text[7]);
					gtk_clist_get_text(GTK_CLIST(gen_item), in_row, 3, &text[6]);
					set_text(lentry,"LECountryEntry",text[6]);
					set_text(lentry,"LEStationEntry",text[7]);
				}					
			}
			else {
				set_text(lentry,"LECountryEntry"," ");
				set_text(lentry,"LEStationEntry"," ");
			}
		}
		else {
			gen_item = lookup_widget(log1,"LogList");
			if (gen_item) {
				j = get_list_selected_row(log1,"LogList");
				for (i = 0; i < 8; i++)
					gtk_clist_get_text(GTK_CLIST(gen_item), j, i, &text[i]);
				text[8] = g_list_nth_data(log_details,j);
				len = strlen(text[0])-4;
				temp = g_strndup(text[0],len);
				set_text(lentry,"LEFreqEntry",temp);
				if (text[0][len+1] == 'k')
					set_down(lentry,"LEkHzRBtn",TRUE);
				if (text[0][len+1] == 'M')
					set_down(lentry,"LEMHzRBtn",TRUE);
				set_text(lentry,"LETimeEntry",text[1]);
				set_text(lentry,"LEDateEntry",text[2]);
				set_text(lentry,"LEModeEntry",text[3]);
				set_text(lentry,"LESINPOEntry",text[4]);
				set_text(lentry,"LECallEntry",text[5]);
				set_text(lentry,"LECountryEntry",text[6]);
				set_text(lentry,"LEStationEntry",text[7]);
			}
			gen_item = lookup_widget(lentry,"LogEntryText");
			if (gen_item) {
				if (text[8]) {
					pos = 0;
					len = strlen(text[8]);
					if (len > 0)
						gtk_editable_insert_text(GTK_EDITABLE(gen_item),text[8],len,&pos);
				}
			}
		}
		gtk_widget_show(lentry);
	}
}


void
process_log_entry()
{
	if (log_name == NULL)
		show_message("Log Entry Error","No log is currently loaded.\nPlease load a log file before doing a log function.","warning");
	else {
		switch (log_mode) {
			case 1 : add_log_entry(); break;
			case 2 : modify_log_entry(); break;
			case 3 : delete_log_entry(); break;
		}
	}
}


void
add_log_entry()
{
	int				j;
	guint			len;
	char        	buf[10000];
	FILE			*fp1;
	FILE			*fp2;
	gchar*          crlf;
	gchar*          tab;
	gchar*			tabtab;
	gchar*			temp = NULL;
	gchar*			temp2;
	gchar*			text[9];
	gchar*			file_path;
	gchar*			new_file;
	GnomeVFSResult	result;
	GnomeVFSURI*	uri1;
	GnomeVFSURI*	uri2;
	double			r_freq;
	GtkWidget*		gen_item;
	
	file_path = get_pathname_from_full_path(log_name);
	new_file = g_strdup_printf("%s%s",file_path,"temp.log");
	g_free(file_path);
	
	fp1 = fopen(log_name,"r");
	if (!fp1) {
		temp = g_strdup_printf("Creating new log file:\n%s",log_name);
		show_message("Add Log Entry",temp,"info");
		g_free(temp);
	}
	fp2 = fopen(new_file,"w");
	if (!fp2) {
		temp = g_strdup_printf("Can't create temporary file.\nAborting...");
		show_message("Add Log Entry",temp,"warning");
		g_free(temp);
	}
	else {
		/* read and write the rest of the logs */
		if (fp1) {
			while (!feof(fp1)) {
				for (j = 0; j < 10000; j++)
					buf[j] = 0x00;
				fgets(buf,10000,fp1);
				fputs(buf,fp2);
			}
		}
		
		/* write new log line at the end */
		if (fp1)
			fgets(buf,10000,fp1);
		crlf = g_strdup_printf("%c%c",0x0d,0x0a);
		tab = g_strdup_printf("%c",0x09);
		tabtab = g_strdup_printf("%c%c%c",0xff,0x09,0x09);
		get_text(lentry,"LEFreqEntry",&text[0]);
		text[0] = g_strstrip(text[0]);
		r_freq = g_strtod(text[0],NULL);
		if (get_down(lentry,"LEkHzRBtn"))
			temp = g_strdup_printf("%8.2f%skHz",r_freq,tab);
		if (get_down(lentry,"LEMHzRBtn"))
			temp = g_strdup_printf("%8.5f%sMHz",r_freq,tab);
		get_text(lentry,"LETimeEntry",&text[1]);
		g_strstrip(text[1]);			
		get_text(lentry,"LEDateEntry",&text[2]);
		g_strstrip(text[2]);
		get_text(lentry,"LEModeEntry",&text[3]);
		g_strstrip(text[3]);
		get_text(lentry,"LESINPOEntry",&text[4]);
		g_strstrip(text[4]);
		get_text(lentry,"LECallEntry",&text[5]);
		g_strstrip(text[5]);
		get_text(lentry,"LECountryEntry",&text[6]);
		g_strstrip(text[6]);
		get_text(lentry,"LEStationEntry",&text[7]);
		g_strstrip(text[7]);
		gen_item = lookup_widget(lentry,"LogEntryText");
		len = gtk_text_get_length(GTK_TEXT(gen_item));
		text[8] = gtk_editable_get_chars(GTK_EDITABLE(gen_item),0,len);
		text[8] = g_strdelimit(text[8],crlf,0x20);
		g_strstrip(text[8]);
		temp2 = g_strdup_printf("%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s\n", temp,tab,
																		text[1],tab,
																		text[2],tab,
																		text[3],tab,
																		text[6],tab,
																		text[4],tab,
																		text[5],tab,
																		text[7],tabtab,
																		text[8]);
		fputs(temp2,fp2);
		g_free(crlf);
		g_free(tab);
		g_free(tabtab);
		g_free(temp);
		g_free(temp2);
		
		fclose(fp2);
		if (fp1)
			fclose(fp1);
		
		/* now mv new one into old one */
		if (!gnome_vfs_initialized())
			gnome_vfs_init();	
		uri1 = gnome_vfs_uri_new (log_name);
		uri2 = gnome_vfs_uri_new (new_file);
		result = gnome_vfs_move_uri(uri2, uri1, TRUE);
		if (result != GNOME_VFS_OK)
			show_message("Delete Log Entry", (gchar*)gnome_vfs_result_to_string(result), "error");
			/* reload the log file */
		open_logfile();
	}
}

void
modify_log_entry()
{
	int				i, j, log_num;
	guint			len;
	char        	buf[10000];
	FILE			*fp1;
	FILE			*fp2;
	gchar*          tab;
	gchar*          crlf;
	gchar*			tabtab;
	gchar*			temp = NULL;
	gchar*			temp2;
	gchar*			text[9];
	gchar*			file_path;
	gchar*			new_file;
	GnomeVFSResult	result;
	GnomeVFSURI*	uri1;
	GnomeVFSURI*	uri2;
	double			r_freq;
	GtkWidget*		gen_item;
	
	log_num = get_list_selected_row(log1,"LogList");
	file_path = get_pathname_from_full_path(log_name);
	new_file = g_strdup_printf("%s%s",file_path,"temp.log");
	g_free(file_path);
	
	fp1 = fopen(log_name,"r");
	if (!fp1) {
		temp = g_strdup_printf("Can't open %s.\nAborting...",log_name);
		show_message("Modify Log Entry",temp,"warning");
		g_free(temp);
	}
	else {
		fp2 = fopen(new_file,"w");
		if (!fp2) {
			temp = g_strdup_printf("Can't create temporary file.\nAborting...");
			show_message("Modify Log Entry",temp,"warning");
			g_free(temp);
		}
		else {
			/* read and write all logs until the one to modify */
			for (i = 0; i < log_num; i++) {
				for (j = 0; j < 10000; j++)
					buf[j] = 0x00;
				fgets(buf,10000,fp1);
				fputs(buf,fp2);
			}
			
			/* read, but write new log line */
			fgets(buf,10000,fp1);
			crlf = g_strdup_printf("%c%c",0x0d,0x0a);
			tab = g_strdup_printf("%c",0x09);
			tabtab = g_strdup_printf("%c%c%c",0xff,0x09,0x09);
			get_text(lentry,"LEFreqEntry",&text[0]);
			g_strstrip(text[0]);
			r_freq = g_strtod(text[0],NULL);
			if (get_down(lentry,"LEkHzRBtn"))
				temp = g_strdup_printf("%8.2f%skHz",r_freq,tab);
			if (get_down(lentry,"LEMHzRBtn"))
				temp = g_strdup_printf("%8.5f%sMHz",r_freq,tab);
			get_text(lentry,"LETimeEntry",&text[1]);
			g_strstrip(text[1]);			
			get_text(lentry,"LEDateEntry",&text[2]);
			g_strstrip(text[2]);
			get_text(lentry,"LEModeEntry",&text[3]);
			g_strstrip(text[3]);
			get_text(lentry,"LESINPOEntry",&text[4]);
			g_strstrip(text[4]);
			get_text(lentry,"LECallEntry",&text[5]);
			g_strstrip(text[5]);
			get_text(lentry,"LECountryEntry",&text[6]);
			g_strstrip(text[6]);
			get_text(lentry,"LEStationEntry",&text[7]);
			g_strstrip(text[7]);
			gen_item = lookup_widget(lentry,"LogEntryText");
			len = gtk_text_get_length(GTK_TEXT(gen_item));
			text[8] = gtk_editable_get_chars(GTK_EDITABLE(gen_item),0,len);
			text[8] = g_strdelimit(text[8],crlf,0x20);
			g_strstrip(text[8]);
			temp2 = g_strdup_printf("%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s\n", temp,tab,
																			text[1],tab,
																			text[2],tab,
																			text[3],tab,
																			text[6],tab,
																			text[4],tab,
																			text[5],tab,
																			text[7],tabtab,
																			text[8]);
			fputs(temp2,fp2);
			g_free(crlf);
			g_free(tab);
			g_free(tabtab);
			g_free(temp);
			g_free(temp2);
			
			/* read and write the rest of the logs */
			while (!feof(fp1)) {
				for (j = 0; j < 10000; j++)
					buf[j] = 0x00;
				fgets(buf,10000,fp1);
				fputs(buf,fp2);
			}
			fclose(fp2);
			fclose(fp1);
			
			/* now mv new one into old one */
			if (!gnome_vfs_initialized())
				gnome_vfs_init();	
			uri1 = gnome_vfs_uri_new (log_name);
			uri2 = gnome_vfs_uri_new (new_file);
			result = gnome_vfs_move_uri(uri2, uri1, TRUE);
			if (result != GNOME_VFS_OK)
				show_message("Delete Log Entry", (gchar*)gnome_vfs_result_to_string(result), "error");

			/* reload the log file */
			open_logfile();
			set_list_selected_row(log1,"LogList", log_num);
		}
	}
}


void
delete_log_entry()
{
	int				i, j, log_num, max_row;
	char        	buf[10000];
	FILE			*fp1;
	FILE			*fp2;
	gchar*			temp;
	gchar*			file_path;
	gchar*			new_file;
	GnomeVFSResult	result;
	GnomeVFSURI*	uri1;
	GnomeVFSURI*	uri2;
	
	log_num = get_list_selected_row(log1,"LogList");
	file_path = get_pathname_from_full_path(log_name);
	new_file = g_strdup_printf("%s%s",file_path,"temp.log");
	g_free(file_path);
	
	fp1 = fopen(log_name,"r");
	if (!fp1) {
		temp = g_strdup_printf("Can't open %s.\nAborting...",log_name);
		show_message("Delete Log Entry",temp,"warning");
		g_free(temp);
	}
	else {
		fp2 = fopen(new_file,"w");
		if (!fp2) {
			temp = g_strdup_printf("Can't create temporary file.\nAborting...");
			show_message("Delete Log Entry",temp,"warning");
			g_free(temp);
		}
		else {
			/* read and write all logs until the one to delete */
			for (i = 0; i < log_num; i++) {
				for (j = 0; j < 10000; j++)
					buf[j] = 0x00;
				fgets(buf,10000,fp1);
				fputs(buf,fp2);
			}
			
			/* read, but skip writing the log line */
			fgets(buf,10000,fp1);
			
			/* read and write the rest of the logs */
			while (!feof(fp1)) {
				for (j = 0; j < 10000; j++)
					buf[j] = 0x00;
				fgets(buf,10000,fp1);
				fputs(buf,fp2);
			}
			fclose(fp2);
			fclose(fp1);
			
			/* now mv new one into old one */
			if (!gnome_vfs_initialized())
				gnome_vfs_init();	
			uri1 = gnome_vfs_uri_new (log_name);
			uri2 = gnome_vfs_uri_new (new_file);
			result = gnome_vfs_move_uri(uri2, uri1, TRUE);
			if (result != GNOME_VFS_OK)
				show_message("Delete Log Entry", (gchar*)gnome_vfs_result_to_string(result), "error");

			/* reload the log file */
			open_logfile();
			max_row = get_list_num_rows(log1,"LogList")-1;
			if (log_num > max_row)
				log_num = max_row;
			set_list_selected_row(log1,"LogList", log_num);
		}
	}
}


