#define _XOPEN_SOURCE 500

#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <linux/major.h> 
typedef unsigned char u_char;   /* horrible, for scsi.h */
#include "sg_include.h"
#include "sg_err.h"
#include "llseek.h"

/* A utility program for the Linux OS SCSI generic ("sg") device driver.
*  Copyright (C) 2001, 2002 D. Gilbert
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2, or (at your option)
*  any later version.

   This program reads data from the given SCSI device (typically a disk
   or cdrom) and discards that data. Its primary goal is to time
   multiple reads all from the same logical address. Its interface
   is a subset of another member of this package: sg_dd which is a 
   "dd" variant. The input file can be a scsi generic device, a raw device
   or a seekable file. Streams such as stdin are not acceptable.
   The block size ('bs') is assumed to be 512 if not given. 
   Various arguments can take a multiplier suffix:
     'c','C'  *1       'b','B' *512      'k' *1024      'K' *1000
     'm' *(1024^2)     'M' *(1000^2)     'g' *(1024^3)  'G' *(1000^3)

   The "bpt" (blocks per transfer) argument controls the maximum number
   of blocks in each transfer. The default value is 128.
   For example if "bs=512" and "bpt=32" then a maximum of 32 blocks (16KB
   in this case) is read from the sg device in a single SCSI command.

   This version should compile with Linux sg drivers with version numbers
   >= 20000 . 

*/

static const char * version_str = "0.53 20020226";

#define ME "sg_read: "

#define DEF_BLOCK_SIZE 512
#define DEF_BLOCKS_PER_TRANSFER 128

/* #define SG_DEBUG */

#define SG_HEAD_SZ sizeof(struct sg_header)
#define SCSI_CMD10_LEN 10
#define DEF_TIMEOUT 40000       /* 40,000 millisecs == 40 seconds */

#ifndef RAW_MAJOR
#define RAW_MAJOR 255   /*unlikey value */
#endif 

#define FT_OTHER 0              /* filetype other than sg or raw device */
#define FT_SG 1                 /* filetype is sg char device */
#define FT_RAW 2                /* filetype is raw char device */

#define STR_SZ 1024
#define INF_SZ 512
#define EBUFF_SZ 512


static int dd_count = -1;
static int in_full = 0;
static int in_partial = 0;

static int pack_id_count = 0;

static unsigned char rdCmdBlk[SCSI_CMD10_LEN] =
                    {0x28, 0, 0, 0, 0, 0, 0, 0, 0, 0};


static void install_handler (int sig_num, void (*sig_handler) (int sig))
{
    struct sigaction sigact;
    sigaction (sig_num, NULL, &sigact);
    if (sigact.sa_handler != SIG_IGN)
    {
        sigact.sa_handler = sig_handler;
        sigemptyset (&sigact.sa_mask);
        sigact.sa_flags = 0;
        sigaction (sig_num, &sigact, NULL);
    }
}

void print_stats(int iters)
{
    if (0 != dd_count)
        fprintf(stderr, "  remaining block count=%d\n", dd_count);
    fprintf(stderr, "%d+%d records in", in_full - in_partial, in_partial);
    if (iters > 0)
	fprintf(stderr, ", SCSI commands issued: %d\n", iters);
    else
	fprintf(stderr, "\n");
}

static void interrupt_handler(int sig)
{
    struct sigaction sigact;

    sigact.sa_handler = SIG_DFL;
    sigemptyset (&sigact.sa_mask);
    sigact.sa_flags = 0;
    sigaction (sig, &sigact, NULL);
    fprintf(stderr, "Interrupted by signal,");
    print_stats(0);
    kill (getpid (), sig);
}

static void siginfo_handler(int sig)
{
    fprintf(stderr, "Progress report, continuing ...\n");
    print_stats(0);
}

int dd_filetype(const char * filename)
{
    struct stat st;

    if (stat(filename, &st) < 0)
        return FT_OTHER;
    if (S_ISCHR(st.st_mode)) {
        if (RAW_MAJOR == major(st.st_rdev))
            return FT_RAW;
        else if (SCSI_GENERIC_MAJOR == major(st.st_rdev))
            return FT_SG;
    }
    return FT_OTHER;
}

void usage()
{
    fprintf(stderr, "Usage: "
           "sg_read  if=<infile> [skip=<num>] [bs=<num>] [bpt=<num>] "
           "count=<num>\n"
           "                         [time=<num>]\n"
           " 'if'  is an sg or raw device, or a seekable file (not stdin)\n"
           " 'bs'  must match sector size (when 'if' is sg device) "
	   "(def=512)\n"
           " 'skip' each transfer starts at this logical address (def=0)\n"
           " 'count' total bytes read will be 'bs'*'count' (if no error)\n"
           " 'bpt' is blocks_per_transfer (default is 128, or 64KB for "
	   "def 'bs')\n"
           " 'time' 0->do nothing(def), 1->time from 1st cmd, 2->time "
           "from 2nd cmd\n");
    fprintf(stderr, "\nVersion: %s\n", version_str);
}

/* -1 -> unrecoverable error, 0 -> successful, 1 -> recoverable (ENOMEM),
   2 -> try again */
int sg_bread(int sg_fd, unsigned char * buff, int blocks, int from_block,
             int bs)
{
    int inLen = SG_HEAD_SZ + SCSI_CMD10_LEN;
    int outLen, res;
    unsigned char * rdCmd = buff + SG_HEAD_SZ;
    struct sg_header * isghp = (struct sg_header *)buff;
    struct sg_header * osghp = (struct sg_header *)(buff + SCSI_CMD10_LEN);
    
    outLen = SG_HEAD_SZ + (bs * blocks);
    isghp->pack_len = 0;                /* don't care */
    isghp->pack_id = pack_id_count++;
    isghp->reply_len = outLen;
    isghp->twelve_byte = 0;
    isghp->result = 0;
#ifndef SG_GET_RESERVED_SIZE
    isghp->sense_buffer[0] = 0;
#endif
    memcpy(rdCmd, rdCmdBlk, SCSI_CMD10_LEN);
    rdCmd[2] = (unsigned char)((from_block >> 24) & 0xFF);
    rdCmd[3] = (unsigned char)((from_block >> 16) & 0xFF);
    rdCmd[4] = (unsigned char)((from_block >> 8) & 0xFF);
    rdCmd[5] = (unsigned char)(from_block & 0xFF);
    rdCmd[7] = (unsigned char)((blocks >> 8) & 0xff);
    rdCmd[8] = (unsigned char)(blocks & 0xff);

    while (((res = write(sg_fd, buff, inLen)) < 0) && (EINTR == errno))
        ;
    if (res < 0) {
        if (ENOMEM == errno)
            return 1;
        perror("reading (wr) on sg device, error");
        return -1;
    }
    if (res < inLen) {
        fprintf(stderr, "reading (wr) on sg device, problems, "
		"ask=%d, got=%d\n", inLen, res);
        return -1;
    }

    while (((res = read(sg_fd, buff + SCSI_CMD10_LEN, outLen)) < 0) &&
           (EINTR == errno))
        ;
    if (res < 0) {
        perror("reading (rd) on sg device, error");
        return -1;
    }
    if (res < outLen)
    {
        fprintf(stderr, "reading (rd) on sg device, problems, "
		"ask=%d, got=%d\n", outLen, res);
        return -1;
    }
#ifdef SG_GET_RESERVED_SIZE
    res = sg_err_category(osghp->target_status, osghp->host_status,
                          osghp->driver_status, osghp->sense_buffer, 
                          SG_MAX_SENSE);
    switch (res) {
    case SG_ERR_CAT_CLEAN:
        break;
    case SG_ERR_CAT_RECOVERED:
        fprintf(stderr, "Recovered error while reading block=%d, num=%d\n",
               from_block, blocks);
        break;
    case SG_ERR_CAT_MEDIA_CHANGED:
        return 2;
    default:
        sg_chk_n_print("reading", osghp->target_status, 
                       osghp->host_status, osghp->driver_status, 
                       osghp->sense_buffer, SG_MAX_SENSE);
        return -1;
    }
#else
    if ((osghp->result != 0) || (0 != osghp->sense_buffer[0])) {
        fprintf(stderr, "reading result=%d\n", osghp->result);
        if (0 != osghp->sense_buffer[0])
            sg_print_sense("after read(rd)", osghp->sense_buffer, SG_MAX_SENSE);
        return -1;
    }
#endif
    return 0;
}

int get_num(char * buf)
{
    int res, num;
    char c;

    res = sscanf(buf, "%d%c", &num, &c);
    if (0 == res)
        return -1;
    else if (1 == res)
        return num;
    else {
        switch (c) {
        case 'c':
        case 'C':
            return num;
        case 'b':
        case 'B':
            return num * 512;
        case 'k':
            return num * 1024;
        case 'K':
            return num * 1000;
        case 'm':
            return num * 1024 * 1024;
        case 'M':
            return num * 1000000;
        case 'g':
            return num * 1024 * 1024 * 1024;
        case 'G':
            return num * 1000000000;
        default:
            fprintf(stderr, "unrecognized multiplier\n");
            return -1;
        }
    }
}


int main(int argc, char * argv[])
{
    int skip = 0;
    int bs = 0;
    int bpt = DEF_BLOCKS_PER_TRANSFER;
    char str[STR_SZ];
    char * key;
    char * buf;
    char inf[INF_SZ];
    int in_type = FT_OTHER;
    int do_time = 0;
    int res, k, t, buf_sz, iters, orig_count;
    int infd, blocks;
    unsigned char * wrkBuff = NULL;
    unsigned char * wrkPos;
    char ebuff[EBUFF_SZ];
    int blocks_per;
    struct timeval start_tm, end_tm;
    size_t psz = getpagesize();

    inf[0] = '\0';
    if (argc < 3) {
        fprintf(stderr, "'if' and 'count' arguments must be given\n");
        usage();
        return 1;
    }

    for(k = 1; k < argc; k++) {
        if (argv[k]) {
            strncpy(str, argv[k], STR_SZ);
	    str[STR_SZ - 1] = '\0';
	}
        else
            continue;
        for(key = str, buf = key; *buf && *buf != '=';)
            buf++;
        if (*buf)
            *buf++ = '\0';
        if (strcmp(key,"if") == 0)
            strncpy(inf, buf, INF_SZ);
        else if (0 == strcmp(key,"bs"))
            bs = get_num(buf);
        else if (0 == strcmp(key,"bpt"))
            bpt = get_num(buf);
        else if (0 == strcmp(key,"skip"))
            skip = get_num(buf);
        else if (0 == strcmp(key,"count"))
            dd_count = get_num(buf);
        else if (0 == strcmp(key,"time"))
            do_time = get_num(buf);
        else {
            fprintf(stderr, "Unrecognized argument '%s'\n", key);
            usage();
            return 1;
        }
    }
    if (bs <= 0) {
        bs = DEF_BLOCK_SIZE;
        fprintf(stderr, "Assume default 'bs' (block size) of %d bytes\n", bs);
    }
    if (dd_count < 0) {
        fprintf(stderr, "'count' must be given\n");
        usage();
        return 1;
    }
    if (skip < 0) {
        fprintf(stderr, "skip cannot be negative\n");
        return 1;
    }

#ifdef SG_DEBUG
    fprintf(stderr, ME "if=%s skip=%d count=%d\n", inf, skip, dd_count);
#endif
    install_handler (SIGINT, interrupt_handler);
    install_handler (SIGQUIT, interrupt_handler);
    install_handler (SIGPIPE, interrupt_handler);
    install_handler (SIGUSR1, siginfo_handler);

    if (! inf[0]) {
        fprintf(stderr, "must provide 'if=<filename>'\n");
        usage();
        return 1;
    }
    in_type = dd_filetype(inf);

    if (FT_SG == in_type) {
        if ((infd = open(inf, O_RDWR)) < 0) {
            snprintf(ebuff, EBUFF_SZ, ME "could not open %s for sg reading",
	    	     inf);
            perror(ebuff);
            return 1;
        }
        t = bs * bpt;
        res = ioctl(infd, SG_SET_RESERVED_SIZE, &t);
        if (res < 0) {
            perror(ME "SG_SET_RESERVED_SIZE error");
	    return 1;
	}
    }
    else {
        if ((infd = open(inf, O_RDONLY)) < 0) {
            snprintf(ebuff, EBUFF_SZ, ME "could not open %s for reading", inf);
            perror(ebuff);
            return 1;
        }
        else if (skip > 0) {
            llse_loff_t offset = skip;

            offset *= bs;       /* could exceed 32 bits here! */
            if (llse_llseek(infd, offset, SEEK_SET) < 0) {
                snprintf(ebuff, EBUFF_SZ, ME "couldn't skip to required "
			 "position on %s", inf);
                perror(ebuff);
                return 1;
            }
        }
    }

    if (0 == dd_count)
        return 0;
    orig_count = dd_count;

    if (FT_RAW == in_type) {
        wrkBuff = malloc(bs * bpt + psz);
        if (0 == wrkBuff) {
            fprintf(stderr, "Not enough user memory for raw\n");
            return 1;
        }
        wrkPos = (unsigned char *)(((unsigned long)wrkBuff + psz - 1) &
                                   (~(psz - 1)));
    }
    else {
        wrkBuff = malloc(bs * bpt);
        if (0 == wrkBuff) {
            fprintf(stderr, "Not enough user memory\n");
            return 1;
        }
        wrkPos = wrkBuff;
    }

    blocks_per = bpt;
#ifdef SG_DEBUG
    fprintf(stderr, "Start of loop, count=%d, blocks_per=%d\n", 
            dd_count, blocks_per);
#endif
    start_tm.tv_sec = 0;   /* just in case start set condition not met */
    start_tm.tv_usec = 0;

    /* main loop */
    for (iters = 0; dd_count > 0; ++iters) {
	if ((do_time > 0) && (iters == (do_time - 1)))
	    gettimeofday(&start_tm, NULL);
        blocks = (dd_count > blocks_per) ? blocks_per : dd_count;
        if (FT_SG == in_type) {
            res = sg_bread(infd, wrkPos, blocks, skip, bs);
            if (1 == res) {     /* ENOMEM, find what's available+try that */
                if (ioctl(infd, SG_GET_RESERVED_SIZE, &buf_sz) < 0) {
                    perror("RESERVED_SIZE ioctls failed");
                    break;
                }
                blocks_per = (buf_sz + bs - 1) / bs;
                blocks = blocks_per;
                fprintf(stderr, 
                        "Reducing read to %d blocks per loop\n", blocks_per);
                res = sg_bread(infd, wrkPos, blocks, skip, bs);
            }
            else if (2 == res) {
                fprintf(stderr, 
                        "Unit attention, media changed, continuing (r)\n");
                res = sg_bread(infd, wrkPos, blocks, skip, bs);
            }
            if (0 != res) {
                fprintf(stderr, ME "failed, skip=%d\n", skip);
                break;
            }
            else {
                in_full += blocks;
            }
        }
        else {
	    if (iters > 0) { /* subsequent iteration reset skip position */
		llse_loff_t offset = skip;

		offset *= bs;       /* could exceed 32 bits here! */
		if (llse_llseek(infd, offset, SEEK_SET) < 0) {
		    perror(ME "could not reset skip position");
		    break;
		}
	    }
            while (((res = read(infd, wrkPos, blocks * bs)) < 0) &&
                   (EINTR == errno))
                ;
            if (res < 0) {
                snprintf(ebuff, EBUFF_SZ, ME "reading, skip=%d ", skip);
                perror(ebuff);
                break;
            }
            else if (res < blocks * bs) {
		fprintf(stderr, ME "short read: wanted/got=%d/%d bytes"
			", stop\n", blocks * bs, res);
                blocks = res / bs;
                if ((res % bs) > 0) {
                    blocks++;
                    in_partial++;
                }
		dd_count -= blocks;
		in_full += blocks;
		break;
            }
            in_full += blocks;
        }
        if (dd_count > 0)
            dd_count -= blocks;
    }
    if (do_time > 0) {
	gettimeofday(&end_tm, NULL);
	if (start_tm.tv_sec || start_tm.tv_usec) {
	    struct timeval res_tm;
	    double a, b, c;

	    res_tm.tv_sec = end_tm.tv_sec - start_tm.tv_sec;
	    res_tm.tv_usec = end_tm.tv_usec - start_tm.tv_usec;
	    if (res_tm.tv_usec < 0) {
		--res_tm.tv_sec;
		res_tm.tv_usec += 1000000;
	    }
	    a = res_tm.tv_sec;
	    a += (0.000001 * res_tm.tv_usec);
	    b = (double)bs * (orig_count - dd_count);
	    if (do_time > 1)
		c = b - ((double)bs * ((do_time - 1.0) * bpt));
	    else
		c = 0.0;

	    if (1 == do_time) {
		fprintf(stderr, "time for all (SCSI) commands was "
		    "%d.%06d secs", (int)res_tm.tv_sec, (int)res_tm.tv_usec);
		if ((a > 0.00001) && (b > 511))
		    fprintf(stderr, ", %.2f MB/sec\n", b / (a * 1000000.0));
		else
		    fprintf(stderr, "\n");
	    }
	    else if (2 == do_time) {
		fprintf(stderr, "time from second (SCSI) command to end "
		    "was %d.%06d secs", (int)res_tm.tv_sec, 
		    (int)res_tm.tv_usec);
		if ((a > 0.00001) && (c > 511))
		    fprintf(stderr, ", %.2f MB/sec\n", c / (a * 1000000.0));
		else
		    fprintf(stderr, "\n");
	    }
	    else { 
		fprintf(stderr, "time from start of (SCSI) command "
			"#%d to end was %d.%06d secs", do_time,
			(int)res_tm.tv_sec, (int)res_tm.tv_usec);
		if ((a > 0.00001) && (c > 511))
		    fprintf(stderr, ", %.2f MB/sec\n", c / (a * 1000000.0));
		else
		    fprintf(stderr, "\n");
	    }
	}
    }

    if (wrkBuff)
	free(wrkBuff);

    close(infd);
    res = 0;
    if (0 != dd_count) {
        fprintf(stderr, "Some error occurred,");
        res = 2;
    }
    if (FT_SG == in_type)
	print_stats(iters);
    else
	print_stats(0);

    return res;
}
