/*
 * Copyright (C) 1999, 2000  Upi Tamminen <desaster@imnetti.fi>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <signal.h>

#include "reppu.h"

extern int debug;

void server(int foreground)
{
    char reppudir[512];
    int sockfd, new_fd, sin_size;
    struct sockaddr_in my_addr, their_addr;

    sprintf(reppudir, "%s/.reppu", getenv("HOME"));
    mkdir(reppudir, 0700);
    if (chdir(reppudir) != 0) {
	fprintf(stderr, "failed to chdir %s\n", reppudir);
	exit(1);
    }
    
    sockfd = socket(AF_INET, SOCK_STREAM, 0);
    
    my_addr.sin_family = AF_INET;
    my_addr.sin_port = htons(REPPU_PORT);
    my_addr.sin_addr.s_addr = htonl(INADDR_ANY);
    bzero(&(my_addr.sin_zero), 8);
    
    if (bind(sockfd, (struct sockaddr *)&my_addr,
		sizeof(struct sockaddr)) == -1) {
	perror("bind");
	exit(1);
    }

    if (listen(sockfd, 2) == -1) {
	perror("listen");
	exit(1);
    }
    
    if (!foreground) {

	int pid;

	if ((pid = fork()) == -1) {
	    perror("fork");
	    exit(1);
	}
	if (pid) {
	    printf("reppu %s server started (pid %d)\n", REPPU_VERSION, pid);
	    exit(0);
	}
	signal(SIGUSR1, sighan);
	signal(SIGHUP, SIG_IGN);
	signal(SIGTERM, sighan);
	signal(SIGINT, sighan);
	signal(SIGQUIT, sighan);
	signal(SIGTSTP, sighan);
	signal(SIGUSR2, SIG_IGN);
	signal(SIGPIPE, sighan);
    } else printf("reppu %s server started\n", REPPU_VERSION);

    sin_size = sizeof(struct sockaddr_in);

    for (;;) {
	new_fd = accept(sockfd, &their_addr, &sin_size);
	if (debug) printf("Incoming connection: %s\n", inet_ntoa(their_addr.sin_addr));

	server_loop(new_fd);
	
	if (debug) printf("Closing connection\n\n");
	close(new_fd);
    }
}

void server_loop(int new_fd) {
    
    char buf[MAXDATASIZE];

    for (;;) {

	buf[recv(new_fd, buf, MAXDATASIZE, 0)] = '\0';

	if (strcmp(buf, "BLEND") == 0) {

	    char recv_buf[MAXDATASIZE];
	    long total_size, pos = 0;
	    FILE *blend_file;
	    
	    if (debug) printf("Received request: BLEND\n");
	    if (send(new_fd, "OK", 2, 0) == -1) return;
	    recv_buf[recv(new_fd, recv_buf, MAXDATASIZE, 0)] = '\0';
	    
	    total_size = atol(recv_buf);
	    if (debug) printf("Receiving %ld bytes ...\n", total_size);
	    if (send(new_fd, "OK", 2, 0) == -1) return;
	    
	    if ((blend_file = fopen(REMOTE_BLEND, "w")) == NULL) {
		fprintf(stderr, "error opening %s\n", REMOTE_BLEND);
		exit(1);
	    }
	    
	    while (pos < total_size) {
		
		int size;
		size = recv(new_fd, &recv_buf, MAXDATASIZE, 0);
		pos += size;
		fwrite(recv_buf, size, 1, blend_file);
	    }
	    
	    fclose(blend_file);
	    if (debug) printf("Finished receiving %s\n", REMOTE_BLEND);
	    if (send(new_fd, "OK", 2, 0) == -1) return;
	}
	
	else if (strcmp(buf, "RENDER") == 0) {
	    
	    char recv_buf[MAXDATASIZE], arglist[MAXDATASIZE], msg[MAXDATASIZE];
	    char stdout_buf[MAXDATASIZE];
	    FILE *pipe_stdout, *created_file;
	    long size, total_size;
	    
	    if (send(new_fd, "OK", 2, 0) == -1) return;
	    recv_buf[recv(new_fd, recv_buf, MAXDATASIZE, 0)] = '\0';
	    
	    if (debug) printf("Rendering frame: %d\n", atoi(recv_buf));
	    sprintf(arglist, "blender -b %s -f %d",
		    REMOTE_BLEND, atoi(recv_buf));
	    pipe_stdout = popen(arglist, "r");
	    fread(&stdout_buf, 1, MAXDATASIZE, pipe_stdout);
	    pclose(pipe_stdout);
	    
	    if (strstr(stdout_buf, "ERROR")) {
		printf("Rendering failed. corrput file ?\n");
		send(new_fd, "ERROR", 5, 0);
		return;
	    } else if (!strstr(stdout_buf, "Saved")) {
		printf("Executing blender failed\n");
		send(new_fd, "ERROR", 5, 0);
		return;
	    }

	    strcpy(stdout_buf, crop_output(stdout_buf));
	    if (send(new_fd, stdout_buf, strlen(stdout_buf), 0) == -1) return;

	    if (waitforok(new_fd) != 0) return;

	    /* send file back to the client */

	    if ((created_file = fopen(stdout_buf, "r")) == NULL) {
		fprintf(stderr, "error opening %s\n", stdout_buf);
		return;
	    }
	    fseek(created_file, 0, SEEK_END);
	    total_size = ftell(created_file);

	    sprintf(msg, "%ld", total_size);
	    if (send(new_fd, msg, strlen(msg), 0) == -1) return;

	    if (waitforok(new_fd) != 0) return;

	    fseek(created_file, 0, SEEK_SET);

	    while (!feof(created_file)) {
		
		size = ftell(created_file) + MAXDATASIZE > total_size ?
		       MAXDATASIZE - ((long) ftell(created_file) +
		       MAXDATASIZE - total_size) : MAXDATASIZE;
		
		fread(&msg, MAXDATASIZE, 1, created_file);
		if (send(new_fd, msg, size, 0) == -1) return;
	    }
	    
	    fclose(created_file);

	    if (remove(stdout_buf) == -1)
		fprintf(stderr, "Removing %s failed\n", stdout_buf);
	}
	
	else if (strcmp(buf, "DONE") == 0) {
	    if (debug) printf("Received request: DONE\n");
	    send(new_fd, "OK", 2, 0); return;
	}
	
	else {
	    if (debug) printf("Received unknown request: [%s]\n", buf);
	    send(new_fd, "ERROR", 5, 0); return;
	}
    }
}

char *crop_output(char *buf)
{
    char tmp1[MAXDATASIZE], tmp2[MAXDATASIZE];

    bzero(&tmp1, MAXDATASIZE);
    strncat(tmp1, buf, strlen(buf) - 2 - strlen("Blender quit"));
    strcpy(tmp1, strstr(tmp1, " ") + 1);

    bzero(&tmp2, MAXDATASIZE);
    strncpy(tmp2, tmp1,
	    strlen(tmp1) - strlen(strstr(tmp1, " Time:")));

    strcpy(buf, tmp2);
 
    return buf;
}

void sighan(int x)
{
    exit(x);
}
