/*
 * CCameraPathInterpolator.cpp
 * $Id: 
 *
 * Copyright (C) 2001 Thomas Woerner, Michael Meissner
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */


// Qt
///////
#include <qstring.h>

// Own
////////
#include "CCameraPathInterpolator.h"


// Function : comute
// Purpose  : compute CameraPath
// Comments :
void CCameraPathInterpolator::compute()
/*************************************************************/
{
   int k0, k1, k2, k3; // previous, this, next, after next key

   // copy camera type etc. from first key
   CCamera newCamera(keys[0].getCamera());
   CCameraKeyPathAttributes newAttribute;

   // set frames to 1
   newAttribute.setFrames(1);

   // calculate shape
   ////////////////////
   m_ShapeList.clear();
   for (k1=0; k1<keys.getNumObjects(); k1++) { // iterate keys sampling points
      CCamera Camera1, Camera2, Camera3, Camera4;

      // first key: if path is open insert the first else the last
      k0 = k1 - 1;
      if (k0 < 0) {
         if (m_nPathType == PATH_CLOSED)
            k0 += keys.getNumObjects();
         else
            k0 = k1;
      }

      k2 = k1 + 1;
      if (k2 > keys.getNumObjects() - 1) {
         if (m_nPathType == PATH_CLOSED)
            k2 -= keys.getNumObjects();
         else
            k2 = k1; // k0 ???
      }
	
	//
      k3 = k1 + 2;
      if (k3 > keys.getNumObjects() - 1) {
         if (m_nPathType == PATH_CLOSED)
            k3 -= keys.getNumObjects();
         else
            k3 = k2;
      }

      if (m_nShapeType == SHAPE_TCB || m_nShapeType == SHAPE_CATMULL_ROM) {

         double o_a, o_b, o_adj, i_a, i_b, i_adj;
         CCamera o_T, i_T; // outgoing, incoming tangent

         if (m_nShapeType == SHAPE_TCB) {
            // outgoing tangent factors
            o_a = (1.0 - keys[k1].getAttributes().getTension())
		    * (1.0 + keys[k1].getAttributes().getContinuity())
		    * (1.0 + keys[k1].getAttributes().getBias());
            o_b = (1.0 - keys[k1].getAttributes().getTension())
		    * (1.0 - keys[k1].getAttributes().getContinuity())
		    * (1.0 - keys[k1].getAttributes().getBias());
            o_adj = (double) keys[k0].getAttributes().getFrames()
		    / (keys[k0].getAttributes().getFrames()
		       + keys[k1].getAttributes().getFrames());
		
            // incoming tangent factors
            i_a = (1.0 - keys[k2].getAttributes().getTension())
		    * (1.0 - keys[k2].getAttributes().getContinuity())
		    * (1.0 + keys[k2].getAttributes().getBias());
            i_b = (1.0 - keys[k2].getAttributes().getTension())
		    * (1.0 + keys[k2].getAttributes().getContinuity())
		    * (1.0 - keys[k2].getAttributes().getBias());
            i_adj = (double) keys[k1].getAttributes().getFrames()
		       / (keys[k1].getAttributes().getFrames()
		       + keys[k2].getAttributes().getFrames());
         }

         if (m_nShapeType == SHAPE_CATMULL_ROM) {
            o_a = o_b = i_a = i_b = 1.0;
            o_adj = i_adj = 0.5;
         }

         // outgoing tangent
         o_T = mul(add(mul(sub(keys[k2].getCamera(),
				  keys[k1].getCamera()),
			      o_a),
			  mul(sub(keys[k1].getCamera(),
				  keys[k0].getCamera()),
			      o_b)),
		      o_adj);
         //incoming tangent
         i_T = mul(add(mul(sub(keys[k3].getCamera(),
				  keys[k2].getCamera()),
			      i_a),
			  mul(sub(keys[k2].getCamera(),
				  keys[k1].getCamera()),
			      i_b)),
		      i_adj);

         // the vector
         Camera1 = keys[k1].getCamera();
         Camera2 = o_T;
         Camera3 = sub(sub(mul(sub(keys[k2].getCamera(),
                               keys[k1].getCamera()),
			      3.0),
			  mul(o_T, 2.0)),
		      i_T);
         Camera4 = add(add(mul(sub(keys[k2].getCamera(),
                               keys[k1].getCamera()),
			      -2.0),
			  o_T),
		      i_T);
      }
	
      int nMaxJ = keys[k1].getAttributes().getFrames() - 1;

      // add only last key for an open path
      if (m_nPathType == PATH_OPEN  &&  k1 == keys.getNumObjects() - 1)
         nMaxJ = 0;

      // iterate frames for sampling point
      for (int j=0; j<=nMaxJ; j++) {
         double rdWeight;

         if (keys[k1].getAttributes().getFrames() > 1)
            rdWeight= (double) j / (keys[k1].getAttributes().getFrames());
         else
            rdWeight= 0.0;
	    
         // compute new camera, attributes and name
         if (m_nShapeType == SHAPE_LINE)
            newCamera = add(mul(keys[k1].getCamera(), 1.0 - rdWeight), mul(keys[k2].getCamera(), rdWeight));
	 
         if (m_nShapeType == SHAPE_TCB || m_nShapeType == SHAPE_CATMULL_ROM)
            newCamera = add(Camera1, mul(add(Camera2, mul(add(Camera3, mul(Camera4, rdWeight)), rdWeight)), rdWeight));

	 if(rdWeight)
	    newCamera.setTimeStep(static_cast<int>(keys[k1].getCamera().getTimeStep() 
						+ ((keys[k2].getCamera().getTimeStep() - keys[k1].getCamera().getTimeStep())
						   *rdWeight)
						+0.5));
	 else
	    newCamera.setTimeStep(keys[k1].getCamera().getTimeStep());

         QString newName = QString(keys[k0].getName()).append("_"+j);

         // append computed frame to shape
         m_ShapeList.insertAsLast(new CCameraKeyPathPoint(newCamera, newAttribute, newName));
      }
   }
}
