/* $Id: _chararraymodule.c,v 1.13.2.1 2004/11/16 23:02:47 jaytmiller Exp $ */

#include <Python.h>

#include <stdio.h>
#include <math.h>
#include <signal.h>

#include <libnumarray.h>

static PyObject *_Error;

/* returns length of string 's', or 'maxlen', whichever is smaller. */
#if !defined(strnlen)
size_t strnlen(const char *s, size_t maxlen)
{
	int i;
	for(i=0; i<maxlen; i++)
		if (s[i] == 0)
			return i;
	return maxlen;
}
#endif

#if !defined(min)
static int min(int x, int y)
{
	if (x < y)
		return x;
	else
		return y;
}
#endif

#if !defined(max)
static int max(int x, int y)
{
	if (x > y)
		return x;
	else
		return y;
}
#endif

/* mystrdup duplicates the given string 's', copying up to 'size' bytes.
   This still works when 's' is not terminated within 'size' bytes. 
   Also uses python memory allocation.
*/
static char *mystrdup(char *s, int size)
{
	char *u = PyMem_Malloc(size), *v = u;
	if (!u)
		return (char *) PyErr_Format(_Error,
					     "mycat: Error allocating memory.");
	while(size-- && (*u++ = *s++));
	return v;
}

/* rstripw strips whitespace from the right side of 's', which can be up to
'n' characters long.  rstripw interprets 0 as terminating 's'.
rstripw never strips a string down to length 0; the last whitespace is always
spared.
*/

#if defined(isspace)
#undef isspace
#define isspace(c)  ((c==' ')||(c=='\t')||(c=='\n')||(c=='\r')||(c=='\v')||(c=='\f'))
#endif

static void rstripw(char *s, int n)
{
	int i;
	for(i=strnlen(s,n)-1; i>=1; i--)  /* Never strip to length 0. */
	{
		int c = s[i];
		if (!c || isspace(c))
			s[i] = 0;
		else
			break;
	}
}

/* padw pads string 's' with blanks, for a total length of 'n' characters. */
static void 
padw(char *s, int n)
{
	int i;
	for(i=strnlen(s, n); i<n; i++)
		s[i] = ' ';
}

/* Strip executes rstripw on a python string. */
static PyObject *
Strip(PyObject *obj, PyObject *args)
{
	char *s, *t;
	int  slen;  

	if (!PyArg_ParseTuple(args, "s#:Strip", &s, &slen))
		return NULL;
	if ((t = mystrdup(s, slen)))
	{
		rstripw(t, slen);
		obj = Py_BuildValue("s#", t, strnlen(t, slen));
		PyMem_Free(t);
		return obj;
	}
	else
		return NULL;
}


/* Pad executes padw on a python string. */
static PyObject *
Pad(PyObject *obj, PyObject *args)
{
	char *s, *t;
	int  slen, rsize;

	if (!PyArg_ParseTuple(args, "s#i:Pad", &s, &slen, &rsize))
		return NULL;
	rsize = max(slen, rsize);
	if ((t = mystrdup(s, rsize)))
	{
		padw(t, rsize);
		obj = Py_BuildValue("s#", t, strnlen(t, rsize));
		PyMem_Free(t);
		return obj;
	}
	else
		return NULL;
}

static int 
Concat(PyObject *aux, long nnumarray, PyArrayObject *numarray[])
{
	int rused, bused;
	PyArrayObject *ai = numarray[0];
	PyArrayObject *bi = numarray[1];
	PyArrayObject *ri = numarray[2];
	char *a = (char *) ai->data;
	char *b = (char *) bi->data;
	char *r = (char *) ri->data;
	if (!PyInt_Check(aux)) {
		PyErr_SetString(_Error, "Bad call to Concat.");
		return -1;
	}
	if (PyInt_AsLong(aux)) {
		memcpy(r, a, ai->itemsize);
		memcpy(r+ai->itemsize, b, bi->itemsize);
	} else {
		strncpy(r, a, ai->itemsize);
		rused = strnlen(r, ai->itemsize);
		bused = min(strnlen(b, bi->itemsize), ri->itemsize-rused);
		strncpy(r+rused, b, bused);
		memset(r+rused+bused, 0, ri->itemsize-rused-bused);
	}
	return 0;
}

NSTRIDING_DESCR3(Concat);

static int 
StrCmp(PyObject *aux, long nnumarray, PyArrayObject *numarray[])
{
	PyArrayObject *ai = numarray[0];
	PyArrayObject *bi = numarray[1];
	PyArrayObject *ri = numarray[2];
	char   *a = (char *) ai->data;
	char   *b = (char *) bi->data;
	Int8   *r = (Int8 *) ri->data;
	int nc = min(ai->itemsize, bi->itemsize);
	if (!PyInt_Check(aux) || (nnumarray != 3)) {
		PyErr_SetString(_Error, "Bad call to StrCmp.");
		return -1;
	}
	if (PyInt_AsLong(aux))
		*r = memcmp(a, b, nc);
	else
		*r = strncmp(a, b, nc);
	return 0;
}

NSTRIDING_DESCR3(StrCmp);

/* StripAll applies rstripw to every element of a CharArray.  This can be used
to normalize charnumarray initialized from a binary source.
*/
static int 
StripAll( PyObject *aux, int nnumarray, PyArrayObject *numarray[])
{
	PyArrayObject *ai = numarray[0];
	char   *a  = (char *) ai->data;

	if (nnumarray != 1) {
		PyErr_Format(_Error, "StripAll: invalid parameters.");
		return -1;
	}
	if (!PyArray_ISWRITABLE(ai)) {
		PyErr_Format(_Error, "StripAll: result array not writeable.");
		return -1;
	}
	rstripw(a, ai->itemsize);
	return 0;
}

NSTRIDING_DESCR1(StripAll);

static int 
PadAll( PyObject *aux, int nnumarray, PyArrayObject *numarray[])
{
	PyArrayObject *ai = numarray[0];
	char   *a  = (char *) ai->data;

	if (nnumarray != 1) {
		PyErr_Format(_Error, "PadAll: invalid parameters.");
		return -1;
	}
	if (!PyArray_ISWRITABLE(ai)) {
		PyErr_Format(_Error, "PadAll: result array not writeable.");
		return -1;
	}
	padw(a, ai->itemsize);
	return 0;
}

NSTRIDING_DESCR1(PadAll);

/* MaxLen computes the maximum length of all stripped elements in a CharArray.
This information can be used to reduce the itemsize of the array.
*/
static PyObject *
MaxLen(PyObject *obj, PyObject *args)
{
	int        i, len=0;
	PyArrayObject *ao;
	char       *a;
  
	if (!PyArg_ParseTuple(args, "O:MaxLen", &ao))
		return NULL;

	if (!NA_NDArrayCheck((PyObject *) ao))
		return PyErr_Format(PyExc_TypeError, 
				    "MaxLen: object not an NDArray");

	if (!NA_updateDataPtr(ao))
		return NULL;

	a = (char *) ao->data;

	for(i=0; i<NA_elements(ao); i++, a+=ao->bytestride)
	{
		rstripw(a, ao->itemsize);
		len = max(len, strnlen(a, ao->itemsize));
	}

	return Py_BuildValue("i", len);
}

static int 
Eval(PyObject *args, int nnumarray, PyArrayObject *numarray[])
{
	PyArrayObject *ai = numarray[0],
		       *ni = numarray[1];
	char *a = (char *) ai->data;
	Float64 *n = (Float64 *) ni->data;
	char buffer[64], *ptr;
	int len;

	if (!PyArg_ParseTuple(args, ":Eval")) 
		return -1;
	
	len = strnlen(a, ai->itemsize);
	if  (len > ELEM(buffer)-1) {
		PyErr_Format(PyExc_ValueError, 
			     "string too long to convert.");
		return -1;
	}
	memcpy(buffer, a, len);
	buffer[len] = 0;
	*(double *) n = strtod(buffer, &ptr);
	if ((ptr == buffer) && (*n == 0)) {
		PyErr_Format(_Error, "Eval: error evaluating string."); 
		return -1;
	}
	return 0;
}
NSTRIDING_DESCR2(Eval);

static int 
Format(PyObject *format, int nnumarray, PyArrayObject *numarray[])
{
	PyArrayObject *ni = numarray[0];
	PyArrayObject *ai = numarray[1];
	char *a = (char *) ai->data;
	Float64 n = NA_get1_Float64(ni, 0);
	PyObject *args, *astr;
	char *s;
	if (!(args = Py_BuildValue("(d)", n))) {
		PyErr_Format(_Error, 
			     "Format: error building args tuple.");
		return -1;
	}
	if (!(astr = PyString_Format(format, args)))
		return -1;
	s = PyString_AsString(astr);
	if (strlen(s) > ai->itemsize) {
		PyErr_Warn(PyExc_RuntimeWarning,
			   "formatted value too large"
			   " for CharArray itemsize.");
	}
	strncpy(a, s, ai->itemsize);
	Py_DECREF(astr);
	Py_DECREF(args);
	return 0;
}

NSTRIDING_DESCR2(Format);

static PyMethodDef _chararrayMethods[] = {
	{"Strip",  Strip, METH_VARARGS},
	{"Pad",    Pad, METH_VARARGS},
	{"MaxLen",  MaxLen, METH_VARARGS},
	{NULL,     NULL}        /* Sentinel */
};

/* platform independent*/
#ifdef MS_WIN32
__declspec(dllexport)
#endif

void init_chararray(void) {
	PyObject *m, *d;
	m = Py_InitModule("_chararray", _chararrayMethods);
	d = PyModule_GetDict(m);
	_Error = PyErr_NewException("_chararray.error", NULL, NULL);
	PyDict_SetItemString(d, "error", _Error);

	import_libnumarray();

	NA_add_cfunc(d, "Concat", &Concat_descr);
	NA_add_cfunc(d, "StrCmp", &StrCmp_descr);
	NA_add_cfunc(d, "Format", &Format_descr);
	NA_add_cfunc(d, "Eval", &Eval_descr);
	NA_add_cfunc(d, "StripAll", &StripAll_descr);
	NA_add_cfunc(d, "PadAll", &PadAll_descr);
}
