/****************************************************************************

    percent_cmy - display the percentage of CMY colours in a colour .png
    file

    Copyright (C) 2000, 2001, 2002, 2003 Daniel Franklin

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 ****************************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <png.h>
#include <stdlib.h>
#define NUMBER 8

int main (int argc, char **argv)
{
	FILE *f;
	char header [NUMBER];
	png_structp png_ptr;
	png_infop info_ptr;
	png_uint_32 width, height;
	int depth, colour_type;
	int i, j, k;
	long fsize;
	struct stat buf;
	float cyan, magenta, yellow;
	png_bytep *array;
	int rowbytes;
	
	if (argc != 2) {
		fprintf (stderr, "Error: need one argument (the filename).\n");
		return -1;
	}
	
	if (stat (argv [1], &buf) != 0) {
		fprintf (stderr, "Error: could not stat file %s\n", argv [1]);
		return -2;
	}
	
	f = fopen (argv [1], "rb");
	
	if (!f) {
		fprintf (stderr, "Error: could not open file %s for binary reading.\n", argv [1]);
		return -3;
	}
	
	fread (header, 1, NUMBER, f);
	
	if (png_sig_cmp (header, 0, NUMBER)) {
		fprintf (stderr, "Error: %s does not appear to be a PNG image.\n", argv [1]);
		return -4;
	}

	png_ptr = png_create_read_struct (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);

	if (!png_ptr) {
		fprintf (stderr, "Error: allocation of png structure failed\n");
		return -5;
	}

	info_ptr = png_create_info_struct (png_ptr);
	
	if (!info_ptr) {
		png_destroy_read_struct (&png_ptr, (png_infopp) NULL, (png_infopp) NULL);
		fprintf (stderr, "Error: allocation of png info structure failed\n");
		return -6;
	}
	
	png_init_io (png_ptr, f);
	png_set_sig_bytes (png_ptr, NUMBER);
	png_read_info (png_ptr, info_ptr);
	png_get_IHDR (png_ptr, info_ptr, &width, &height, &depth, &colour_type, NULL, NULL, NULL);

	if (depth != 8 || colour_type != PNG_COLOR_TYPE_RGB) {
		fprintf (stderr, "Error: only 24-bit RGB supported.\n");
		return -7;
	}
	
	array = (png_bytep*) malloc (height * sizeof (png_bytep));
	
	if (!array) {
		fprintf (stderr, "Error: allocation of image data structure failed\n");
		png_destroy_read_struct (&png_ptr, (png_infopp) NULL, (png_infopp) NULL);
		return -8;
	}
	
	rowbytes = png_get_rowbytes (png_ptr, info_ptr);
	
	for (i = 0; i < height; i++)
		array [i] = (png_bytep) malloc (rowbytes);

	png_read_image (png_ptr, array);
	
	fsize = ftell (f) + 16;
	
	cyan = 0;
	magenta = 0;
	yellow = 0;
	
	for (i = 0; i < height; i++)
		for (j = 0; j < width; j++) {
			k = j * 3;
	
// Transform to 'CMYK' colourspace:
	
			cyan += 255 - array [i][k];
			magenta += 255 - array [i][k + 1];
			yellow  += 255 - array [i][k + 2];
		}
		
	cyan /= 255.0;
	magenta /= 255.0;
	yellow /= 255.0;

	for (i = 0; i < height; i++)
		free (array [i]);
		
	free (array);
	
	png_destroy_read_struct (&png_ptr, (png_infopp) NULL, (png_infopp) NULL);

	printf ("%.3f:%.3f:%.3f:%i\n",
			100 * cyan / (float) (height * width),
			100 * magenta / (float) (height * width),
			100 * yellow / (float) (height * width),
			(int)(buf.st_size / (int) fsize));
	
	return 0;
}
