/*
 * video-v4l.cc --
 *
 *      Driver file for "Video 4 Linux" Interface.
 *
 *      Implementation of the Video4Linux-Card Capture Object,
 *      aka VideoCapture/V4lx (in tcl) or VideoCaptureV4l (in C++)
 */

/* =========================================================================

     Copyright (c) 1997 Regents of Koji OKAMURA, oka@kobe-u.ac.jp
     All rights reserved.

     largely rewritten for new bttv/video4linux interface
     by Gerd Knorr <kraxel@cs.tu-berlin.de>

     Adapted for the Mash environment by Hank Magnuski <hankm@netvideo.com>

   ========================================================================= */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/mman.h>

#include <linux/types.h>

extern "C" {
#include <linux/videodev.h>
}

#include "video-device.h"
#include "Tcl.h"
#include "module.h"
#include "yuv_convert.h"

/* here you can tune the device names */
static const char *devlist[] = {
    "/dev/video0", "/dev/video1", "/dev/video2", "/dev/video3", "/dev/video4",
    "/dev/video5", "/dev/video6", "/dev/video7", "/dev/video8", "/dev/video9",
    NULL
};

/* display memory */
//unsigned char* v4l_memory;
//unsigned int v4l_offset;

// #define DEBUGGING
#ifdef DEBUGGING
#define DEBUG(x) (x)
#else
#define DEBUG(x)
#endif

#define PAL	    0
#define NTSC	    1
#define NTSC_WIDTH  640
#define NTSC_HEIGHT 480
#define PAL_WIDTH   768
#define PAL_HEIGHT  576
#define CIF_WIDTH   352
#define CIF_HEIGHT  288

/* pass 0/1 by reference */
static const int  one = 1, zero = 0;

#define CF_422 0
#define CF_411 1
#define CF_CIF 2

// once more types are supported, should just set a function pointer & call it
// typedef void (VideoCaptureV4l::*functionPointer) (char*, char*);

class VideoCaptureV4l : public VideoCapture {
public:
	VideoCaptureV4l(const char * cformat, const char *dev);
	virtual ~VideoCaptureV4l();

	virtual int  command(int argc, const char*const* argv);
	virtual void start();
	virtual void stop();
	virtual void grab();
protected:
	int capture();
	void format();
	void setsize();

	struct video_capability  capability;
	struct video_channel     *channels;
	struct video_picture     pict;
	struct video_window      win;

	/* mmap */
	int                      have_mmap; // tracks if mmap supported (mmap is required)
//	int                      grab_count;
//	int                      sync_count;
	int frames_queued; // tracks how many frames have been given to the driver
	int next_frame_submit; // next frame to be given to driver to place video image in
	int next_frame_receive; // next frame to be returned by the driver
//	struct video_mmap        gb_even;
//	struct video_mmap        gb_odd;
	struct video_mbuf        gb_buffers;
	char *mem; // pointer to beginning of mmap()ed region

	int fd_; // file descriptor associated with device
	int format_;
	int v4lformat_; // video format that is being provided by hardware
	// use pointer once more types supported
	// functionPointer convertVideo;
	int cformat_; // video format that mach expects to receive
	int port_; // hardware port that is being used to receive video
	int norm_; // video norm (e.g., ntsc, pal) that is being used to interpret video

	// unsigned char *tm_;
	int width_; // width that mash expects to receive
	int height_; // height that mash expects to receive
	int v4l_width_; // current mmap()ed width
	int v4l_height_; // current mmap()ed height
    int v4l_min_width_; // minimum width supported by device
    int v4l_min_height_; // minimum height supported by device
	int v4l_max_width_; // maximum width supported by device
	int v4l_max_height_; // maximum height supported by device
    int v4l_scales_; // non-zero if v4l driver can scale video to arbitrary sizes
	int decimate_; // specifies what size of video should be used (1=d1, 2=cif, 4=thumbnail)
};

/* ----------------------------------------------------------------- */

class V4lDevice : public VideoDevice {
public:
	V4lDevice(const char* clsname, const char* nickname,
			const char* devname, const char* attr, int free);
	TclObject* create(int argc, const char*const* argv) {
		if (argc != 5) {
			fprintf(stderr,"Wrong number of arguments passed to V4lDevice::VideoDevice\n");
			abort();/*FIXME*/
		}
		if (strcmp(argv[4], "422") == 0)
				return (new VideoCaptureV4l("422",name_));
		else if (strcmp(argv[4], "411") == 0)
				return (new VideoCaptureV4l("411",name_));
		else if (strcmp(argv[4], "cif") == 0)
				return (new VideoCaptureV4l("cif",name_));
		return (0);
	}
protected:
	const char* name_;
};

V4lDevice::V4lDevice(const char* clsname, const char* nickname,
                     const char *devname, const char* attr, int free)
	: VideoDevice(clsname, nickname), name_(devname)
{
	if (free)
			attributes_ = attr;
	else
			attributes_ = "disabled";

	DEBUG(fprintf(stderr,"V4l: ==> %s, %s, %s\n",clsname,nickname,name_));
	DEBUG(fprintf(stderr,"V4l: ==> %s\n",attributes_));
}

/* ----------------------------------------------------------------- */

class V4lScanner {
public:
	V4lScanner(const char **dev);
};

static V4lScanner find_video4linux_devices(devlist);

V4lScanner::V4lScanner(const char **dev)
{
#ifdef DEBUGGING
	static const char *palette_name[] = {
			"<none>", "Grey", "Hi240", "RGB565 (16bit)", "RGB24", "RGB32",
			"RGB555 (15bit)", "YUV422", "YUYV", "UYVY", "YUV420", "YUV411",
			"Raw (BT848)", "YUV422 planar", "YUV411 planar", "YUV420 planar",
			"YUV410 planar"};
	static int palette_size=17; // number of items in array
#endif

	struct video_capability  capability;
	struct video_channel     channel;
	struct video_picture     pict;
	int  j,i,fd;
	const char *v4lclassnames[] = {"VideoCapture/V4l0",
	                               "VideoCapture/V4l1",
	                               "VideoCapture/V4l2",
	                               "VideoCapture/V4l3",
	                               "VideoCapture/V4l4",
	                               "VideoCapture/V4l5",
	                               "VideoCapture/V4l6",
	                               "VideoCapture/V4l7",
	                               "VideoCapture/V4l8",
	                               "VideoCapture/V4l9"};

	char *nick, *attr;

	// try to open each of the devices, and determine capabilities of device
	for (i = 0; dev[i] != NULL; i++) {
		DEBUG(fprintf(stderr,"V4l: trying %s... ",dev[i]));
		if (-1 == (fd = open(dev[i],O_RDONLY))) {
			DEBUG(perror("open"));
			continue;
		}

		// VIDIOCGCAP: ask for capabilities
		if (-1 == ioctl(fd,VIDIOCGCAP,&capability)) {
			perror("ioctl VIDIOCGCAP");
			close(fd);
			continue;
		}

		// verify that the device can capture
		if (!(capability.type & VID_TYPE_CAPTURE)) {
			perror("device can't capture\n");
			close(fd);
			continue;
		}

		DEBUG(fprintf(stderr,"ok\n"));
		DEBUG(fprintf(stderr,"V4l:\tcapabilities (VIDIOCGCAP): name=%s; %s; channels=%d, audios=%d ;size=%dx%d to %dx%d%s\n",
				capability.name,
				capability.type & VID_TYPE_MONOCHROME ? "mono" : "color",
				capability.channels, capability.audios,
				capability.minwidth,capability.minheight,
				capability.maxwidth,capability.maxheight,
				capability.type & VID_TYPE_SCALES ? " (supports image scaling)" : ""
				));

		attr = new char[512];
		strcpy(attr,"format { 411 422 cif } ");

		// check if our capture card supports "large" images
		if (capability.maxwidth  >= 600 &&
				capability.maxheight >= 400) {
			strcat(attr,"size { cif large } "); // "small" removed since not truely supported 
		} else {
			strcat(attr,"size { cif } "); // "small" removed since not truely supported 
		}

		// VIDIOCGCHAN: get channel information
		DEBUG(fprintf(stderr,"V4l:   ports:"));
		strcat(attr,"port { ");
		for (j = 0; j < capability.channels; j++) {
			channel.channel = j;
			if (-1 == ioctl(fd,VIDIOCGCHAN,&channel)) {
				perror("ioctl VIDIOCGCHAN");
			} else {
				DEBUG(fprintf(stderr,"chan %d: %s ; tuners=%d, flags=%s, type=%s, (norm:%d)", j, channel.name, channel.tuners, (channel.flags==3?"Tuner/Audio":(channel.flags==2?"Audio":(channel.flags==1?"Tuner":(channel.flags==0?"no flags":"Invalid Flags")))), (channel.type==1?"TV":(channel.type==2?"Camera":"Invalid Type")), channel.norm));
				strcat(attr,channel.name);
				strcat(attr," ");
			}
		}
		DEBUG(fprintf(stderr,"\n"));
		strcat(attr,"} ");

		strcat(attr,"norm { auto ntsc pal secam } "); // this is bad: some may not be supported

		// VIDIOCGPICT: ask for image properties
		if (-1 == ioctl(fd,VIDIOCGPICT,&pict)) {
			perror("ioctl VIDIOCGPICT");
		}
		DEBUG(fprintf(stderr,"V4l:\timage propierties (VIDIOCGPICT): brightness=%i; hue=%i; colour=%i; contrast=%i; whiteness=%i; depth=%d, palette=%d: %s\n",
				pict.brightness,
				pict.hue,
				pict.colour,
				pict.contrast,
				pict.whiteness,
				pict.depth,
				pict.palette,
				(pict.palette<palette_size)?(palette_name[pict.palette]):"Invalid Palette"));

		nick = new char[strlen(capability.name)+6];
		sprintf(nick,"V4l: %s",capability.name);
		new V4lDevice(v4lclassnames[i],nick,dev[i],attr,1);

		close(fd);
	}
}

/* ----------------------------------------------------------------- */

VideoCaptureV4l::VideoCaptureV4l(const char *cformat, const char *dev)
{
	int i=0;
	channels=NULL;
	DEBUG(fprintf(stderr,"V4l: constructor (fmt=%s, dev=%s)\n",
			cformat, dev));
	have_mmap = 0;
	int num_palettes=5;
	frames_queued=0;
	next_frame_submit=0;
	next_frame_receive=0;
	int palettes[]={
			VIDEO_PALETTE_YUV422, // BT878 (OS driver broken, so listed first)
			VIDEO_PALETTE_YUV422P, // BT878 (untested)
			VIDEO_PALETTE_YUV420P, // (untested)
			VIDEO_PALETTE_UYVY, // LML33
			VIDEO_PALETTE_YUYV  // others
	}; // listed in order of preference
	v4lformat_=0;

	// initialized norm_ to an invalid value (valid values are 0-3)
	norm_ = -1; 

	// search for a supported palette
	for(int i=0; i<num_palettes && v4lformat_==0; ++i) {
		fd_ = open(dev, O_RDWR);
		if (fd_ < 0) {
			perror("open");
			exit(1);
		}
		struct video_capability  capability;
		// VIDIOCGCAP: ask for capabilities
		if (-1 == ioctl(fd_,VIDIOCGCAP,&capability)) {
		  perror("ioctl VIDIOCGCAP");
		  close(fd_);
		  continue;
		}

		// store the minimum/maximum video sizes supported by the device
		// v4l (the api and its implementations) doesn't provide a good 
		// way to determine which sizes are supported by the device.  The
		// api says that a driver should use the next smallest size that
		// the device supports, yet no driver I've found does this, and
		// even if a driver did do this there a no way for an application 
		// to determine what video size is being mmap()ed by the driver

		v4l_min_width_=capability.minwidth;
		v4l_min_height_=capability.minheight;
		v4l_max_width_=capability.maxwidth;
		v4l_max_height_=capability.maxheight;
		v4l_scales_=(capability.type & VID_TYPE_SCALES)?1:0;

		// mmap() before calling ioctl(VIDIOCMCAPTURE) because 
		// some devices assume that mmap() is called first
		
		gb_buffers.size = 2*0x151000;
		gb_buffers.frames=2;
		gb_buffers.offsets[0] = 0;
		gb_buffers.offsets[1] = 0x151000;
		if (-1 == ioctl(fd_,VIDIOCGMBUF,&gb_buffers)) {
			close(fd_);
			continue;
		}

		mem = (char *)mmap(0,gb_buffers.size,PROT_READ|PROT_WRITE,MAP_SHARED,fd_,0);
		if ((char*)-1 == mem) {
			close(fd_);
			continue;
		} else {
			have_mmap = 1;
		}

		struct video_mmap vid_mmap;
		vid_mmap.format = palettes[i];
		vid_mmap.frame = 0;
		vid_mmap.width = CIF_WIDTH;
		vid_mmap.height = CIF_HEIGHT;

		// try to mmap the size that mash wants
		if (-1 != ioctl(fd_, VIDIOCMCAPTURE, &vid_mmap)) {
		  DEBUG(fprintf(stderr, "first mmap() had success\n"));
			//  have_422P = 1;
			v4lformat_=palettes[i];
			v4l_width_=CIF_WIDTH;
			v4l_height_=CIF_HEIGHT;
			DEBUG(fprintf(stderr, "selected format index #%d\n", i));
		}
		// if mmap failed, then try a different size if the device doesn't scale
		else if( !(capability.type & VID_TYPE_SCALES) ) {
		  DEBUG(fprintf(stderr, "first mmap() failed, but doesn't scale\n"));
		  // try a different video size if the minimum is somewhere near CIF size
		  if(capability.minwidth>=300 && capability.minwidth <=400 &&
		     capability.minheight>=200 && capability.minheight<=300) {
		   DEBUG(fprintf(stderr, "first mmap() failed, but doesn't scale and is in valid size range %dx%d\n", capability.minwidth, capability.minheight));  
		    vid_mmap.format = palettes[i];
		    vid_mmap.frame = 0;    
		    vid_mmap.width=capability.minwidth;
		    vid_mmap.height=capability.minheight;
		    if (-1 != ioctl(fd_, VIDIOCMCAPTURE, &vid_mmap)) {
		      //  have_422P = 1;
		      v4lformat_=palettes[i];
		      v4l_width_=capability.minwidth;
		      v4l_height_=capability.minheight;
		      DEBUG(fprintf(stderr, "selected format index #%d of size %d x %d\n", i,width_,height_));
		    }
		    else {
		      DEBUG(fprintf(stderr, "second mmap() failed\n"));  
		    }
		  }
		  else {
		    DEBUG(fprintf(stderr, "minsize not near CIF, so not attempting different video size\n"));
		  }
		}
		else {
		  DEBUG(fprintf(stderr, "mmap failed, but scales, so not attempting different video size\n"));
		}
		/* Release device */
		munmap(mem,gb_buffers.size);
		close(fd_);
	}
	if(v4lformat_==0) {
		fprintf(stderr, "Could not find supported video format\n");
		exit(1);
	}

	// open the device
	fd_ = open(dev, O_RDWR);
	if (fd_ < 0) {
		perror("open");
		exit(1);
	}

	// VIDIOCGCAP: ask for capabilities
	if (-1 == ioctl(fd_,VIDIOCGCAP,&capability)) {
		perror("ioctl VIDIOCGCAP");
		exit(1);
	}
	channels = (struct video_channel*) calloc(capability.channels,
			sizeof(struct video_channel));
	for (i = 0; i < capability.channels; i++) {
		channels[i].channel = i;
		if (-1 == ioctl(fd_,VIDIOCGCHAN,&channels[i])) {
			perror("ioctl VIDIOCGCHAN");
		}
	}
	if (-1 == ioctl(fd_,VIDIOCGPICT,&pict)) {
		perror("ioctl VIDIOCGPICT");
	}

	// map grab buffer
	/* This is for backward compatibility with earlier drivers.
	   VIDIOCMGBUF is a "get" call and will overwrite gb_buffers with
	   new information if it is implemented in this driver */

	gb_buffers.size = 2*0x151000;
	gb_buffers.frames=2;
	gb_buffers.offsets[0] = 0;
	gb_buffers.offsets[1] = 0x151000;

	if (-1 == ioctl(fd_,VIDIOCGMBUF,&gb_buffers)) {
		perror("ioctl VIDIOCGMBUF");
	}

	mem = (char *)mmap(0,gb_buffers.size,PROT_READ|PROT_WRITE,MAP_SHARED,fd_,0);
	if ((char*)-1 == mem) {
		perror("mmap");
		fprintf(stderr,"V4l: device has no mmap support\n");
	} else {
		DEBUG(fprintf(stderr,"V4l: mmap()'ed buffer size = 0x%x\n", gb_buffers.size));
		have_mmap = 1;
		//v4l_memory = (unsigned char*)mem;
		//v4l_offset = gb_buffers.offsets[1];
	}

	// fill in defaults
	if(!strcmp(cformat, "411"))
			cformat_ = CF_411;
	if(!strcmp(cformat, "422"))
			cformat_ = CF_422;
	if(!strcmp(cformat, "cif"))
			cformat_ = CF_CIF;

	port_      = 0;
	decimate_  = 2;

	// only set norm to current value if it hasn't been set before.
	// if users have called [grabber_ norm something] then we should
	// stick to that value.
	if (norm_ == -1) 
	    norm_ = channels[port_].norm;

}

VideoCaptureV4l::~VideoCaptureV4l()
{
	DEBUG(fprintf(stderr,"V4l: destructor\n"));

	if (have_mmap)
		munmap(mem,gb_buffers.size);
	if(channels) {
		free(channels);
		channels=NULL;
	}
	close(fd_);
}

int VideoCaptureV4l::command(int argc, const char*const* argv)
{
	int i;

	if (argc == 3) {

		if (strcmp(argv[1], "decimate") == 0) {
		  int newDecimate=atoi(argv[2]);
		  if(newDecimate>0 && newDecimate<10) {
		    if(newDecimate!=decimate_) {
		      if(newDecimate==1 && decimate_==2) { // double size
			if(v4l_scales_) {
			  if(v4l_max_width_>=v4l_width_*2 &&
			     v4l_max_height_>=v4l_height_*2) {
			    v4l_width_*=2;
			    v4l_height_*=2;
			  }
			}
			else {
			  if(v4l_max_width_>= 600 && v4l_max_width_<=800 &&
			     v4l_max_height_>= 400 &&v4l_max_height_<=600) {
			    v4l_width_=v4l_max_width_;
			    v4l_height_=v4l_max_height_;
			  } 
			}
			decimate_ = newDecimate;
			if (running_)
			  format();
		      }
		      else if(newDecimate==2 && decimate_==1) { // halve size
			if(v4l_scales_) {
			  if(v4l_min_width_<=v4l_width_/2 &&
			     v4l_min_height_<=v4l_height_/2) {
			    v4l_width_/=2;
			    v4l_height_/=2;
			  }
			}
			else {
			  if(v4l_min_width_>= 300 && v4l_min_width_<=400 &&
			     v4l_min_height_>= 200 &&v4l_min_height_<=300) {
			    v4l_width_=v4l_min_width_;
			    v4l_height_=v4l_min_height_;
			  } 
			}
			decimate_ = newDecimate;
			if (running_)
			  format();
		      }
		      else {
			fprintf(stderr, "mash bug: small wasn't reported as being supported!\n");
		      }
		      //decimate_ = newDecimate;
		      //	      if (running_)
		      //	format();
		    }
		  }
		}

		if (strcmp(argv[1], "port") == 0) {
			for (i = 0; i < capability.channels; i++)
				if(!strcmp(argv[2], channels[i].name))
						port_ = i;
			if (running_)
					format();
			return (TCL_OK);
		}

		if (strcmp(argv[1], "norm") == 0) {
			DEBUG(fprintf(stderr,"norm %s\n",argv[2]));
			if (!strcmp(argv[2], "pal")) {
				norm_ = VIDEO_MODE_PAL;
			} else if (!strcmp(argv[2], "ntsc")) {
				norm_ = VIDEO_MODE_NTSC;
			} else if (!strcmp(argv[2], "secam")) {
				norm_ = VIDEO_MODE_SECAM;
			} else if (!strcmp(argv[2], "auto")) {
				norm_ = VIDEO_MODE_AUTO;
			} else {
				norm_ = VIDEO_MODE_NTSC;
			}
			if (running_) {
				format();
			}
			return (TCL_OK);
		}

		if (strcmp(argv[1], "fps") == 0) {
			DEBUG(fprintf(stderr,"V4l: fps %s\n",argv[2]));
		}

		if (strcmp(argv[1], "brightness") == 0) {
			u_char val = atoi(argv[2]);
			DEBUG(fprintf(stderr,"V4l: brightness %i\n", val));
			pict.brightness=val*256;
			if (-1 == ioctl(fd_,VIDIOCSPICT,&pict)) {
				perror("ioctl VIDIOCSPICT");
			}
			return (TCL_OK);
		}

		if (strcmp(argv[1], "hue") == 0) {
			u_char val = atoi(argv[2]);
			DEBUG(fprintf(stderr,"V4l: hue %i\n", val));
			pict.hue=val*256;
			if (-1 == ioctl(fd_,VIDIOCSPICT,&pict)) {
				perror("ioctl VIDIOCSPICT");
			}
			return (TCL_OK);
		}

		if (strcmp(argv[1], "saturation") == 0) {
			u_char val = atoi(argv[2]);
			DEBUG(fprintf(stderr,"V4l: saturation %i\n", val));
			pict.colour=val*256;
			if (-1 == ioctl(fd_,VIDIOCSPICT,&pict)) {
				perror("ioctl VIDIOCSPICT");
			}
			return (TCL_OK);
		}

		if (strcmp(argv[1], "contrast") == 0) {
			u_char val = atoi(argv[2]);
			DEBUG(fprintf(stderr,"V4l: contrast %i\n", val));
			pict.contrast=val*256;
			if (-1 == ioctl(fd_,VIDIOCSPICT,&pict)) {
				perror("ioctl VIDIOCSPICT");
			}
			return (TCL_OK);
		}

		if (strcmp(argv[1], "controls") == 0) {
			if (strcmp(argv[2], "reset") == 0) {
				DEBUG(fprintf(stderr,"V4l: controls reset\n"));
				pict.brightness=32768;
				pict.hue=32768;
				pict.colour=32768;
				pict.contrast=32768;
				if (-1 == ioctl(fd_,VIDIOCSPICT,&pict)) {
					perror("ioctl VIDIOCSPICT");
				}
			}
			return (TCL_OK);
		}
	}

	return (VideoCapture::command(argc, argv));
}

void VideoCaptureV4l::start()
{
	DEBUG(fprintf(stderr,"V4l: start, calling CMCAPTURE twice\n"));

	format();

	if (have_mmap) {
//		grab_count = 0;
//		sync_count = 0;

		for(int i=0; i<gb_buffers.frames; ++i) {
			struct video_mmap  gb_frame;
			gb_frame.frame  = i;
			gb_frame.format = v4lformat_;
			gb_frame.width  = v4l_width_;
			gb_frame.height = v4l_height_;
			if (-1 == ioctl(fd_, VIDIOCMCAPTURE, &gb_frame))
				perror("ioctl VIDIOCMCAPTURE");
		}
		frames_queued=gb_buffers.frames; // all frames have been queued
		next_frame_receive=0; // first frame that we expect to receive is #0
		next_frame_submit=0;  // next frame we will submit to driver is #0
	}
	VideoCapture::start();
}

void VideoCaptureV4l::stop()
{
	DEBUG(fprintf(stderr,"V4l: stop (frames_queued=%d), calling VIDIOSYNC\n", frames_queued));

	if (have_mmap) {
		while (frames_queued>0) {
			int u=next_frame_receive;
			if (-1 == ioctl(fd_, VIDIOCSYNC, &u)) {
				perror("ioctl VIDIOCSYNC");
				break;
			} else {
				--frames_queued;
				++next_frame_receive;
				if(next_frame_receive>=gb_buffers.frames)
					next_frame_receive=0;
			}
		}
	}

	VideoCapture::stop();
}

int VideoCaptureV4l::capture()
{
  char  *fr=NULL;
  DEBUG(fprintf(stderr,"V4l::capture() called, calling SYNC then CMCAPTURE\n"));
  DEBUG(fprintf(stderr,"%d", next_frame_receive));

	if (have_mmap) {
		fr = mem + (gb_buffers.offsets[next_frame_receive]);
		int u=next_frame_receive;
		if (-1 == ioctl(fd_, VIDIOCSYNC, &u))
				perror("ioctl VIDIOCSYNC");
		else {
			--frames_queued;
			++next_frame_receive;
			if(next_frame_receive>=gb_buffers.frames)
				next_frame_receive=0;
		}
	} else {
		/* FIXME: read() */
	}
	switch (cformat_) {
	case CF_422:
		if (v4lformat_ == VIDEO_PALETTE_YUV422P) {
			planarYUYV422_to_planarYUYV422((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_YUV420P) {
			planarYUYV420_to_planarYUYV422((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_UYVY) {
			packedUYVY422_to_planarYUYV422((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_YUYV ||
				v4lformat_ == VIDEO_PALETTE_YUV422) {
			packedYUYV422_to_planarYUYV422((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else {
			printf("unsupported conversion in V4L");
			exit(1);
		}
		break;

	case CF_411:
		if (v4lformat_ == VIDEO_PALETTE_YUV422P) {
			planarYUYV422_to_planarYUYV411((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_YUV420P) {
			planarYUYV420_to_planarYUYV411((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_UYVY) {
			packedUYVY422_to_planarYUYV411((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_YUYV ||
				v4lformat_ == VIDEO_PALETTE_YUV422) {
			packedYUYV422_to_planarYUYV411((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else {
			printf("unsupported conversion in V4L");
			exit(1);
    }
		break;

	case CF_CIF:
		if (v4lformat_ == VIDEO_PALETTE_YUV422P) {
			planarYUYV422_to_planarYUYV420((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_YUV420P) {
			planarYUYV420_to_planarYUYV420((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_UYVY) {
			packedUYVY422_to_planarYUYV420((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else if (v4lformat_ == VIDEO_PALETTE_YUYV ||
				v4lformat_ == VIDEO_PALETTE_YUV422) {
			packedYUYV422_to_planarYUYV420((char*)frame_,width_,height_,fr,v4l_width_,v4l_height_);
		} else {
			printf("unsupported conversion in V4L");
			exit(1);
    }
		break;
	}

	if (have_mmap) {
		struct video_mmap  gb_frame;
		gb_frame.frame  = next_frame_submit;
		gb_frame.format = v4lformat_;
		gb_frame.width  = v4l_width_;
		gb_frame.height = v4l_height_;
		if (-1 == ioctl(fd_, VIDIOCMCAPTURE, &gb_frame))
			perror("ioctl VIDIOCMCAPTURE");
		else {
			++frames_queued;
			++next_frame_submit;
			if(next_frame_submit>=gb_buffers.frames)
				next_frame_submit=0;
		}
	}

	return (1);
}

void VideoCaptureV4l::grab()
{
	if (capture() == 0)
			return;

	suppress(frame_);
	saveblks(frame_);
	YuvFrame f(media_ts(), frame_, crvec_, outw_, outh_);
	target_->recv(&f);
}


void VideoCaptureV4l::format()
{
	//DEBUG(fprintf(stderr,"V4l: format"));

	width_  = CIF_WIDTH  *2  / decimate_;
	height_ = CIF_HEIGHT *2  / decimate_;

	DEBUG(fprintf(stderr,"V4l: format width_=%d, height_=%d, decimate=%d",
			width_,height_,decimate_));
	if (have_mmap) {
#if 0
		gb_even.frame  = 0;
		gb_even.format = v4lformat_; // VIDEO_PALETTE_YUV422;
		gb_even.width  = v4l_width_;
		gb_even.height = v4l_height_;
		gb_odd.frame   = 1;
		gb_odd.format  = v4lformat_; // VIDEO_PALETTE_YUV422;
		gb_odd.width   = v4l_width_;
		gb_odd.height  = v4l_height_;
#endif
		//v4l_width      = width_;
		//v4l_height     = height_;
	} else {
		memset(&win,0,sizeof(win));
		win.width  = width_;
		win.height = height_;
		if (-1 == ioctl(fd_,VIDIOCSWIN,&win))
				perror("ioctl VIDIOCSWIN");
		if (-1 == ioctl(fd_,VIDIOCGWIN,&win))
				perror("ioctl VIDIOCGWIN");
		width_  = win.width;
		height_ = win.height;

		pict.palette = v4lformat_;
		if (-1 == ioctl(fd_,VIDIOCSPICT,&pict)) {
			perror("ioctl VIDIOCSPICT");
		}
	}
	if (-1 == ioctl(fd_,VIDIOCGPICT,&pict)) {
		perror("ioctl VIDIOCGPICT");
	}

	switch (cformat_) {
	case CF_CIF:
		set_size_cif(width_, height_);
		DEBUG(fprintf(stderr," cif"));
		break;
	case CF_411:
		set_size_411(width_, height_);
		DEBUG(fprintf(stderr," 411"));
		break;
	case CF_422:
		set_size_422(width_, height_);
		DEBUG(fprintf(stderr," 422"));
		break;
	}

	channels[port_].norm = norm_;

	DEBUG(fprintf(stderr," size=%dx%d, port=%d, norm=%d",width_,height_,port_, norm_));

	if (-1 == ioctl(fd_, VIDIOCSCHAN, &channels[port_]))
	  perror("ioctl VIDIOCSCHAN");
	DEBUG(fprintf(stderr," port=%d\n",port_));

	allocref();
}
