# ui-tipwidget.tcl --
#
#       Popup tips for Tk

# ---------------------------------------------------------------------------
# This software is in the public domain, furnished "as is", without technical
# support, and with no warranty, express or implied, as to its usefulness for
# any purpose.
#
# tips.tcl
# Popup tips for Tk
# ---------------------------------------------------------------------------

# -- module -----------------------------------------------------------------
#
# This package provides popup tips for widgets in Tcl/Tk.
#
# Simply prefix your widget creation statement with TipWidget and use a new
# option "-tip msg" while creating the widget
#   e.g. TipWidget button .b -text "Foo" -tip "Bar"
# ---------------------------------------------------------------------------


Class TipManager

TipManager proc.invoke { } {
	option add *TipManager*background  LemonChiffon
	option add *TipManager*foreground  Black
	option add *TipManager*relief      raised
	option add *TipManager*borderWidth 1
	option add *TipManager*justify     center
	option add *TipManager*font        {Helvetica 12 normal}
	option add *TipManager*windowdelay 500
	option add *TipManager*yoffset     5
	option add *TipManager*xoffset     20


	$self instvar enabled_ currently_tipping_
	set enabled_ 1
	set currently_tipping_ ""

	# Bindings for the TipManagerBindings tag.
	bind TipManagerBindings <Enter> {
		TipManager enter %W
	}
	bind TipManagerBindings <Leave> {
		TipManager cancel
	}
	bind TipManagerBindings <Button-1> {
		TipManager cancel
	}
	bind TipManagerBindings <space> {
		TipManager cancel
	}
}


# -- proc -------------------------------------------------------------------
# enable
#
# With boolean true, enable all popup tips.  False, disable all of them.
# ---------------------------------------------------------------------------

TipManager proc.public enable { {bool {}} } {
	$self instvar currently_tipping_ enabled_

	if { $bool=={} } { return $enabled_ }
	if {! $bool && $currently_tipping_ != ""} {
		$self cancel
	}
	set enabled_ $bool
}


TipManager proc.public is_enabled { } {
	return [$self set enabled_]
}


TipManager proc.public toggle { } {
	$self instvar enabled_
	$self enable [expr !$enabled_]
}


TipManager proc.public tip { widget tip } {
	if { $tip == {} } {
		$self remove_tip $widget
	} else {
		$self add_tip $widget $tip
	}
}




# -- proc ---------------------------------------------------------------
# add_tip
#
# Add a tip message to the given widget.  When the pointer enters the
# widget, it'll wait a bit, then if it's still there, it'll map the
# tip window.  As soon as the pointer leaves, then it unmaps it.
# ---------------------------------------------------------------------------

TipManager proc.private add_tip { widget tip } {
	$self instvar tips_

	# Save the message text for later use.
	set tips_($widget) $tip

	# Add the tip bindings to the window ... now, it might already have
	# them if we got called to change the text, so check first.
	set tags [bindtags $widget]
	if {[lsearch -exact $tags TipManagerBindings] == -1} {
		# They're not in there, add 'em.
		bindtags $widget [concat TipManagerBindings [bindtags $widget]]
	}
}


# -- instproc ---------------------------------------------------------------
# remove_tip
#
# Remove the popup tip for this window.
# ---------------------------------------------------------------------------
TipManager proc.private remove_tip { widget } {
	$self instvar tips_
	TipManager instvar currently_tipping_

	# Do we have saved text for this window?  If so, then we have to get
	# rid of it and the window's class binding, too.
	if [info exists tips_($widget)] {
		# Okay, nuke the text.
		unset tips_($widget)

		# Now, get the tags and see if the TipManagerBindings is
		# among them.
		if [winfo exists $widget] {
			set tags [bindtags $widget]
			if {[set index [lsearch -exact $tags \
					TipManagerBindings]] >=0} {
				# Yep, it's there.  Set the new bindings
				# without TipManagerBindings present.
			bindtags $widget [lreplace $tags $index $index]
			}
		}

		# Also, if the tip window is visible for this widget, nuke it.
		if {$currently_tipping_ == $widget} {
			$self cancel
		}
	}
}


# -- proc ---------------------------------------------------------------
# enter
#
# This is an internal method that is invoked when the <Enter> event occurs
# ---------------------------------------------------------------------------
TipManager proc.private enter { widget } {
	$self instvar enabled_ currently_tipping_ after_id_
	if {! $enabled_ } return

	set tip [$self make_tip_window $widget]
	if { $tip!="" } {
		wm withdraw $tip
		set delay [option get $tip windowdelay TipManager]
		if { $delay=="" } { set delay 500 }
		set after_id_ [after $delay \
				"catch {wm deiconify $tip}"]
	}
}


# -- proc -------------------------------------------------------------------
# cancel
#
# Cancel any currently displayed tip by unmapping the window (if it's mapped)
# and by canceling the timer to display a window.
# ---------------------------------------------------------------------------
TipManager proc.private cancel { } {
	$self instvar enabled_ currently_tipping_ tip_window_ after_id_

	if {! $enabled_ } return

	if {$currently_tipping_ != ""} {
		catch {destroy $tip_window_}
		set currently_tipping_ ""
		set tip_window_ ""
	}
	catch {after cancel $after_id_}
}


# -- proc ---------------------------------------------------------------
# make_tip_window
#
# Make and map the tip window for the given widget.
# ---------------------------------------------------------------------------
TipManager proc.public make_tip_window { widget } {
	$self instvar currently_tipping_ tip_window_ tips_

	if { ![info exists tips_($widget)] || $tips_($widget)=="" } return ""

	# cancel any previous tip
	TipManager cancel

	# Note that we're mapping the tip window.
	set currently_tipping_ $widget

	# create the tip window and prevent the window manager from slapping a
	# border around it
	set count 0
	while { [winfo exists .tip$count] } { incr count }
	set tip_window_ [toplevel .tip$count -class TipManager -bd 0 \
			-relief flat]
	wm overrideredirect $tip_window_ yes

	# Figure out the X and Y coord of the tip window.
	# It should be below the widget it's tipping,
	# and slightly to the right of its left edge, too.
	set yoffset [option get $tip_window_ yoffset TipManager]
	if { $yoffset=="" } { set yoffset 2 }
	set xoffset [option get $tip_window_ xoffset TipManager]
	if { $xoffset=="" } { set xoffset 20 }

	set y [expr [winfo rooty $currently_tipping_] \
			+ [winfo height $currently_tipping_] + $yoffset]
	set x [expr [winfo rootx $currently_tipping_] + $xoffset]

	# set the position to the tip window
	wm geometry $tip_window_ +$x+$y

	# Insert the message and we're done.
	pack [label $tip_window_.message -text $tips_($widget) -padx 2]
	return $tip_window_
}





WidgetClass TipWidget -configspec {
	{ -tip tip Tip {foobar} config_tip cget_tip }
} -default {
	{ *background  LemonChiffon }
	{ *forground   Black }
	{ *relief      raised }
	{ *borderWidth 1 }
	{ *justify     center }
	{ *font        "-*-helvetica-medium-r-normal-*-*-120-*-*-*-*-*-*" }
	{ *windowdelay 500 }
	{ *yoffset     5 }
	{ *xoffset     20 }
}


# -- proc ------------------------------------------------------------------
# create
#
# The widget creation method accepts the widget class (button, entry, etc.)
# as the first argument and the widget name as the second. But the default
# "WidgetClass create" method expects the widget name as the first argument,
# so flip ther two args here
# --------------------------------------------------------------------------
TipWidget proc create { cl widget args } {
	if { $cl == {} } {
		eval [list $self] next [list _tip_$widget] [list $cl] $widget \
				$args
	} else {
		eval [list $self] next [list $widget] [list $cl] $args
	}
}


# -- proc -------------------------------------------------------------------
# init_
#
# Initialize the tips.  Currently, this just means adding X resources
# and setting up bindings.  Override these in your own X resources, if
# you must.
# ---------------------------------------------------------------------------
TipWidget proc init_ {} {
	$self instvar enabled_ currently_tipping_
	set enabled_ 1
	set currently_tipping_ ""

	# Bindings for the TipWidgetBindings tag.
	bind TipWidgetBindings <Enter> {
		%W enter
	}
	bind TipWidgetBindings <Leave> {
		TipWidget cancel
	}
	bind TipWidgetBindings <Button-1> {
		TipWidget cancel
	}
	bind TipWidgetBindings <space> {
		TipWidget cancel
	}
}


# -- proc -------------------------------------------------------------------
# enable
#
# With boolean true, enable all popup tips.  False, disable all of them.
# ---------------------------------------------------------------------------

TipWidget proc enable { {bool {}} } {
	$self instvar currently_tipping_ enabled_

	if { $bool=={} } { return $enabled_ }
	if {! $bool && $currently_tipping_ != ""} {
		$self cancel
	}
	set enabled_ $bool
}


# -- instproc --------------------------------------------------------------
# init
#
# the widget constructor (note that the widget and class arguments are in
# reverse order from "TipWidget create")
# --------------------------------------------------------------------------
TipWidget instproc init { widget cl args } {
	if { $cl == {} } {
		$self set path_ [lindex $args 0]
		eval $self configure [lrange $args 1 end]
	} else {
		$self set class_ $cl
		eval [list $self] next [list $widget] $args
	}
}


# -- instproc --------------------------------------------------------------
# destroy
#
# the widget destructor: remove the tip, if any
# --------------------------------------------------------------------------
TipWidget instproc destroy { } {
	$self remove_tip
	$self next
}


# -- instproc --------------------------------------------------------------
# create_root_widget
#
# create the widget itself
# --------------------------------------------------------------------------
TipWidget instproc create_root_widget { widget } {
	$self instvar class_
	$class_ $widget
}


# -- instproc ---------------------------------------------------------------
# enter
#
# This is an internal method that is invoked when the <Enter> event occurs
# ---------------------------------------------------------------------------
TipWidget instproc enter { } {
	TipWidget instvar enabled_ currently_tipping_ after_id_
	if {! $enabled_ } continue

	set tip [$self make_tip_window]
	if { $tip!="" } {
		wm withdraw $tip
		set delay [option get $tip windowdelay TipWidget]
		if { $delay=="" } { set delay 500 }
		set after_id_ [after $delay \
				"catch {wm deiconify $tip}"]
	}
}


# -- proc -------------------------------------------------------------------
# cancel
#
# Cancel any currently displayed tip by unmapping the window (if it's mapped)
# and by canceling the timer to display a window.
# ---------------------------------------------------------------------------
TipWidget proc cancel { } {
	$self instvar enabled_ currently_tipping_ tip_window_ after_id_

	if {! $enabled_ } return

	if {$currently_tipping_ != ""} {
		catch {destroy $tip_window_}
		set currently_tipping_ ""
		set tip_window_ ""
	}
	catch {after cancel $after_id_}
}


# -- instproc ---------------------------------------------------------------
# make_tip_window
#
# Make and map the tip window for the given widget.
# ---------------------------------------------------------------------------
TipWidget instproc make_tip_window { } {
	TipWidget instvar currently_tipping_ tip_window_

	set msg [$self cget -tip]
	if { $msg=="" } return ""

	# cancel any previous tip
	TipWidget cancel

	# Note that we're mapping the tip window.
	set currently_tipping_ [$self info path]

	# create the tip window and prevent the window manager from slapping a
	# border around it
	set count 0
	while { [winfo exists .tip$count] } { incr count }
	set tip_window_ [toplevel .tip$count -class TipWidget -bd 0 \
			-relief flat]
	wm overrideredirect $tip_window_ yes

	# Figure out the X and Y coord of the tip window.
	# It should be below the widget it's tipping,
	# and slightly to the right of its left edge, too.
	set yoffset [option get $tip_window_ yoffset TipWidget]
	if { $yoffset=="" } { set yoffset 2 }
	set xoffset [option get $tip_window_ xoffset TipWidget]
	if { $xoffset=="" } { set xoffset 20 }

	set y [expr [winfo rooty $currently_tipping_] \
			+ [winfo height $currently_tipping_] + $yoffset]
	set x [expr [winfo rootx $currently_tipping_] + $xoffset]

	# set the position to the tip window
	wm geometry $tip_window_ +$x+$y

	# Insert the message and we're done.
	pack [label $tip_window_.message -text $msg]
}


TipWidget instproc config_tip { option tip } {
	$self instvar tip_

	if { $tip=={} } {
		$self remove_tip
	} else {
		$self add_tip $tip
	}
}


TipWidget instproc cget_tip { option } {
	$self instvar tip_
	if [info exists tip_] {
		return $tip_
	} else {
		return ""
	}
}


# -- instproc ---------------------------------------------------------------
# add_tip
#
# Add a tip message to the given widget.  When the pointer enters the
# widget, it'll wait a bit, then if it's still there, it'll map the
# tip window.  As soon as the pointer leaves, then it unmaps it.
# ---------------------------------------------------------------------------

TipWidget instproc add_tip { tip } {
	$self instvar tip_

	# Save the message text for later use.
	set tip_ $tip

	# Add the tip bindings to the window ... now, it might already have
	# them if we got called to change the text, so check first.
	set window [$self info path]
	set tags [bindtags $window]
	if {[lsearch -exact $tags TipWidgetBindings] == -1} {
		# They're not in there, add 'em.
		bindtags $window [concat TipWidgetBindings [bindtags $window]]
	}
}


# -- instproc ---------------------------------------------------------------
# remove_tip
#
# Remove the popup tip for this window.
# ---------------------------------------------------------------------------
TipWidget instproc remove_tip { } {
	$self instvar tip_
	TipWidget instvar currently_tipping_

	# Do we have saved text for this window?  If so, then we have to get
	# rid of it and the window's class binding, too.
	if [info exists tip_] {
		# Okay, nuke the text.
		unset tip_

		# Now, get the tags and see if the TipWidgetBindings is
		# among them.
		set window [$self info path]
		set tags [bindtags $window]
		if {[set index [lsearch -exact $tags TipWidgetBindings]] >=0} {
			# Yep, it's there.  Set the new bindings without
			# TipWidgetBindings present.
			bindtags $window [lreplace $tags $index $index]
		}

		# Also, if the tip window is visible for this widget, nuke it.
		if {$currently_tipping_ == [$self info path]} {
			TipWidget cancel
		}
	}
}



# initialize the TipWidget class
TipWidget init_

