# ui-dcthumbnail.tcl --
#
#       Stuff for thumbnail frame, including video thumbnails as well as
#       other services.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import Observer
import DragNDrop

# this is a hack to fix the circular dependency
# ie - don't move this lower then to imports below
Class CDcUIThumbnail

import CDcUIThumbnail/Service/Replay
import CDcUIThumbnail/Video

Class CDcUIThumbnailFrame -superclass Observer

CDcUIThumbnailFrame instproc init { appDc uiMain winFrame } {
    $self instvar m_appDc
    $self instvar m_uiMain
    $self instvar m_winFrame
    $self instvar m_iFrameCounter
    $self instvar m_aVideoThumbnail
    $self instvar m_aWaitingService
    $self instvar m_aWaitingSource

    $self next

    # initialize member variables
    set m_iFrameCounter 0
    set m_aWaitingSerivce("") ""
    set m_aWaitingSource("") ""
    $self instvar m_thumbnailsInfo
    set m_thumbnailsInfo(windows) [list]

    # copy arguments to member variables
    set m_appDc $appDc
    set m_uiMain $uiMain
    set m_winFrame $winFrame

    # lets attach this object to the dc agent
    set studioAgent [$m_appDc get_studio_agent]
    $studioAgent attach $self

    # link the service session so that it'll call this function
    # when something new happens
    # get the service object
    set serviceSession [$m_appDc get_sds_session]
    $serviceSession UpdateInfo "$self BuildMenu"

    $self instvar m_font
    set m_font [$self get_option smallfont]

    # build the ui
    $self BuildUI
}

CDcUIThumbnailFrame instproc BuildUI {} {
    $self instvar m_winFrame m_font
    

    # the layout of this frame goes like this:
    # Inside the main frame, there are four frames
    # the top frame has a label in it
    # the middle frames has a column of thumbnails
    # the bottom frame has a row of menu buttons

    # the top frame
    label $m_winFrame.title -text "Sources" -relief ridge -font $m_font
    pack $m_winFrame.title -side top -fill x


    # the middle frames
    frame $m_winFrame.video -relief ridge
    pack $m_winFrame.video -side top -fill both

    # the frame for the bottom
    frame $m_winFrame.menubar -relief ridge
    pack $m_winFrame.menubar -side bottom -fill x

    # the frame for the services
    frame $m_winFrame.service -relief ridge
    pack $m_winFrame.service -side bottom -fill both

    # the button in the bottom frame
    menubutton $m_winFrame.menubar.button -text "Service" \
	    -menu $m_winFrame.menubar.button.menu -relief raised -font $m_font
    pack $m_winFrame.menubar.button -fill x -expand 1

    # the menu from which entries will be deleted and added
    menu $m_winFrame.menubar.button.menu

    $self BuildMenu
}

CDcUIThumbnailFrame instproc BuildMenu { } {
    $self instvar m_appDc
    $self instvar m_winFrame
    $self instvar m_font

    # first get the data from the service session object
    set serviceSession [$m_appDc get_sds_session]
    set data [$serviceSession set m_data]

    # now let's parse the data
    set lTree [$self ParseData $data]

    # destroy the old menu
    destroy $m_winFrame.menubar.button.menu

    # construct the new menu
    menu $m_winFrame.menubar.button.menu -font $m_font
    $self AddMenuEntry $m_winFrame.menubar.button.menu $lTree root
}

CDcUIThumbnailFrame instproc AddMenuEntry { menuCurrent lTree parent } {
    array set aTree $lTree
    $self instvar m_font

    set iCount 0
    set children $aTree($parent)
    set f $m_font
    foreach child $children {
	if { [info exists aTree($child)] } {
	    $menuCurrent add cascade -label $child -menu $menuCurrent.$iCount
	    menu $menuCurrent.$iCount -font $m_font
	    $self AddMenuEntry $menuCurrent.$iCount $lTree $child
	} else {
	    $menuCurrent add command -label $child \
		    -command "$self StartService [list $child]" 
	}
	set iCount [expr "$iCount + 1"]
    }
}

CDcUIThumbnailFrame instproc ParseData { data } {
    set aTree(root) ""

    # iterate through each pair of contact info and properties
    foreach { contactInfo properties } $data {

	# iterate through all the prperties
	foreach property $properties {

	    # get which property it is. ie which attribute
	    set attribute [lindex $property 0]

	    # it's a location property and construct a tree with location
	    # information
	    if { $attribute == "LOCATION:" } {
		set parent root
		foreach node [lindex $property 1] {

		    if { [info exists aTree($parent)] } {
			set children $aTree($parent)
		    } else {
			set children ""
		    }

		    if { ![string match $node $children] } {
			set aTree($parent) [concat [list $node] $children]
		    }

		    set parent $node
		}
		set lastNode $parent
	    }

	    # save the type information for the leaf node of the location tree
	    if { $attribute == "TYPE:" } {
		set infoType [lindex $property 1]
	    }
	}

	# construct the leaf node
	if { [info exists aTree($lastNode)] } {
	    set children $aTree($lastNode)
	} else {
	    set children {}
	}
	set node "$infoType : $contactInfo"
	set aTree($lastNode) [concat [list $node] $children]
    }

    puts "tree = [array get aTree]"
    return [array get aTree]
}

#
# This is called when someone selects a service from the "Service" button in
#    DC.
#
CDcUIThumbnailFrame instproc StartService { menuEntry } {
    # first of all I need to parse the service entry
    # the format shoube be "type:contact info"
    set lEntry [split $menuEntry :]

    set strType [lindex $lEntry 0]
    set lContactInfo [lindex $lEntry 1]
    set inetAddr [lindex $lContactInfo 0]
    set iPort [lindex $lContactInfo 1]

    # get some needed member variables
    $self instvar m_appDc
    $self instvar m_uiMain
    $self instvar m_winFrame
    $self instvar m_iFrameCounter

    # create the new thumbnail object
#    puts "strType = $strType;"

    switch -exact -- $strType {
	"Video Capture Device " {
	    # so the idea here is that we want to connect temporarily to send
	    # the start message and then the rvc will initiate another service
	    # with the appropriate video source id.  Once the service is gotten
	    # the thunmbnailframe will hook up the correct service to the
	    # thumbnail

	    # get the service manager and create a new service
	    set service [$m_appDc new_service $inetAddr $iPort]

	    # get options to give to the video service
	    set options [$m_appDc options]
	    set inetMStudioAddr [$options get_option optMStudioAddress]
	    set iMStudioPort [$options get_option optMStudioPort]
	    set spec [$m_appDc get_service_spec]
	    set inetClientAddr [lindex $spec 0]
	    set iClientPort [lindex $spec 1]
	    set arguments [list $inetMStudioAddr $iMStudioPort \
		    $inetClientAddr $iClientPort]

	    $service Send "SYN_START" $arguments
	    delete $service 
	}

	"Archive Replay " {
	    # we have to create the frame
	    set winFrame [frame $m_winFrame.service.$m_iFrameCounter]
	    pack $winFrame -side bottom -fill x

	    # increment the frame counter
	    set m_iFrameCounter [expr "$m_iFrameCounter + 1"]

	    # create the new thumbnail object
	    set uiThumbnail [new CDcUIThumbnail/Service/Replay $m_appDc \
		    $m_uiMain $winFrame]

	    # get the service manager and create a new service
	    set service [$m_appDc new_service $inetAddr $iPort]

	    $uiThumbnail StartService $service
	}

	"Special FX "  {
	    # we have to create the frame
	    set winFrame [frame $m_winFrame.service.$m_iFrameCounter]
	    pack $winFrame -side bottom -fill x

	    # increment the frame counter
	    set m_iFrameCounter [expr "$m_iFrameCounter + 1"]

	    # create the new thumbnail object
	    set uiThumbnail [new CDcUIThumbnail/Service/SpecialFx $m_appDc \
		    $m_uiMain $winFrame]

	    # get the service manager and create a new service
	    set service [$m_appDc new_service $inetAddr $iPort]

	    $uiThumbnail StartService $service
	}

	"405 Room Control " -
	"Real Networks Switcher " {
	    # we have to create the frame
	    set winFrame [frame $m_winFrame.service.$m_iFrameCounter]
	    pack $winFrame -side bottom -fill x

	    # increment the frame counter
	    set m_iFrameCounter [expr "$m_iFrameCounter + 1"]

	    # create the new thumbnail object
	    set uiThumbnail [new CDcUIThumbnail/Service $m_appDc \
		    $m_uiMain $winFrame]

	    # get the service manager and create a new service
	    set service [$m_appDc new_service $inetAddr $iPort]

	    $uiThumbnail StartService $service
	}
    }
}

CDcUIThumbnailFrame instproc activate {source} {
    #
    # As noted in vic/ui-activesourcemgr.tcl,
    # we need to wait until the decoder is created.
    # This needs to be fixed when mash is fixed??
    #
    after idle "$self ReallyActivate $source"
}

CDcUIThumbnailFrame instproc ReallyActivate {source} {
    $self instvar m_aVideoThumbnail
    $self instvar m_aWaitingService
    $self instvar m_aWaitingSource

    set iSourceId [$source srcid]

    # so the first thing is to find if a service already exists waiting
    # for this incoming source. So search through the table of waiting
    # services
    if { [info exists m_aWaitingService($iSourceId)] } {
	# create the new video thumbnail
	set uiThumbnail [$self NewVideoThumbnail $source \
		$m_aWaitingService($iSourceId)]
	set m_aVideoThumbnail($iSourceId) $uiThumbnail

	# remove this entry from the list
	unset m_aWaitingService($iSourceId)

	return
    }

    # otherwise we have to wait for a while to let the incomming
    # service come in.  After the wait just start the source without
    # a service

    set m_aWaitingSource($iSourceId) $source
    after 5000 "$self ServiceTimeout $source"
}


CDcUIThumbnailFrame instproc deactivate {source} {
    $self DeleteVideoThumbnail $source    
}


##############################################################################
#
# CDcUIThumbnailFrame public ServiceTimeout { source } {
#
# Input:
# source - the source that is waiting for the service to come in but it never
#    did so a timeout occured
#
# Output:
# none
#
# Description:
# This is called some time after the source comes in.  If no service connects
# between that time then this source goes out as a serverless thumbnail.
#
##############################################################################
CDcUIThumbnailFrame public ServiceTimeout { source } {
    $self instvar m_aWaitingSource
    $self instvar m_aVideoThumbnail

    set iSourceId [$source srcid]

    # check if the waiting source has be taken
    if { ![info exists m_aWaitingSource($iSourceId)] } {
	return
    }

    # otherwise we need to create a new thumbnail without the service

    # create the new video thumbnail
    set uiThumbnail [$self NewVideoThumbnail $source]
    set m_aVideoThumbnail($iSourceId) $uiThumbnail

    # remove this entry from the list
    unset m_aWaitingSource($iSourceId)
}


##############################################################################
#
# CDcUIThumbnailFrame public NewConnection { service }
#
# Input:
# service - the newly connected service object
#
# Output:
# none
#
# Description:
# This is called by the dc application object when a service of the correct
# type has arrived.  We take this service object and try to match it to one
# of the waiting sources.  If none are waiting then we put them into
# a waiting list so that the thumbnails can get them when the source is
# activated
#
##############################################################################
CDcUIThumbnailFrame public NewConnection { service } {
    $self instvar m_aVideoThumbnail
    $self instvar m_aWaitingSource
    $self instvar m_aWaitingService

    # first check that the service is of type VideoService
    if { [$service set m_szType] != "VideoService" } {
	return
    }

    set iSourceId [$service set m_szAttribute]

    # check if the source is waiting at the moment
    if { [info exists m_aWaitingSource($iSourceId)] } {
	# create the new video thumbnail
	set uiThumbnail [$self NewVideoThumbnail \
		$m_aWaitingSource($iSourceId) $service]
	set m_aVideoThumbnail($iSourceId) $uiThumbnail

	# remove this entry from the list
	unset m_aWaitingSource($iSourceId)

	return
    }

    # otherwise we have to wait until a source comes in
    set m_aWaitingService($iSourceId) $service
}

##############################################################################
#
# CDcUIThumbnailFrame public NewVideoThumbnail { source {service 0} }
#
# Input:
# source - the source object of the video stream
# service - the service object - defaulted to 0
#
# Output:
# CDcUIThumbnail/Video object that is created
#
# Description:
# will
#
##############################################################################
CDcUIThumbnailFrame public NewVideoThumbnail { source {service 0} } {
    $self instvar m_appDc
    $self instvar m_uiMain
    $self instvar m_winFrame
    $self instvar m_iFrameCounter

    # we have to create the frame
    set winFrame [frame $m_winFrame.video.$m_iFrameCounter]
    pack $m_winFrame.video.$m_iFrameCounter -side top

    # increment the frame counter
    set m_iFrameCounter [expr "$m_iFrameCounter + 1"]

    # create the new thumbnail object
    set uiThumbnail [new CDcUIThumbnail/Video $m_appDc $m_uiMain $winFrame]

    set inetAddr [$source addr]
    set hostname [lookup_host_name $inetAddr]
    set hostname [string tolower $hostname]
    $self instvar m_thumbnailsInfo
    # FIXME - need to deal with this case better
    if {[info exists m_thumbnailsInfo($hostname,addr)]} {
	puts stdout "warning, service from $inetAddr already exists!!!"
    }
    lappend m_thumbnailsInfo(windows) $uiThumbnail
    set m_thumbnailsInfo($uiThumbnail,hostname) $hostname
    set m_thumbnailsInfo($uiThumbnail,addr) $inetAddr


    # start the video
    $uiThumbnail StartVideo $source

    # start the service if we have one
    if { $service != 0 } {
	$uiThumbnail StartService $service
    }

    return $uiThumbnail
}


#---------------------------------------------------------------------------
#
# CDcUIThumbnailFrame public DeleteVideoThumbnail { source }
#
# Input:
# source - the source object of the video stream whose thumbnail we 
#          are going to delete
#
#---------------------------------------------------------------------------
CDcUIThumbnailFrame public DeleteVideoThumbnail { source } {

    #
    # Retrieve the associated thumbnail
    #
    $self instvar m_aVideoThumbnail
    set srcid [$source srcid]
    if ![info exists m_aVideoThumbnail($srcid)] {
	puts "WARNING: no video thumbnail exists for $srcid"
	return
    }
    set thumbnail $m_aVideoThumbnail($srcid)

    #
    # Cleanup thumbnails info
    #
    $self instvar m_thumbnailsInfo
    set pos [lsearch m_thumbnailsInfo(windows) $thumbnail]
    if {$pos > 0} {
	lreplace m_thumbnailsInfo(windows) $pos $pos
    } 
    unset m_thumbnailsInfo($thumbnail,hostname) 
    unset m_thumbnailsInfo($thumbnail,addr) 

    $thumbnail StopVideo 

    #
    # Cleanup UI stuffs
    #
    set winFrame [$thumbnail GetWinFrame]
    destroy $winFrame 

    delete $thumbnail
}


#---------------------------------------------------------------------------
#
# CDcUIThumbnailFrame public GetSourceIDs { }
#
# Purpose:
#   Return a list of source ids for all thumbnails managed by this frame.
#
#---------------------------------------------------------------------------
CDcUIThumbnailFrame public GetSourceIDs { } {
    $self instvat m_aVideoThumbnail
    if {[info exists m_aVideoThumbnail]} {
	return [array names $m_aVideoThumbnail]
    } else {
	return ""
    }
}


#
# Return the CDcUIPreview object containing the source 
#
CDcUIThumbnailFrame public GetUIVideoWithSource { source } {
    $self instvar m_thumbnailsInfo
    foreach v $m_thumbnailsInfo(windows) {
	if {[$v GetSource] == $source} {
	    return $v
	} 
    }
    return ""
}



# Class declaration at the top
CDcUIThumbnail public init { appDc uiMain winFrame } {
    $self instvar m_appDc m_uiMain m_winFrame

    # store away input
    set m_appDc $appDc
    set m_uiMain $uiMain
    set m_winFrame $winFrame
}


CDcUIThumbnail public GetWinFrame { } {
    return [$self set m_winFrame]
}


CDcUIThumbnailFrame public getThumbnailInfo {} {
    $self instvar m_thumbnailsInfo

    set retList [array get m_thumbnailsInfo]
    return $retList
}

