# dc-api.tcl --
#
#       Object contained in the app that opens a TclDP RPC port to allow
#       remote processes to control the DC, such as choosing which streams
#       to broadcast.
#
# Copyright (c) 2000-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import DpClient

# this class provides remote control of the DC using the TclDP RPC mechanism;
#   it affects the DC by simulating GUI events

namespace eval DcApi {
    variable self
}

Class DcApi

DcApi instproc init {appDC uiDC port} {
    set DcApi::self $self

    $self instvar m_appHandle m_uiHandle m_server
    set m_appHandle $appDC
    set m_uiHandle $uiDC

    # get copies of these variables from the UI for future use
    $self instvar m_uiThumbnailFrame m_uiPreviewFrame m_uiBroadcastFrame
    $self instvar m_framePV m_frameBC
    set m_uiThumbnailFrame [$m_uiHandle set m_uiThumbnailFrame]
    set m_uiPreviewFrame [$m_uiHandle set m_uiPreviewFrame]
    set m_uiBroadcastFrame [$m_uiHandle set m_uiBroadcastFrame]
    set m_framePV [$m_uiHandle set m_framePV]
    set m_frameBC [$m_uiHandle set m_frameBC]

    # create a TclDP RPC server
    set m_server [new DpServer $port]
}

# FIXME - we should catch errors or something here so we can't crash DC

DcApi public broadcastThumbnail {uiThumbnail} {
    $self instvar m_uiBroadcastFrame m_frameBC

    set x [winfo rootx $m_frameBC]
    set y [winfo rooty $m_frameBC]
    # use a dummy dragNDrop since don't need to zoom
    $m_uiBroadcastFrame DropThumbnail "dummy" $uiThumbnail $x $y
}

DcApi public previewThumbnail {uiThumbnail} {
    $self instvar m_uiPreviewFrame m_framePV

    set x [winfo rootx $m_framePV]
    set y [winfo rooty $m_framePV]
    # use a dummy dragNDrop since don't need to zoom
    $m_uiPreviewFrame DropThumbnail "dummy" $uiThumbnail $x $y
}

DcApi public removePreview {uiPreview} {
    $self instvar m_uiPreviewFrame

    $m_uiPreviewFrame RemovePreview $uiPreview
}

DcApi public removeBroadcast {uiBroadcast} {
    $self instvar m_uiBroadcastFrame

    $m_uiBroadcastFrame RemoveBroadcast $uiBroadcast
}

#
# the following 3 functions return info about the windows present in the
#   DC
#
# they all return an array in the form of a list, to use the info, you should:
#
#         set infoList [$self getThumbnailInfo]
#         array set infoArray $infoList
#
# the "windows" element of the array is a list of windows in the given frame
#
#         set windowList $infoArray(windows)
#
# you can then go through each item and look at the values associated with
#    each item
#
#         foreach index $windowList {
#             set addr $infoArray($index,addr)
#             set addr $infoArray($index,hostname)
#             $self broadcastThumbnail $index
#         }
#
# the index itself is a handle to the thumbnail, preview, or broadcast window
#    and can be passed to the DcApi functions to cause/cease transmission of
#    video from a given source (as seen in the above "foreach" loop)

DcApi public getThumbnailInfo {} {
    $self instvar m_uiThumbnailFrame

    return [$m_uiThumbnailFrame getThumbnailInfo]
}

DcApi public getPreviewInfo {} {
    $self instvar m_uiPreviewFrame

    return [$m_uiPreviewFrame getPreviewInfo]
}

DcApi public getBroadcastInfo {} {
    $self instvar m_uiBroadcastFrame

    return [$m_uiBroadcastFrame getBroadcastInfo]
}

DcApi public clearBroadcastPane {} {
    $self instvar m_uiBroadcastFrame

    set broadcastInfoList [$m_uiBroadcastFrame getBroadcastInfo]
    array set broadcastInfo $broadcastInfoList
    foreach index $broadcastInfo(windows) {
	$self removeBroadcast $index
    }
}

DcApi public clearPreviewPane {} {
    $self instvar m_uiPreviewFrame

    set broadcastInfoList [$m_uiPreviewFrame getPreviewInfo]
    array set broadcastInfo $broadcastInfoList
    foreach index $broadcastInfo(windows) {
	$self removePreview $index
    }
}

DcApi instproc getInfo {} {
    $self instvar m_uiThumbnailFrame m_uiPreviewFrame m_uiBroadcastFrame

    puts stdout "CDcUI::getInfo: called"

    set thumbs [$m_uiThumbnailFrame getThumbnailInfo]

    array set thumbsInfo $thumbs
    set thumbsIndices $thumbsInfo(windows)

    puts stdout "thumbs: $thumbs"
    puts stdout "thumbsIndices: $thumbsIndices"

    $self clearPreviewPane
    set x 1
    after 1000 {set x 0}
    vwait x
    $self clearBroadcastPane
    set x 1
    after 1000 {set x 0}
    vwait x

    # preview window
    set index [lindex $thumbsIndices 0]
    $self previewThumbnail $index

    # broadcast window
    set index [lindex $thumbsIndices 1]
    $self broadcastThumbnail $index
}

#
# API
#
# The functions below can be called remotely using the TclDP RPC mechanism
#   to cause GUI events to occur, simulating a director drag-n-dropping
#

# we need wrappers since the remote callers don't have a handle to the DcApi
#   object

proc dcApi_getInfo {} {
    return [$DcApi::self getInfo]
}

proc dcApi_clearPreviewPane {} {
    return [$DcApi::self clearPreviewPane]
}

proc dcApi_clearBroadcastPane {} {
    return [$DcApi::self clearBroadcastPane]
}

proc dcApi_getThumbnailInfo {} {
    return [$DcApi::self getThumbnailInfo]
}

proc dcApi_getPreviewInfo {} {
    return [$DcApi::self getPreviewInfo]
}

proc dcApi_getBroadcastInfo {} {
    return [$DcApi::self getBroadcastInfo]
}

proc dcApi_previewThumbnail {uiThumbnail} {
    return [$DcApi::self previewThumbnail $uiThumbnail]
}

proc dcApi_broadcastThumbnail {uiThumbnail} {
    return [$DcApi::self broadcastThumbnail $uiThumbnail]
}

proc dcApi_removePreview {uiPreview} {
    return [$DcApi::self removePreview $uiPreview]
}

proc dcApi_removeBroadcast {uiBroadcast} {
    return [$DcApi::self removeBroadcast $uiBroadcast]
}
