/*
 * module-spat_combine-sc.cc --
 *
 *      This file contains the definition and methods for the
 *      SCSpatialCombineModule class and associated helper classes.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "inet.h"
#include "rtp.h"
#include "tclcl.h"
#include "pktbuf.h"
#include "codec/module.h"
#include "rtp/media-timer.h"

/*
 * Debuging macros.
 */

#ifdef KPATEL_SPAT_COMB_DEBUG
#include "logger.h"

#define LOG_MESG(x) \
{ \
  sprintf(log_mesg, "Comb %s %u %u %u", x , ssrc_id, seq_no, ts); \
  logger.gen_log(log_mesg); \
}
#else
#define LOG_MESG(x)
#endif

/*
 * Sequence number comparison macros
 */

#define SEQNO_GT(s1, s2) (s1) > (s2)
#define SEQNO_LT(s1, s2) (s1) < (s2)
#define SEQNO_EQ(s1, s2) (s1) == (s2)

/*
 * Media timestamp comparison macros
 */

#define MTS_LT(m1, m2) (m1) < (m2)
#define MTS_EQ(m1, m2) (m1) == (m2)
#define MTS_GT(m1, m2) (m1) > (m2)


class SourceQueue;

class SourceQueue {
public:
  SourceQueue(u_int32_t ssrc, u_int16_t seqno, u_int32_t ts) {
    ssrc_id_ = ssrc;
    last_seqno_= seqno - 1;
    next = prev = 0;
    last_ts_ = ts;
    ts_delta_est_ = 0;
  };

  u_int32_t ssrc_id_;
  u_int32_t last_ts_;
  u_int16_t last_seqno_;
  u_int32_t ts_delta_est_;
  SourceQueue *next;
  SourceQueue *prev;

  void update_ts_delta(u_int32_t ts) {
    if (MTS_GT(ts,last_ts_)) {
      ts_delta_est_ = (ts_delta_est_ / 2) + ((ts - last_ts_)/2);
      last_ts_ = ts;
    }
  }
};

class SCSpatialCombineModule : public PacketModule {
public:
  SCSpatialCombineModule() : PacketModule(), ssrc_(0), seq_no_(0),
    last_sent_flag_(0), first_time_flag_(1),  future_queue_(0),
    next_frame_pkt_queue_(0), behind_(0), ahead_(0), off_synch_cmd_(0) {
      stats_[0] = 0;
      stats_[1] = 0;
      stats_[2] = 0;
      stats_[3] = 0;
      stats_[4] = 0;
      stats_[5] = 0;
      stats_[6] = 0;
      stats_[7] = 0;
  };

  void recv(pktbuf *pb);
  virtual int command(int argc, const char*const* argv);

protected:
  u_int32_t ssrc_;                /* Source id for resulting packet stream. */

  u_int32_t last_ts_;             /* Timestamp of last emitted packet.      */

  u_int16_t seq_no_;              /* Next packet sequence number to use.    */

  int last_sent_flag_;            /* Indicates whether frame marker bit of
				   * last packet was set. */

  int first_time_flag_;           /* Indicates whether next packet will be
				   * first packet processed. */

  pktbuf *future_queue_;
  pktbuf *next_frame_pkt_queue_;

  SourceQueue *behind_;
  SourceQueue *ahead_;

  static const int BEHIND;
  static const int AHEAD;

  void move_from_behind_to_ahead(SourceQueue *sq);
  void queue_for_send(pktbuf *pb);
  void queue_for_future(pktbuf *pb);
  void advance_frame();
  void reset_next_frame_queue();
  void frame_skip();
  void process_future_queue();

  int stats_[8];
  static const int STAT_AHEAD;
  static const int STAT_BEHIND_BADSEQ_OUTORDER;
  static const int STAT_BEHIND_BADSEQ_TS_GT;
  static const int STAT_BEHIND_BADSEQ_TS_EQ;
  static const int STAT_BEHIND_BADSEQ_TS_LT;
  static const int STAT_BEHIND_GOODSEQ_TS_GT;
  static const int STAT_BEHIND_GOODSEQ_TS_EQ;
  static const int STAT_BEHIND_GOODSEQ_TS_LT;

  char *off_synch_cmd_;
};

const int SCSpatialCombineModule::BEHIND = 0;
const int SCSpatialCombineModule::AHEAD = 1;

const int SCSpatialCombineModule::STAT_AHEAD = 0;
const int SCSpatialCombineModule::STAT_BEHIND_BADSEQ_OUTORDER = 1;
const int SCSpatialCombineModule::STAT_BEHIND_BADSEQ_TS_GT = 2;
const int SCSpatialCombineModule::STAT_BEHIND_BADSEQ_TS_EQ = 3;
const int SCSpatialCombineModule::STAT_BEHIND_BADSEQ_TS_LT = 4;
const int SCSpatialCombineModule::STAT_BEHIND_GOODSEQ_TS_GT = 5;
const int SCSpatialCombineModule::STAT_BEHIND_GOODSEQ_TS_EQ = 6;
const int SCSpatialCombineModule::STAT_BEHIND_GOODSEQ_TS_LT = 7;

void
SCSpatialCombineModule::recv(pktbuf *pb)
{
#ifdef KPATEL_SPAT_COMB_DEBUG
  char log_mesg[200];
#endif

  rtphdr *rh = (rtphdr *)pb->data;

  u_int32_t ts = ntohl(rh->rh_ts);
  u_int32_t ssrc_id = ntohl(rh->rh_ssrc);
  u_int16_t seq_no = ntohs(rh->rh_seqno);
  int marker_flag = ntohs(rh->rh_flags) & RTP_M;

  if (first_time_flag_) {
    last_ts_ = ts;
    first_time_flag_ = 0;
  }


  /* Find the source queue */
  SourceQueue *sq;
  int which_list;

  sq = behind_;
  which_list = BEHIND;
  while (sq != 0) {
    if (sq->ssrc_id_ == ssrc_id) {
      break;
    }
    sq = sq->next;
  }
  if (sq == 0) {
    sq = ahead_;
    which_list = AHEAD;
    while (sq != 0) {
      if (sq->ssrc_id_ == ssrc_id) {
	break;
      }
      sq = sq->next;
    }
    if (sq == 0) {
      /* New source. */
      which_list = BEHIND;
      sq = new SourceQueue(ssrc_id, seq_no, ts);
      sq->next = behind_;
      sq->prev = 0;
      if (behind_ != 0) {
	behind_->prev = sq;
      }
      behind_ = sq;
    }
  }

  sq->update_ts_delta(ts);

  if (which_list == BEHIND) {
    if (seq_no != sq->last_seqno_ + 1) {
      if (SEQNO_GT(seq_no, sq->last_seqno_)) {
	if (MTS_LT(ts, last_ts_)) {
	  LOG_MESG("Behind_BadSeq_SmallTS");
	  stats_[STAT_BEHIND_BADSEQ_TS_LT] += 1;
	  sq->last_seqno_ = seq_no;
	  pb->release();
	  return;
	} else if (MTS_EQ(ts, last_ts_)) {
	  LOG_MESG("Behind_BadSeq_EqTS");
	  stats_[STAT_BEHIND_BADSEQ_TS_EQ] += 1;

	  sq->last_seqno_ = seq_no;
	  pb->release();
	  frame_skip();
	  return;
	} else {
	  LOG_MESG("Behind_BadSeq_BigTS");
	  stats_[STAT_BEHIND_BADSEQ_TS_GT] += 1;

	  move_from_behind_to_ahead(sq);
	  queue_for_future(pb);
	  frame_skip();
	  return;
	}
      } else {
	LOG_MESG("Behind_BadSeq_OutOrder");
	stats_[STAT_BEHIND_BADSEQ_OUTORDER] += 1;

	pb->release();
	return;
      }
    } else {
      if (MTS_LT(ts, last_ts_)) {
	LOG_MESG("Behind_GoodSeq_SmallTS");
	stats_[STAT_BEHIND_GOODSEQ_TS_LT] += 1;

	sq->last_seqno_ = seq_no;
	pb->release();
	return;
      } else if (MTS_EQ(ts, last_ts_)) {
	LOG_MESG("Behind_GoodSeq_EqTS");
	stats_[STAT_BEHIND_GOODSEQ_TS_EQ] += 1;

	sq->last_seqno_ = seq_no;
	queue_for_send(pb);
	if (marker_flag) {
	  move_from_behind_to_ahead(sq);
	  if (behind_ == 0) {
	    advance_frame();
	  }
	}
	return;
      } else {
	LOG_MESG("Behind_GoodSeq_BigTS");
	stats_[STAT_BEHIND_GOODSEQ_TS_GT] += 1;

	move_from_behind_to_ahead(sq);
	queue_for_future(pb);
	frame_skip();
	return;
      }
    }
  } else {
    LOG_MESG("Ahead");
    stats_[STAT_AHEAD] += 1;

    queue_for_future(pb);
    return;
  }
}

void
SCSpatialCombineModule::move_from_behind_to_ahead(SourceQueue *sq) {

  if (sq->prev != 0) {
    sq->prev->next = sq->next;
  } else {
    behind_ = sq->next;
  }
  if (sq->next != 0) {
    sq->next->prev = sq->prev;
  }
  sq->next = ahead_;
  sq->prev = 0;
  if (ahead_ != 0) {
    ahead_->prev = sq;
  }
  ahead_ = sq;
}

void
SCSpatialCombineModule::frame_skip() {
#ifdef KPATEL_SPAT_COMB_DEBUG
  char log_mesg[200];

  sprintf(log_mesg, "Comb Frame_Skip %u", last_ts_);
  logger.gen_log(log_mesg);

#endif
  reset_next_frame_queue();

  if (ahead_ != 0) {
    SourceQueue *temp_sq;

    temp_sq = ahead_;

    while(temp_sq->next != 0) {
      temp_sq = temp_sq->next;
    }
    temp_sq->next = behind_;
    if (behind_ != 0) {
      behind_->prev = temp_sq;
    }
    behind_ = ahead_;
    ahead_ = 0;
  }

  if (off_synch_cmd_ != 0) {
    Tcl &tcl = Tcl::instance();
    tcl.eval(off_synch_cmd_);
  }

  process_future_queue();
}

void
SCSpatialCombineModule::reset_next_frame_queue() {
  pktbuf *pb, *next_pb;

  pb = next_frame_pkt_queue_;
  while (pb != 0) {
    next_pb = pb->next;
    pb->release();
    pb = next_pb;
  }
  next_frame_pkt_queue_ = 0;
}


void
SCSpatialCombineModule::advance_frame() {
  pktbuf *pb, *next_pb;

#ifdef KPATEL_SPAT_COMB_DEBUG
  char log_mesg[200];

  sprintf(log_mesg, "Comb Advance_Frame %u", last_ts_);
  logger.gen_log(log_mesg);

#endif

  pb = next_frame_pkt_queue_;

  while (pb != 0) {
    rtphdr *rh = (rtphdr *)pb->data;
    u_int16_t *sc_hdr = (u_int16_t *)(rh+1);

    rh->rh_seqno = htons(seq_no_++);
    rh->rh_ssrc = htonl(ssrc_);
    rh->rh_ts = htonl(last_ts_);
    if (pb->next != 0)
      rh->rh_flags = rh->rh_flags & (~htons(RTP_M));
    else
      rh->rh_flags = rh->rh_flags & (htons(RTP_M));
    sc_hdr[0] = sc_hdr[2];
    sc_hdr[1] = sc_hdr[3];
    sc_hdr[4] = htons(0);
    sc_hdr[5] = htons(0);

    next_pb = pb->next;
    if (target_) {
      target_->recv(pb);
    } else {
      pb->release();
    }
    pb = next_pb;
  }
  next_frame_pkt_queue_ = 0;

  behind_ = ahead_;
  ahead_ = 0;
  process_future_queue();
}

void
SCSpatialCombineModule::process_future_queue() {
  pktbuf *pb, *next_pb;

  pb = future_queue_;
  future_queue_ = 0;

  if (pb != 0) {
    rtphdr *rh = (rtphdr *) pb->dp;
    last_ts_ = ntohl(rh->rh_ts);

    while(pb != 0) {
      next_pb = pb->next;
      pb->next = 0;
      recv(pb);
      pb = next_pb;
    }
  } else {
    first_time_flag_ = 1;
  }
}

void
SCSpatialCombineModule::queue_for_send(pktbuf *pb) {
  pb->next = next_frame_pkt_queue_;
  next_frame_pkt_queue_ = pb;
}

void
SCSpatialCombineModule::queue_for_future(pktbuf *pb) {
  rtphdr *rh = (rtphdr *) pb->dp;
  pktbuf *cur = future_queue_;
  pktbuf *prev = 0;

  while (cur != 0) {
    rtphdr *qrh = (rtphdr *) cur->dp;
    if (ntohl(rh->rh_ts) <= ntohl(qrh->rh_ts)) {
      if (ntohl(rh->rh_ssrc) <= ntohl(qrh->rh_ssrc)) {
	if (ntohs(rh->rh_seqno) <= ntohs(qrh->rh_seqno)) {
	  if (prev == 0) {
	    future_queue_ = pb;
	  } else {
	    prev->next = pb;
	  }
	  pb->next = cur;
	  return;
	}
      }
    }
    prev = cur;
    cur = cur->next;
  }

  if (prev == 0) {
    future_queue_ = pb;
  } else {
    prev->next  = pb;
  }
  pb->next = 0;
};



int
SCSpatialCombineModule::command(int argc, const char*const* argv)
{
  Tcl &tcl = Tcl::instance();

  if (argc == 3) {
    if (strcmp(argv[1], "ssrc") == 0) {
      u_int32_t new_ssrc;

      new_ssrc = atoi(argv[2]);
      ssrc_ = new_ssrc;
      return TCL_OK;
    }
    if (strcmp(argv[1], "off_synch_cmd") == 0) {
      if (off_synch_cmd_ != 0) {
	free(off_synch_cmd_);
	off_synch_cmd_ = 0;
      }
      if (strlen(argv[2]) != 0) {
	off_synch_cmd_ = (char *) malloc(strlen(argv[2])+1);
	strcpy(off_synch_cmd_, argv[2]);
      }
      return TCL_OK;
    }
  }
  if (argc == 2) {
    if (strcmp(argv[1], "stats") == 0) {
      printf("Ahead: %d\n", stats_[STAT_AHEAD]);
      printf("Behind_Bad_Seq    LT: %d  EQ: %d  GT: %d  OutOrder: %d\n",
	     stats_[STAT_BEHIND_BADSEQ_TS_LT],
	     stats_[STAT_BEHIND_BADSEQ_TS_EQ],
	     stats_[STAT_BEHIND_BADSEQ_TS_GT],
	     stats_[STAT_BEHIND_BADSEQ_OUTORDER]);
      printf("Behind_Good_Seq   LT: %d  EQ: %d  GT: %d\n",
	     stats_[STAT_BEHIND_GOODSEQ_TS_LT],
	     stats_[STAT_BEHIND_GOODSEQ_TS_EQ],
	     stats_[STAT_BEHIND_GOODSEQ_TS_GT]);
      return TCL_OK;
    }
    if (strcmp(argv[1], "clear_stats") == 0) {
      for(int i=0; i<8; i++) {
	stats_[i] = 0;
      }
      return TCL_OK;
    }
    if (strcmp(argv[1], "ts_delta_est") == 0) {
      u_int32_t max = 0;
      SourceQueue *sq;

      sq = ahead_;
      while (sq != 0) {
	if (sq->ts_delta_est_ > max) {
	  max = sq->ts_delta_est_;
	}
	sq = sq->next;
      }
      sq = behind_;
      while (sq != 0) {
	if (sq->ts_delta_est_ > max) {
	  max = sq->ts_delta_est_;
	}
	sq = sq->next;
      }
      tcl.resultf("%u", max);
      return TCL_OK;
    }
  }

  return (PacketModule::command(argc, argv));
}

static class SCSpatialCombineModuleClass : public TclClass {
public:
  SCSpatialCombineModuleClass() : TclClass("Module/Combine/Spatial/SC") {}
  TclObject *create(int argc, const char*const* argv) {
    return (new SCSpatialCombineModule());
  }
} sc_spatial_combine_;

#undef SEQNO_GT
#undef SEQNO_LT
#undef SEQNO_EQ
#undef MTS_LT
#undef MTS_EQ
#undef MTS_GT
