/*
 * net-ip.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1993-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: /usr/mash/src/repository/mash/mash-1/net/net-ip.cc,v 1.30 2002/02/03 04:13:45 lim Exp $";

#include <stdio.h>
#include <errno.h>
#include <string.h>
#if defined(_AIX)
#include <strings.h>
#endif
#ifdef WIN32
#include <io.h>
#define close closesocket
#else
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#endif
#if defined(sun) && defined(__svr4__)
#include <sys/systeminfo.h>
#endif

#include "config.h"
#include "net.h"
#include "tclcl.h"

/*
 * To enable unicast compatibility with rtptv, set RTPTV to 1.
 */

#define RTPTV 0


class IPNetwork : public Network {
    public:
	~IPNetwork();
	virtual int command(int argc, const char*const* argv);
	virtual void reset();
    protected:
#ifdef IP_ADD_SOURCE_MEMBERSHIP
        int open(u_int32_t addr, u_int32_t s_addr, int sport, 
		 int rport, int ttl);
        int open(u_int32_t addr, u_int32_t s_addr, int port, int ttl);
        u_int32_t s_addr_;
#else
	int open(u_int32_t addr, int sport, int rport, int ttl);
#endif  /* IP_ADD_SOURCE_MEMBERSHIP */
        int open(u_int32_t addr, int port, int ttl);
	int open(int port);
	int close();
	void localname(sockaddr_in*);
	int openssock(u_int32_t addr, u_short port, int ttl);
#ifdef IP_ADD_SOURCE_MEMBERSHIP
	int openrsock(u_int32_t addr, u_int32_t s_addr, u_short port,
		      const struct sockaddr_in& local);
#else
        int openrsock(u_int32_t addr, u_short port,
		      const struct sockaddr_in& local);    
#endif
	int add_membership();
	void drop_membership();
	time_t last_reset_;
        int recv_bufset(int size);
        void recv_flush();
};

static class IPNetworkClass : public TclClass {
    public:
	IPNetworkClass() : TclClass("Network") {}
        TclObject* create(int, const char*const*) {
	  // vic array: _o32, _o32, Network, create-shadow
		return (new IPNetwork);
	}
} nm_ip;

int IPNetwork::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc == 2) {
		if (strcmp(argv[1], "close") == 0) {
			close();
			return (TCL_OK);
		}
		char* cp = tcl.result();
		if (strcmp(argv[1], "addr") == 0) {
			strcpy(cp, intoa(addr_));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "interface") == 0) {
			strcpy(cp, intoa(local_));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "sport") == 0) {
			sprintf(cp, "%d", sport_);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "localsport") == 0) {
			sprintf(cp, "%d", localsport_);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "rport") == 0) {
			sprintf(cp, "%d", rport_);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ttl") == 0) {
			sprintf(cp, "%d", ttl_);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ismulticast") == 0) {
			tcl.result(IN_CLASSD(ntohl(addr_))? "1" : "0");
			return (TCL_OK);
		}
		if (strcmp(argv[1], "add-membership") == 0) {
			(void)add_membership();
			return (TCL_OK);
		}
		if (strcmp(argv[1], "drop-membership") == 0) {
			drop_membership();
			return (TCL_OK);
		}
		if (strcmp(argv[1], "recv_flush") == 0) {
		        recv_flush();
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(argv[1], "recv_bufset") == 0) {
		        int size = atoi(argv[2]);
			tcl.result(Tcl_NewIntObj(recv_bufset(size)));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "open") == 0) {
			int port = atoi(argv[2]);
			if (open(port) < 0)
				tcl.result("0");
			else
				tcl.result("1");
			return (TCL_OK);
		}

		if (strcmp(argv[1], "loopback") == 0) {
			char c = atoi(argv[2]);
			/*
			 * Note that some network stacks don't allow
			 * loopback to be disabled, but that's okay
			 * because looped-back packets are filtered out
			 * on the recv path anyway.
			 */
			(void)setsockopt(ssock_, IPPROTO_IP, IP_MULTICAST_LOOP,
					 &c, 1);
			return (TCL_OK);
		}

		if (strcmp(argv[1], "ttl") == 0) {
#ifdef WIN32
			u_int t;
#else
			u_char t;
#endif
			int ttl = atoi(argv[2]);
			t = (ttl > 255) ? 255 : (ttl < 0) ? 0 : ttl;
			setsockopt(ssock_, IPPROTO_IP, IP_MULTICAST_TTL,
				   (char*)&t, sizeof(t));
			ttl_ = t;
			return (TCL_OK);
		}
	} else if (argc == 5) {
		if (strcmp(argv[1], "open") == 0) {
#ifdef IP_ADD_SOURCE_MEMBERSHIP
			u_int32_t s_addr, g_addr;
			int type = LookupSGAddr(argv[2], &g_addr, &s_addr);
			int port = atoi(argv[3]);
			int ttl = atoi(argv[4]);

			if ((type == 0) && (!IN_CLASSD_SSM(ntohl(g_addr)))) {
				fprintf(stderr, "Error: group not in ssm space\n");
				exit (1);
			} 
				
			if ((type == 1) && (IN_CLASSD_SSM(ntohl(g_addr)))) {
				fprintf(stderr, "Error: group in ssm space, specify address source,group\n");
				exit (1);
			} 

			if (open(g_addr, s_addr, port, ttl) < 0)
				tcl.result("0");
			else
				tcl.result("1");
			return (TCL_OK);
#else
			u_int32_t addr = LookupHostAddr(argv[2]);
			int port = atoi(argv[3]);
			int ttl = atoi(argv[4]);
			if (open(addr, port, ttl) < 0)
				tcl.result("0");
			else
				tcl.result("1");
			return (TCL_OK);
#endif
		}
	} else if (argc == 6) {
		if (strcmp(argv[1], "open") == 0) {
#ifdef IP_ADD_SOURCE_MEMBERSHIP
		    u_int32_t s_addr, g_addr;
		    int type = LookupSGAddr(argv[2], &g_addr, &s_addr);
		    int sport = atoi(argv[3]);
		    int rport = atoi(argv[4]);
		    int ttl = atoi(argv[5]);

		    if (type == 0 && !IN_CLASSD_SSM(ntohl(g_addr))) {
			tcl.resultf("group %s not in ssm", intoa(g_addr));
			return (TCL_ERROR);
		    }
		    if (type == 1 && IN_CLASSD_SSM(ntohl(g_addr))) {
			tcl.resultf("group %s in ssm address, need src addr",
				    intoa(g_addr));
			return (TCL_ERROR);
		    }
		    
		    if (open(g_addr, s_addr, sport, rport, ttl) < 0)
			tcl.result("0");
		    else
			tcl.result("1");
		    return (TCL_OK);
#else
		    u_int32_t addr = LookupHostAddr(argv[2]);
		    int sport = atoi(argv[3]);
		    int rport = atoi(argv[4]);
		    int ttl = atoi(argv[5]);
		    if (open(addr, sport, rport, ttl) < 0)
			tcl.result("0");
		    else
			tcl.result("1");
		    return (TCL_OK);
#endif
		}
	}
	return (Network::command(argc, argv));
}

#ifdef IP_ADD_SOURCE_MEMBERSHIP
int IPNetwork::open(u_int32_t addr, u_int32_t s_addr, int port, int ttl)
{
        return (open(addr, s_addr, port, port, ttl));
}
#endif

int IPNetwork::open(u_int32_t addr, int port, int ttl)
{
	return (open(addr, port, port, ttl));
}

#ifdef IP_ADD_SOURCE_MEMBERSHIP
int IPNetwork::open(u_int32_t addr, u_int32_t s_addr, int sport, int rport, int ttl)
#else
int IPNetwork::open(u_int32_t addr, int sport, int rport, int ttl)
#endif
{
	addr_ = addr;
#ifdef IP_ADD_SOURCE_MEMBERSHIP
	s_addr_ = s_addr; 
#endif
	sport_ = sport;
	rport_ = rport;
	ttl_ = ttl;

	ssock_ = openssock(addr, sport, ttl);
	if (ssock_ < 0)
		return (-1);

	/*
	 * Open the receive-side socket if requested.
	 */
	if (rport != 0) {
		if (add_membership() < 0)
			return (-1);
	}
	last_reset_ = 0;
	return (0);
}

/* UDP unicast server */
int IPNetwork::open(int port)
{
	int fd;
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		perror("IPNetwork::open socket");
		return (-1);
	}
	addr_ = INADDR_ANY;
	rport_ = port;
	ttl_ = 0;

	int on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on,
			sizeof(on)) < 0) {
		perror("SO_REUSEADDR");
	}
#ifdef SO_REUSEPORT
	on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEPORT, (char *)&on,
		       sizeof(on)) < 0) {
		perror("SO_REUSEPORT");
		exit(1);
	}
#endif
	struct sockaddr_in saddr;
	bzero((char*)&saddr, sizeof(saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_addr.s_addr = INADDR_ANY;
	saddr.sin_port = htons(port);
	if (::bind(fd, (struct sockaddr*)&saddr, sizeof(saddr)) < 0) {
		perror("bind");
		return(-1);
	}
	if (port == 0) {
		bzero((char*)&saddr, sizeof(saddr));
		socklen_t slen = sizeof(saddr);
		if (getsockname(fd, (struct sockaddr*)&saddr, &slen) < 0) {
			perror("getsockname");
		} else {
			rport_ = ntohs(saddr.sin_port);
		}
	}
	rsock_ = fd;
	last_reset_ = 0;
	return (0);
}

IPNetwork::~IPNetwork()
{
	close();
}

int IPNetwork::close()
{
	if (ssock_ >= 0) {
		::close(ssock_);
		ssock_ = -1;
	}
	if (rsock_ >= 0) {
		::close(rsock_);
		rsock_ = -1;
	}
	return (0);
}

/*
 * Control group membership by closing the receive socket.
 * There are too many potential problems across different
 * systems in using the add/drop membership ioctls.
 * (besides, group membership should be implicit in the
 * socket multicast address binding and we should leave
 * the igmp group by closing the socket anyway).
 */
void IPNetwork::drop_membership()
{
	::close(rsock_);
	rsock_ = -1;
}

int IPNetwork::add_membership()
{
	if (rsock_ >= 0)
		return (0);

	/*
	 * Connecting the send socket also bound the local address.
	 * On a multihomed host we need to bind the receive socket
	 * to the same local address the kernel has chosen to send on.
	 */
	sockaddr_in local;
	localname(&local);
#ifdef IP_ADD_SOURCE_MEMBERSHIP
	rsock_ = openrsock(addr_, s_addr_, rport_, local);
#else
	rsock_ = openrsock(addr_, rport_, local);
#endif /* IP_ADD_SOURCE_MEMBERSHIP */
	if (rsock_ < 0) {
		if (ssock_ >= 0)
			(void)::close(ssock_);
		return (-1);
	}
	local_ = local.sin_addr.s_addr;
#if defined(sun) && defined(__svr4__)
	/*
	 * gethostname on solaris prior to 2.6 always returns 0 for
	 * udp sockets.  do a horrible kluge that often fails on
	 * multihomed hosts to get the local address (we don't use
	 * this to open the recv sock, only for the 'interface'
	 * tcl command).
	 */
	if (local_ == 0) {
		char myhostname[1024];
		int error;

		error = sysinfo(SI_HOSTNAME, myhostname, sizeof(myhostname));
		if (error == -1) {
			perror("Getting my hostname");
			exit(-1);
		}
		local_ = LookupHostAddr(myhostname);
	}
#endif
	lport_ = ntohs(local.sin_port);

	return (0);
}

void IPNetwork::localname(sockaddr_in* p)
{
	memset((char *)p, 0, sizeof(*p));
	p->sin_family = AF_INET;
	socklen_t len = sizeof(*p);
	if (getsockname(ssock_, (struct sockaddr *)p, &len) < 0) {
		perror("getsockname");
		p->sin_addr.s_addr = 0;
		p->sin_port = 0;
	}
#ifdef WIN32
	if (p->sin_addr.s_addr == 0) {
		char hostname[80];
		struct hostent *hp;

		if (gethostname(hostname, sizeof(hostname)) >= 0) {
			if ((hp = gethostbyname(hostname)) >= 0) {
				p->sin_addr.s_addr = ((struct in_addr *)hp->h_addr)->s_addr;
			}
		}
	}
#endif
}

void IPNetwork::reset()
{
	time_t t = time(0);
	int d = int(t - last_reset_);
	if (d > 3) {
		last_reset_ = t;
		if (ssock_ >= 0) {
			(void)::close(ssock_);
			ssock_ = openssock(addr_, sport_, ttl_);
		}
	}
}

#ifdef IP_ADD_SOURCE_MEMBERSHIP 
int IPNetwork::openrsock(u_int32_t addr, u_int32_t s_addr, u_short port,
			    const struct sockaddr_in& local)
#else
int IPNetwork::openrsock(u_int32_t addr, u_short port,
			    const struct sockaddr_in& local)
#endif /* IP_ADD_SOURCE_MEMBERSHIP */
{
	int fd;
	struct sockaddr_in sin;

	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (fd < 0) {
		perror("IPNetwork::openrsock socket");
		exit(1);
	}

	nonblock(fd);

	int on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on,
			sizeof(on)) < 0) {
		perror("SO_REUSEADDR");
	}
#ifdef SO_REUSEPORT
	on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEPORT, (char *)&on,
		       sizeof(on)) < 0) {
		perror("SO_REUSEPORT");
		exit(1);
	}
#endif

	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = htons(port);

#ifdef IP_ADD_MEMBERSHIP
	if (IN_CLASSD(ntohl(addr))) {
		/*
		 * Try to bind the multicast address as the socket
		 * dest address.  On many systems this won't work
		 * so fall back to a destination of INADDR_ANY if
		 * the first bind fails.
		 */
		sin.sin_addr.s_addr = addr;
		if (::bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
			sin.sin_addr.s_addr = INADDR_ANY;
			if (::bind(fd, (struct sockaddr*)&sin, sizeof(sin)) < 0) {
				perror("bind");
				return (-1);
			}
		}
		/*
		 * FIXME This is bogus multicast setup that really
		 * shouldn't have to be done (group membership should be
		 * implicit in the IP class D address, route should contain
		 * ttl & no loopback flag, etc.).  Steve Deering has promised
		 * to fix this for the 4.4bsd release.  We're all waiting
		 * with bated breath.
		 */
		struct ip_mreq mr;
#ifdef IP_ADD_SOURCE_MEMBERSHIP
		struct ip_mreq_source mrs;

		if (IN_CLASSD_SSM(ntohl(addr))) {

			mrs.imr_sourceaddr.s_addr = s_addr;
			mrs.imr_multiaddr.s_addr = addr;
			mrs.imr_interface.s_addr = INADDR_ANY;
			if (setsockopt(fd, IPPROTO_IP, IP_ADD_SOURCE_MEMBERSHIP,
					(char*)&mrs, sizeof(mrs)) < 0) {
				perror("IP_ADD_SOURCE_MEMBERSHIP");
				exit (1);
			}
		} else
			
#endif /* IP_ADD_SOURCE_MEMBERSHIP */
		{
		        mr.imr_multiaddr.s_addr = addr;
		        mr.imr_interface.s_addr = INADDR_ANY;
		        if (setsockopt(fd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
			              (char *)&mr, sizeof(mr)) < 0) {
			       perror("IP_ADD_MEMBERSHIP");
			       return (-1);
			}
		}
	} else
#endif /* IP_ADD_MEMBERSHIP */
	{
		/*
		 * bind the local host's address to this socket.  If that
		 * fails, another vic probably has the addresses bound so
		 * just exit.
		 */
		sin.sin_addr.s_addr = local.sin_addr.s_addr;
		if (::bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
			perror("bind");
			return (-1);
		}

		/*
		 * MS Windows currently doesn't compy with the Internet Host
		 * Requirements standard (RFC-1122) and won't let us include
		 * the source address in the receive socket demux state.
		 * (The consequence of this is that all conversations have
		 * to be assigned a unique local port so the "vat side
		 * conversations" --- initiated by middle-clicking on
		 * the site name --- doesn't work under windows.)
		 */
#ifndef WIN32
		/*
		 * (try to) connect the foreign host's address to this socket.
		 */
#if RTPTV
		/*
		 * For some unknown reason, a unicast connection with rtptv
		 * doesn't work unless you specify the port number.  However,
		 * this breaks compatibility with all other vics.  Therefore,
		 * we're not enabling this code by default.  --LL
		 */
		sin.sin_port = htons(port);
		sin.sin_addr.s_addr = addr;
		if (connect(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
			perror("connect");
			return(-1);
		}
#else
		sin.sin_port = 0;
		sin.sin_addr.s_addr = addr;
		connect(fd, (struct sockaddr *)&sin, sizeof(sin));
#endif
#endif
	}

	/*
	 * FIXME don't need this for the session socket.
	 */
	int bufsize = 80 * 1024;
	if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (char *)&bufsize,
			sizeof(bufsize)) < 0) {
		bufsize = 32 * 1024;
		if (setsockopt(fd, SOL_SOCKET, SO_RCVBUF, (char *)&bufsize,
				sizeof(bufsize)) < 0)
			perror("SO_RCVBUF");
	}
	return (fd);
}

int IPNetwork::openssock(u_int32_t addr, u_short port, int ttl)
{
	int fd;
	struct sockaddr_in sin;

	fd = socket(AF_INET, SOCK_DGRAM, 0);
#ifdef WIN32
	if (fd == INVALID_SOCKET) {
	    	int err = WSAGetLastError();
		fprintf(stderr, "IPNetwork::openssock error : %d\n", err);
		exit(1);
	}
#else
	if (fd < 0) {
		perror("IPNetwork::openssock socket");
		exit(1);
	}
#endif

	nonblock(fd);

#if RTPTV
	int on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on,
			sizeof(on)) < 0) {
		perror("SO_REUSEADDR");
	}
#ifdef SO_REUSEPORT
	on = 1;
	if (setsockopt(fd, SOL_SOCKET, SO_REUSEPORT, (char *)&on,
		       sizeof(on)) < 0) {
		perror("SO_REUSEPORT");
		exit(1);
	}
#endif
#endif

	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
#if RTPTV
	sin.sin_port = htons(port);
#else
	sin.sin_port = 0;
#endif
	sin.sin_addr.s_addr = INADDR_ANY;
	if (::bind(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		perror("bind");
		return(-1);
	}

	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = htons(port);
	sin.sin_addr.s_addr = addr;
	if (connect(fd, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
		perror("connect");
		return(-1);
	}

	socklen_t len = sizeof(sin);

	if (getsockname(fd, (struct sockaddr *)&sin, &len) < 0) {
#ifdef WIN32
	    	int err = WSAGetLastError();
		fprintf(stderr, "IPNetwork::openssock error in getsockname : %d\n",
			err);
		return (1);
#else
		perror("getsockname");
		exit(-1);
#endif
	} else {
	    	localsport_ = ntohs(sin.sin_port);
	}

	if (IN_CLASSD(ntohl(addr))) {
#ifdef IP_ADD_MEMBERSHIP
		char c;
		/*
		 * Turn off loopback by default.
		 * Note that some network stacks don't allow
		 * loopback to be disabled, but that's okay
		 * because looped-back packets are filtered out
		 * on the recv path anyway.
		 */
		c = 0;
		(void)setsockopt(fd, IPPROTO_IP, IP_MULTICAST_LOOP, &c, 1);

		/* set the multicast TTL */
#ifdef WIN32
		u_int t;
#else
		u_char t;
#endif
		t = (ttl > 255) ? 255 : (ttl < 0) ? 0 : ttl;
		if (setsockopt(fd, IPPROTO_IP, IP_MULTICAST_TTL,
			       (char*)&t, sizeof(t)) < 0) {
			perror("IP_MULTICAST_TTL");
			exit(1);
		}
#else
		fprintf(stderr, "\
not compiled with support for IP multicast\n\
you must specify a unicast destination\n");
		exit(1);
#endif
	}

	/*
	 * FIXME don't need this for the session socket.
	 */
	int bufsize = 80 * 1024;
	if (setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (char *)&bufsize,
		       sizeof(bufsize)) < 0) {
		bufsize = 48 * 1024;
		if (setsockopt(fd, SOL_SOCKET, SO_SNDBUF, (char *)&bufsize,
			       sizeof(bufsize)) < 0)
			perror("SO_SNDBUF");
	}
	return (fd);
}

int IPNetwork::recv_bufset(int size) {
  int old_size;
  socklen_t option_length;

  if (rsock_ < 0) {
    return 0;
  }


  getsockopt(rsock_, SOL_SOCKET, SO_RCVBUF, (char *) &old_size, &option_length);
  option_length = sizeof(int);

  setsockopt(rsock_, SOL_SOCKET, SO_RCVBUF, (char *) &size, option_length);

  return old_size;
}

void IPNetwork::recv_flush() {
  char dummy_buffer[4000];

  if (rsock_ < 0) {
    return;
  }

  int cc;
  int tot = 0;
  int num_pkt_read = 0;
  int rbuf_size;
  socklen_t option_length = sizeof(rbuf_size);

  getsockopt(rsock_, SOL_SOCKET, SO_RCVBUF, (char *) &rbuf_size, &option_length);

  while ((cc = ::recvfrom(rsock_, dummy_buffer, 4000, 0, 0, 0)) > 0) {
    tot += cc;
    num_pkt_read++;
    if (tot >= rbuf_size) {
      break;
    }
  }
  return;
}
