/*
 * rpi.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

//#ifndef lint
//static const char *rcsid = "@(#) $Header: /usr/mash/src/repository/mash/mash-1/mplug/rpi.cc,v 1.14 2002/02/03 04:13:23 lim Exp $";
//#endif


#include "rpi.h"
#include <sys/types.h>
#include <string.h>
#include <errno.h>
#include "net.h"


timeval operator - (timeval tv1, timeval tv2)
{
	timeval ret;

	if (tv2.tv_usec > tv1.tv_usec) {
		tv1.tv_usec += 1000000;
		tv2.tv_sec += 1;
	}
	ret.tv_sec  = tv1.tv_sec  - tv2.tv_sec;
	ret.tv_usec = tv1.tv_usec - tv2.tv_usec;
	return ret;
}


RPI::ArgArray::ArgArray()
	: argc_(0), argv_(NULL), maxArgc_(0)
{
	maxArgc_ = 4;
	argv_ = new char * [maxArgc_];
	argv_[0] = NULL;
}


RPI::ArgArray::~ArgArray()
{
	for (int i=0; i<argc_; i++) {
		delete [] argv_[i];
	}

	delete [] argv_;
}


void
RPI::ArgArray::GrowToAtLeast(int newArgc)
{
	if (newArgc > maxArgc_) {
		while (newArgc > maxArgc_) maxArgc_ *= 2;

		char **tmpArgv = new char *[maxArgc_];
		memset(tmpArgv, 0, maxArgc_ * sizeof(char*));
		for (int i=0; i < argc_; i++)
			tmpArgv[i] = argv_[i];
		delete [] argv_;

		argv_ = tmpArgv;
	}
}



void
RPI::ArgArray::Insert(const char *list, int pos)
{
	const char *end, *start, *save=list;
	char **newArg;
	int totArgs=0;
	NPBool hasEscape=FALSE, quoted=FALSE;

	LOG(("Inside ArgArray::Insert(\"%s\", %d)\n", list, pos));

	/*
	 * perform two passes through the string
	 * on the first pass (pass=0), count the number of args in the string
	 * on the second pass (pass=1), fill in argv with the new arguments
	 */
	for (int pass=0; pass <2; pass++) {
		list = save;
		while (*list!='\0') {
			hasEscape = FALSE;

			// skip leading white space
			while (*list==' ' || *list=='\t') list++;

			// skip to end of word
			if (*list=='"') {
				quoted = TRUE;
				list++;
			} else {
				quoted = FALSE;
			}

			start = list;
			while (quoted ?
			       (*list != '"' && *list != '\0') :
			       (*list != ' ' && *list != '\t' &&
				*list != '"' && *list != '\0') ) {
				if (*list == '\\' && *(list+1) == '"') {
					hasEscape = TRUE;
					list += 2;
				} else {
					list++;
				}
			}
			end = list;
			if (quoted && *list=='"') list++;

			// got the word
			if (pass==0) {
				totArgs++;
			} else {
				// insert the word into the arglist
				*newArg = new char [end-start+1];

				if (hasEscape) {
					char *ptr = *newArg;
					while (start!=end) {
						if (*start=='\\' &&
						    *(start+1)=='"') {
							start++;
						}
						*ptr++ = *start++;
					}

					*ptr = '\0';
				} else {
					strncpy(*newArg, start, end-start);
					(*newArg)[end-start] = '\0';
				}
				newArg++;
			}
		}

		if (pass==0) {
			// we have counted up all the arguments
			// create space for the new args

			int newArgc = argc_ + totArgs + 1;
			if (newArgc > maxArgc_) GrowToAtLeast(newArgc);

			if (pos < 0) pos = 0;
			if (pos > argc_) pos = argc_;

			for (int i=argc_-1; i >= pos; i--) {
				argv_[i+totArgs] = argv_[i];
			}

			newArg = argv_ + pos;
			argc_ += totArgs;
		}
	}

	argv_[argc_] = NULL;
	LOG(("Done with ArgArray::Insert (argc=%d, maxArgc=%d)\n",
	     argc_, maxArgc_));
}


void
RPI::ArgArray::CreateCmdline(char *&cmdline)
{
	int len=0, i;
	NPBool needQuotes=FALSE;
	char *ptr;

	for (i=0; i < argc_; i++) {
		len += 3 + strlen(argv_[i]);
	}

	cmdline = new char[len+1];
	*cmdline = '\0';
	for (i=0, len=0; i < argc_; i++) {
		// search for special chars in the argument
		// FIXME: We are ignoring the presence of quote characters
		// in the arguments for now.
		needQuotes = FALSE;
		for (ptr=argv_[i]; *ptr!='\0'; ptr++) {
			if (*ptr==' ' || *ptr=='\t') {
				needQuotes = TRUE;
				break;
			}
		}

		if (needQuotes) {
			sprintf(cmdline + len, "\"%s\" ", argv_[i]);
		}
		else {
			sprintf(cmdline + len, "%s ", argv_[i]);
		}
		len += strlen(cmdline+len);
	}

	// remove the trailing space
	*(cmdline + len - 1) = '\0';
}


/*void
RPI::ArgArray::Output(FILE *output)
{
	for (int i=0; i<argc_; i++) {
		char tmp[80];
		sprintf(tmp, "argc: %d, %d: %s\n", argc_, i, argv_[i]);
		LOG(tmp);
		if (strchr(argv_[i], ' ')!=NULL ||
		    strchr(argv_[i], '\t')!=NULL) {
			fprintf(output, "\"%s\" ", argv_[i]);
		} else {
			fprintf(output, "\"%s\" ", argv_[i]);
		}
	}
	fprintf(output, "\n");
}*/



RPI::~RPI()
{
	if (sock_!=-1) closesocket(sock_);
}


NPBool
RPI::NewProcess(RPI::ArgArray *args, const char *stdoutFilename,
		const char *stderrFilename)
{
	// create the server socket
	LOG(("Trying to create server socket\n"));
	if (CreateServer()==FALSE) return FALSE;
	LOG(("Successfully created server socket\n"));

	{
		// add the server port to the arguments
		char buf[80];
		int port = ServerPort();
		if (port==-1) return FALSE;
		sprintf(buf, "-plugin %d", port);
		args->Append(buf);
	}

	LOG(("Trying to fork a new process\n"));
	if (Fork(args, stdoutFilename, stderrFilename)==FALSE) return FALSE;
	LOG(("Successfully forked new process\n"));

	// wait for the real child to connect to the server socket
	return AcceptConnection();
}



void
RPI::DestroyProcess()
{
	char dummy[80];
	fd_set readfds;
	if (sock_==-1) return;

	// notify the plugin process
	Notify("cleanup");

	// wait for the process to close its connection and exit
	timeval tot = { DEFAULT_SELECT_TIMEOUT, 0 };
	timeval tv, diff, start, end;
	int retval;

	while (tot.tv_sec > 0 || (tot.tv_sec==0 && tot.tv_usec > 0)) {
		FD_ZERO(&readfds);
		FD_SET(sock_, &readfds);
		tv = tot;

		::MPlug_GetCurrentTime(&start);
		retval = select(sock_+1, &readfds, NULL, NULL, &tv);
		::MPlug_GetCurrentTime(&end);

		if (retval==SOCKET_ERROR && !SOCK_IS_INTR) {
			break;
		}

		if (retval==0) {
			break;
		}

		// try to read data from the socket
		int rcvd = recv(sock_, dummy, sizeof(dummy), 0);
		if (rcvd <= 0 || rcvd==SOCKET_ERROR) {
			// the socket has been closed
			break;
		}

		diff = end - start;
		tot = tot - diff;
	}

	closesocket(sock_);
	sock_ = -1;

	/*
	 * FIXME: On WIN32, we must call CloseHandle on the process's thread,
	 * and its stdin/out/err handles. Look at tclWinPipe.c (~ Line 1280)
	 * for details.
	 *
	 * I haven't put this code in yet, because I'm not sure exactly what
	 * the semantics are...
	 */
}


NPBool
RPI::CreateServer()
{
#ifndef INADDR_LOOPBACK
#define	INADDR_LOOPBACK		0x7f000001	/* 127.0.0.1   */
#endif

	SOCKET sock;
	struct sockaddr_in sin;

	sock = socket(AF_INET, SOCK_STREAM, 0);
	if (sock==INVALID_SOCKET) {
		return FALSE;
	}

	memset((char *)&sin, 0, sizeof(sin));
	sin.sin_family = AF_INET;
	sin.sin_port = htons(0); // pick a random port
	sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
	if (::bind(sock, (struct sockaddr *)&sin, sizeof(sin))==SOCKET_ERROR) {
		::closesocket(sock);
		return FALSE;
	}

	int on=1;
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR,(char*)&on,
		       sizeof(on)) == SOCKET_ERROR) {
		::closesocket(sock);
		return FALSE;
	}

	if (listen(sock, 5) == SOCKET_ERROR) {
		::closesocket(sock);
		return FALSE;
	}

	sock_ = sock;
	return TRUE;
}



int
RPI::ServerPort()
{
	if (sock_==-1) return -1;
	struct sockaddr_in sin;
	socklen_t len=sizeof(sin);
	if (getsockname(sock_, (sockaddr*) &sin,&len)==SOCKET_ERROR) return -1;
	return ntohs(sin.sin_port);
}


NPBool
RPI::AcceptConnection()
{
	fd_set readfds;
	timeval tv = { DEFAULT_SELECT_TIMEOUT, 0 };
	int retval;

	FD_ZERO(&readfds);
	FD_SET(sock_, &readfds);
	do {
		retval = select(sock_+1, &readfds, NULL, NULL, &tv);
	} while (retval==SOCKET_ERROR && SOCK_IS_INTR);

	if (retval <= 0 || retval==SOCKET_ERROR) {
		// timed out, or error occurred
		closesocket(sock_);
		sock_ = -1;
		MPlug_Output("Could not create connection to plugin process\n"
			     "select failed or timed out\n");
		return FALSE;
	}

	sockaddr_in sfrom;
	socklen_t fromlen = sizeof(sfrom);
	SOCKET newSock;
	newSock = ::accept(sock_, (sockaddr*)&sfrom, &fromlen);
	if (newSock==INVALID_SOCKET) {
		closesocket(sock_);
		sock_ = -1;
		MPlug_Output("Could not create connection to plugin process\n"
			     "accept failed\n");
		return FALSE;
	}

	// close the server socket, since the client has connected to us, now
	if (sock_!= -1) closesocket(sock_);
	sock_ = newSock;
	return TRUE;
}


NPBool
RPI::Notify(const char *string)
{
	if (sock_==-1) return TRUE;

	ArgArray args;
	args.Append(string);
	int argc = args.argc(), len;
	const char * const *argv = args.argv();
	const char *bytes;

	uint32 tmp = argc;
	if (Send_(&tmp, sizeof(uint32))==FALSE) return FALSE;

	for (int i=0; i < argc; i++) {
		bytes = argv[i];
		len = strlen(bytes);

		tmp = len;
		if (Send_(&tmp, sizeof(uint32))==FALSE) return FALSE;
		if (Send_(bytes,(uint32) len)==FALSE) return FALSE;
	}
	return TRUE;
}


NPBool
RPI::Notify(int numElems, ...)
{
	uint32 tmp = numElems;
	va_list ap;
	int len;
	const char *bytes;

	if (Send_(&tmp, sizeof(uint32))==FALSE) return FALSE;

	va_start(ap, numElems);
	for (int i=0; i < numElems; i++) {
		len   = va_arg(ap, int);
		bytes = va_arg(ap, const char *);
		if (len==-1) len = strlen(bytes);

		tmp = len;
		if (Send_(&tmp, sizeof(uint32))==FALSE) return FALSE;
		if (Send_(bytes,(uint32) len)==FALSE) return FALSE;
	}
	va_end(ap);
	return TRUE;
}


NPBool
RPI::NotifyNextPiece(int len, const void *buffer)
{
	if (len==-1) len = strlen((const char*)buffer);
	uint32 tmp = len;
	if (Send_(&tmp, sizeof(uint32))==FALSE) return FALSE;
	return Send_(buffer, (uint32)len);
}



NPBool
RPI::Send_(const void *bytes, uint32 len)
{
	size_t written = 0, thisTime;
	while (written < len) {
		thisTime = send(sock_, ((const char*)bytes)+written,
				len-written, 0);
		if (thisTime <= 0 && errno != EINTR) {
			// an error occurred
			closesocket(sock_);
			sock_ = -1;
			return FALSE;
		}

		written += thisTime;
	}
	return TRUE;
}
