/*
 * jpegwc.c --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <setjmp.h>
#include "jpegwc.h"
#include "jpeglib.h"

int
jpeg_tran_progressive(char *fnsrc, char *fndst)
{
	struct jpeg_decompress_struct srcinfo;
	struct jpeg_compress_struct dstinfo;
	struct jpeg_error_mgr jsrcerr, jdsterr;

	jvirt_barray_ptr * coef_arrays;

	FILE *fsrc, *fdst;

	/* Open the source and destination files */
	fsrc = fopen(fnsrc, "rb");
	fdst = fopen(fndst, "wb");

	/* Initialize the JPEG decompression object with default error handling. */
	srcinfo.err = jpeg_std_error(&jsrcerr);
	jpeg_create_decompress(&srcinfo);

	/* Initialize the JPEG compression object with default error handling. */
	dstinfo.err = jpeg_std_error(&jdsterr);
	jpeg_create_compress(&dstinfo);

	jsrcerr.trace_level = jdsterr.trace_level;

	srcinfo.mem->max_memory_to_use = dstinfo.mem->max_memory_to_use;

	/* Specify data source for decompression */
	jpeg_stdio_src(&srcinfo, fsrc);

	/* Read file header */
	(void) jpeg_read_header(&srcinfo, TRUE);

	/* Read source file as DCT coefficients */
	coef_arrays = jpeg_read_coefficients(&srcinfo);

	/* Initialize destination compression parameters from source values */
	jpeg_copy_critical_parameters(&srcinfo, &dstinfo);

	/* Create a simple progression sequence */
	jpeg_simple_progression(&dstinfo);

	/* Specify data destination for compression */
	jpeg_stdio_dest(&dstinfo, fdst);

	/* Start compressor (note no image data is actually written here) */
	jpeg_write_coefficients(&dstinfo, coef_arrays);

	/* Finish compression and release memory */
	jpeg_finish_compress(&dstinfo);
	jpeg_destroy_compress(&dstinfo);
	(void) jpeg_finish_decompress(&srcinfo);
	jpeg_destroy_decompress(&srcinfo);

	/* Close the source and destination files */
	fclose(fsrc);
	fclose(fdst);

	return 1;
}


int
jpeg_reduce_progressive(char *fnsrc, char *fndst, int level)
{
	int cur_level, processed_level;
	FILE *fsrc, *fdst;
	char fntmp[80];
	int c;
	int prevc = 0;

	/* Open the source and destination files */
	fsrc = fopen(fnsrc, "rb");
	fdst = fopen(fndst, "wb");

	/* Transcode the source file into progressive format if it isn't already */
	if ((cur_level = jpeg_is_progressive(fsrc)) == 0) {
		sprintf(fntmp, "/tmp/%ld.jpg", getpid());
		fclose(fsrc);
		jpeg_tran_progressive(fnsrc, fntmp);
		fsrc = fopen(fntmp, "rb");

		cur_level = 10;
	}

	/* Check whether the desired reduced level is valid */
	if (cur_level <= level || cur_level <= 1 || level > 10 || level < 1) {
		fprintf(stderr, "jpeg_reduce_progressive: cannot reduce to level %d, current level is %d\n", level, cur_level);
		return 0;
	}

	/* Start reducing process */
	processed_level = 0;
	fseek(fsrc, 0, SEEK_SET);
	fseek(fdst, 0, SEEK_SET);
	while ((c = fgetc(fsrc)) != EOF) {
		if (prevc == 0xFF && c == M_SOS)
			processed_level++;
		if (processed_level > level) {
			fputc(M_EOI, fdst);
			break;
		}
		fputc(c, fdst);
		prevc = c;
	}

	/* Close the source and destination files */
	fclose(fsrc);
	fclose(fdst);

	return level;
}


int
jpeg_getrange_progressive(char *fnsrc, char *fnblks, int start_level, int end_level)
{
	FILE *fsrc, *fblks, *ftmp;
	char fntmp[80];
	int cur_level, processed_level;
	int c;
	int prevc = 0;

	/* Open the source and blocks files */
	fsrc = fopen(fnsrc, "rb");
	fblks = fopen(fnblks, "wb");

	/* Transcode the source file into progressive format if it isn't already */
	cur_level = jpeg_is_progressive(fsrc);
	if (cur_level == 0) {
		sprintf(fntmp, "/tmp/%ld.jpg", getpid());
		fclose(fsrc);
		jpeg_tran_progressive(fnsrc, fntmp);
		fsrc = fopen(fntmp, "rb");

		cur_level = 10;
	}

	/* Check if levels in source image is within the range requested */
	if (cur_level < start_level || cur_level < end_level) {
		fprintf(stderr, "jpeg_getrange_progressive: cannot get the levels %d-%d desired, image has level %d\n", start_level, end_level, cur_level);
		return 0;
	}

	/* Copy the levels from the start level to the end level */
	processed_level = 0;
	fseek(fsrc, 0, SEEK_SET);
	fseek(fblks, 0, SEEK_SET);
	while ((c = fgetc(fsrc)) != EOF) {
		if (prevc == 0xFF && c == M_SOS)
			processed_level++;

		if ((prevc == 0xFF && c == M_EOI) || (processed_level > end_level))
			break;
		else if (processed_level >= start_level &&
			 processed_level <= end_level)
			fputc(c, fblks);

		prevc = c;
	}

	/* Close the source and blocks files */
	fclose(fsrc);
	fclose(fblks);

	return 1;
}


int
jpeg_enhance_progressive(char *fnsrc, char *fndst, char *fnblks, int num_level)
{
	FILE *fsrc, *fdst, *fblks;
	int cur_level, processed_level;
	int c;
	int prevc = 0;

	/* Open the source, destination, and blocks files */
	fsrc = fopen(fnsrc, "rb");
	fdst = fopen(fndst, "wb");
	fblks = fopen(fnblks, "rb");

	/* Check if source image is already full quality */
	cur_level = jpeg_is_progressive(fsrc);
	if (cur_level == 10 || cur_level == 0) {
		fprintf(stderr, "jpeg_enhance_progressive: cannot enhance full quality image\n");
		return 0;
	}

	/* Copy each scan record from source to destination file,
	 * except replace the end-of-image marker with another
	 * start of scan marker
	 */
	processed_level = 0;
	fseek(fsrc, 0, SEEK_SET);
	while ((c = fgetc(fsrc)) != EOF) {
		if (prevc == 0xFF && c == M_SOS)
			processed_level++;
		if (prevc == 0xFF && c == M_EOI)
			break;
		fputc(c, fdst);
		prevc = c;
	}

	fseek(fblks, 0, SEEK_SET);
	while ((c = fgetc(fblks)) != EOF) {
		if (prevc == 0xFF && c == M_SOS)
			processed_level++;
		if (processed_level > cur_level + num_level) {
			fputc(M_EOI, fdst);

			break;
		}
		fputc(c, fdst);
		prevc = c;
	}

	/* Close the source, destination, and blocks files */
	fclose(fsrc);
	fclose(fdst);
	fclose(fblks);

	return processed_level;
}


int
jpeg_is_progressive(FILE *f)
{
	int c;
	int prevc = 0;
	int num_level, is_progressive;

	num_level = 0;
	is_progressive = 0;

	while ((c = fgetc(f)) != EOF) {
		if (prevc == 0xFF && c == M_SOS)
			num_level++;
		else if (prevc == 0xFF && c == M_SOF2)
			is_progressive = 1;
		prevc = c;
	}

	/* Set the file pointer back to beginning of the file */
	fseek(f, 0, SEEK_SET);

	/* returns 0 if the image is not progressively coded,
	 * otherwise, returns the number of level it is encoded
	 */
	return is_progressive ? num_level : 0;
}

/*
int
main (int argc, char **argv)
{
	FILE *f1;
	char *fn1, *fn2, *fn3;
	int i, level, start_level, end_level, result;
	JPEGWC_COMMANDS command;

	fn1 = (char*)malloc(sizeof(char)*80);
	fn2 = (char*)malloc(sizeof(char)*80);
	fn3 = (char*)malloc(sizeof(char)*80);

	for (i = 0; i < argc; i++) {
		if (strcmp(argv[i], "-tran") == 0) {
			command = TRAN;
			fn1 = argv[++i];
			fn2 = argv[++i];
			break;
		}
		else if (strcmp(argv[i], "-reduce") == 0) {
			command = REDUCE;
			fn1 = argv[++i];
			fn2 = argv[++i];
			level = atoi(argv[++i]);
			break;
		}
		else if (strcmp(argv[i], "-enhance") == 0) {
			command = ENHANCE;
			fn1 = argv[++i];
			fn2 = argv[++i];
			fn3 = argv[++i];
			level = atoi(argv[++i]);
			break;
		}
		else if (strcmp(argv[i], "-getrange") == 0) {
			command = GETRANGE;
			fn1 = argv[++i];
			fn2 = argv[++i];
			start_level = atoi(argv[++i]);
			end_level = atoi(argv[++i]);
			break;
		}
		else if (strcmp(argv[i], "-isprogress") == 0) {
			command = ISPROGRESS;
			fn1 = argv[++i];
			f1 = fopen(fn1, "rb");
			break;
		}
	}

	switch (command) {
	case TRAN:
		result = jpeg_tran_progressive(fn1,fn2);
		break;
	case REDUCE:
		result = jpeg_reduce_progressive(fn1,fn2,level);
		break;
	case ENHANCE:
		result = jpeg_enhance_progressive(fn1,fn2,fn3,level);
		break;
	case GETRANGE:
		result = jpeg_getrange_progressive(fn1,fn2,start_level,end_level);
		break;
	case ISPROGRESS:
		result = jpeg_is_progressive(f1);
		printf("jpeg_is_progressive: %d\n", result);
		break;
	}
}
*/
