/*
 * mbv2-cmd.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] = "@(#) $Header: /usr/mash/src/repository/mash/mash-1/mbv2/mbv2-cmd.cc,v 1.13 2002/02/03 03:17:08 lim Exp $";
#endif


#include "mbv2-obj.h"
#include "mbv2-cmd.h"
#include "mbv2-canv.h"
#include "mtrace.h"
#include <assert.h>


MBv2Cmd::~MBv2Cmd()
{
}


MBv2Cmd *
MBv2Cmd::create(MBv2CmdId id, const srm_adu_info *info,
		const Byte *pb, int len)
{
	MBv2Cmd *cmd;
	switch(info->type) {
	case MBv2Create:
		cmd = new MBv2CmdCreate();
		break;
	case MBv2Group:
		cmd = MBv2CmdGroup::create(pb, len);
		if (!cmd) return NULL;
		break;
	case MBv2Delete:
		cmd = new MBv2CmdDelete();
		break;
	case MBv2Move:
		cmd = new MBv2CmdMove();
		break;
	case MBv2Copy:
	case MBv2Undelete:
	case MBv2EditText:
		cmd = new MBv2CmdCopy((MBv2CmdType)info->type);
		break;
	case MBv2Char:
		cmd = new MBv2CmdChar();
		break;
	case MBv2Chars:
		cmd = new MBv2CmdChars();
		break;
	default:
		MTrace(trcMB, ("invalid command type %d (id=%u)", info->type,
			       id));
		return NULL;
	}

	cmd->id_ = id;
	cmd->timestamp_.upper = info->timestamp.upper;
	cmd->timestamp_.lower = info->timestamp.lower;
	if (!cmd->extract(pb, len)) {
		MTrace(trcMB, ("error occurred while extracting command type "
			       "%d (id=%u)", info->type, id));
		delete cmd;
		return NULL;
	}

	return cmd;
}


Bool
MBv2Cmd::execute(MBv2Page* page, const MBv2Time& currentTime,
		 const MBv2Time& targetTime)
{
	MTrace(trcMB|trcVerbose, ("executing cmd %d (ts: %u:%u) at %u:%u "
				  "%u:%u   %d %d", id_,
				  timestamp_.upper, timestamp_.lower,
				  currentTime.upper, currentTime.lower,
				  targetTime.upper, targetTime.lower,
				  active_at(currentTime),
				  active_at(targetTime)));
	if (!active_at(currentTime)) {
		if (active_at(targetTime)) {
			MTrace(trcMB|trcVerbose, ("applying cmd %d", id_));
			return apply(page);
		}
	} else if (!active_at(targetTime)) {
		return reverse(page, targetTime);
	}
	return TRUE;
}


MBv2Cmd *
MBv2CmdGroup::create(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_CmdGroup)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_CmdGroup)));
		return NULL;
	}
	ADU_CmdGroup *adu = (ADU_CmdGroup*) pb;
	switch (ntoh(adu->groupType)) {
	case MBv2Line:
		return new MBv2CmdFreeHand();

	case MBv2Text:
		return new MBv2CmdText();

	default:
		MTrace(trcMB, ("invalid group type %d",ntoh(adu->groupType)));
		return NULL;
	}
}


MBv2Cmd *
MBv2CmdGroup::create(MBv2ItemType type, const MBv2Time &ts,
		     MBv2CmdId start, MBv2CmdId end)
{
	switch (type) {
	case MBv2Line:
		return new MBv2CmdFreeHand(ts, start, end);

	case MBv2Text:
		return new MBv2CmdText(ts, start, end);

	default:
		MTrace(trcMB, ("invalid group type %d", type));
		return NULL;
	}
}


Bool
MBv2CmdGroup::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_CmdGroup)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_CmdGroup)));
		return FALSE;
	}
	ADU_CmdGroup *adu = (ADU_CmdGroup*) pb;

	groupType_ = (MBv2ItemType) ntoh(adu->groupType);
	start_ = ntoh(adu->startCmd);
	end_   = ntoh(adu->endCmd);
	return TRUE;
}


Byte *
MBv2CmdGroup::packetize(Byte *pb)
{
	ADU_CmdGroup *adu = (ADU_CmdGroup*) pb;
	adu->groupType = hton((u_int16_t)groupType_);
	adu->startCmd  = hton(start_);
	adu->endCmd    = hton(end_);
	return (Byte*)(adu+1);
}


Bool
MBv2CmdGroup::incomplete(MBv2Page *page, MBv2Dependency *dep)
{
	assert (end_!=id_);
	if (isComplete_) return 0;

	MBv2Cmd *cmd;
	for (MBv2CmdId i=start_; i <= end_; i++) {
		cmd = page->cmd(i);
		if (!cmd) {
			if (dep) {
				dep->cid = page->cid();
				dep->cmd = i;
				dep->degree = 1;
			}
			return 1;
		}

		if (cmd->incomplete(page, dep)) {
			if (dep) dep->degree++;
			return 1;
		}
	}

	isComplete_ = 1;
	return 0;
}


MBv2Item *
MBv2CmdGroup::make_copy(MBv2Page* /*page*/)
{
	return item_->make_copy();
}


Bool
MBv2CmdFreeHand::apply(MBv2Page *page)
{
	MTrace(trcMB, ("DDD applying freehand for %u", id_));
	if (page->get_canvas_item(id_) != MBv2InvalidCanvId) {
		// this command seems to already have been applied
		MTrace(trcMB, ("cmd %u seems to have already been applied",
			       id_));
		return FALSE;
	}

	if (end_ <= start_) {
		MTrace(trcMB, ("cannot apply freehand cmd (id=%u); start (%u) "
			       "is not less than end (%u)", id_, start_,end_));
		return FALSE;
	}

	MBv2Canvas *canv = page->canvas();
	if (!canv) {
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	int numPoints=1, itemPoints;
	MBv2CmdId i;
	MBv2Cmd *cmd;
	MBv2Item *item;

	// loop thru all the items to check if the items are sane
	// also count the total number of points in the line
	for (i=start_; i<=end_; i++) {
		cmd = page->cmd(i);
		if (!cmd) {
			MTrace(trcMB, ("could not find cmd for id %u", i));
			return FALSE;
		}
		if (cmd->type()!=MBv2Create) {
			MTrace(trcMB, ("invalid cmd type: %d",
				       cmd->type()));
			return FALSE;
		}
		item = ((MBv2CmdCreate*)cmd)->item();
		if (!item || item->type()!=MBv2Line) {
			if (item) {
				MTrace(trcMB, ("invalid item type: %d",
					       item->type()));
			} else {
				MTrace(trcMB,("no item associated with cmd %u",
					      i));
			}
			return FALSE;
		}
		itemPoints = item->num_points();
		if (itemPoints < 2) {
			MTrace(trcMB, ("invalid number of points: %d",
				       itemPoints));
			return FALSE;
		}
		numPoints += (itemPoints - 1);
		// FIXME: maybe we should check if the line segments are
		// continuous or not
	} // for (i=start_; ...

	MBv2Point *points = new MBv2Point [numPoints], *p, *ipoints, *ip;
	MBv2CmdCreate *ccmd;
	MBv2CanvId canvId;

	// create an item object for this group
	item_ = ((MBv2CmdCreate*)page->cmd(start_))->item()->make_copy();
	// replace the points array with the real one
	item_->points(points, numPoints);

	for (p=points, i=start_; i<=end_; i++) {
		ccmd = (MBv2CmdCreate*) page->cmd(i);

		// copy the coordinates from the original item into the
		// new one
		item = ccmd->item();
		itemPoints = item->num_points();
		ipoints = item->points();
		for (ip=(p==points ? ipoints : (ipoints+1));
		     ip < ipoints + itemPoints; ip++) {
			*p++ = *ip;
		}

		// delete the object from the canvas
		canvId = page->get_canvas_item(i);
		if (canvId != MBv2InvalidCanvId) {
			if (!canv->delete_item(canvId)) {
				MTrace(trcMB, ("could not delete canvid %u",
					       canvId));
			}
			// remove the mapping from the cmd to the canv item
			page->forget_canvas_item(i);
		}

		// delete the associated MBv2Item object
		// FIXME: this may screw things up for replay
		//YYY: ccmd->delete_item();
	}

	canvId = canv->create_item(item_, page->is_local());
	if (canvId==MBv2InvalidCanvId) return FALSE;
	page->associate_canvas_item(id_, canvId);
	page->tag(this, MBv2InvalidCmdId); // create a new tag for this command
	MTrace(trcMB, ("DDD done applying freehand for %u", id_));
	return TRUE;
}


Bool
MBv2CmdText::apply(MBv2Page *page)
{
	if (page->get_canvas_item(id_) != MBv2InvalidCanvId) {
		// this command seems to already have been applied
		MTrace(trcMB, ("cmd %u seems to have already been applied",
			       id_));
		return FALSE;
	}

	if (end_ <= start_) {
		MTrace(trcMB, ("cannot apply text cmd (id=%u); start (%u) "
			       "is not less than end (%u)", id_, start_,end_));
		return FALSE;
	}

	MBv2Canvas *canv = page->canvas();
	if (!canv) {
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	MBv2CmdId i, tag=MBv2InvalidCmdId, start=MBv2InvalidCmdId;
	MBv2Cmd *cmd;
	MBv2CmdType type;
	MBv2Item *item=NULL;
	int itemPoints;

	item_ = NULL;

	// loop thru all the items to check if the items are sane
	for (i=start_; i<=end_; i++) {
		cmd = page->cmd(i);
		if (!cmd) {
			MTrace(trcMB, ("could not find cmd for id %u", i));
			return FALSE;
		}
		type = cmd->type();
		if ( (i==start_ && type!=MBv2Create && type!=MBv2EditText) ||
			(i > start_ && type!=MBv2Char && type!=MBv2Chars) ) {
			MTrace(trcMB, ("invalid cmd type: %d",
				       cmd->type()));
			return FALSE;
		}
		if (i > start_) {
			if (type==MBv2Chars) {
				u_int16_t index =((MBv2CmdChars*)cmd)->index();
				if (index == MBv2CmdChars_ReplaceText) {
					start = i;
				}
			}
			continue;
		}

		// a create or edittext object
		item = MBv2CmdText::item(cmd);
		// create an item object for this group
		item_ = item->make_copy();
		if (!item_ || item_->type()!=MBv2Text) {
			MTrace(trcMB, ("invalid item type: %d",
				       (item_ ? item_->type() : -1)));
			if (item_) delete item_;
			return FALSE;
		}

		itemPoints = item_->num_points();
		if (itemPoints < 1) {
			MTrace(trcMB, ("invalid number of points: %d",
				       itemPoints));
			delete item_;
			return FALSE;
		}

		if (type==MBv2EditText) {
			// an edittext object
			tag  = cmd->tag();
		}

	} // for (i=start_; ...

	// make sure the canvId exists
	MBv2CanvId canvId = page->get_canvas_item(start_);
	if (canvId==MBv2InvalidCanvId) {
		MTrace(trcMB, ("first text item deleted: should not occur"));
		return FALSE;
	}

	// reconstruct the text; if the cmdchars were received out of order
	// the text may be mangled by now (especially if the cmdchars had \b)
	char *text = ((MBv2CmdCharOrChars*)page->cmd(end_))->
		create_text(page, start);

	((MBv2TextItem*)item_)->set_text(text);
	// the text string will be deleted by item_'s destructor

	// reset the canvas text
	canv->set_text(canvId, text);

	// reassociate the canvid
	page->forget_canvas_item(start_);
	page->associate_canvas_item(id_, canvId);
	page->tag(this, tag); /* create a new tag for this command only if it
			       * isn't a group of an EditText cmd */
	return TRUE;
}


Bool
MBv2CmdCreate::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_CmdCreate)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_CmdCreate)));
		return FALSE;
	}
	ADU_CmdCreate *adu = (ADU_CmdCreate *) pb;
	MBv2ItemType itemType = (MBv2ItemType) ntoh(adu->itemType);
	item_ = MBv2Item::create(itemType, pb+sizeof(ADU_CmdCreate),
				 len - sizeof(ADU_CmdCreate));
	if (!item_) return FALSE;
	return TRUE;
}


Byte *
MBv2CmdCreate::packetize(Byte *pb)
{
	ADU_CmdCreate *adu = (ADU_CmdCreate *) pb;
	adu->itemType = hton((u_int32_t)item_->type());
	return item_->packetize((Byte*)(adu+1));
}


Bool
MBv2CmdCreate::apply(MBv2Page* page)
{
	if (page->get_canvas_item(id_) != MBv2InvalidCanvId) {
		// this command seems to already have been applied
		MTrace(trcMB, ("cmd %u seems to have already been applied",
			       id_));
		return FALSE;
	}

	MBv2Canvas *canv = page->canvas();
	if (!canv) {
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	MBv2CanvId canvId = canv->create_item(item_, page->is_local());
	if (canvId==MBv2InvalidCanvId) return FALSE;
	page->associate_canvas_item(id_, canvId);
	page->tag(this, MBv2InvalidCmdId); // create a new tag for this command
	return TRUE;
}


MBv2Item *
MBv2CmdCreate::make_copy(MBv2Page * /*page*/)
{
	return item_->make_copy();
}


Bool
MBv2CmdMove::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_CmdMove)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_CmdMove)));
		return FALSE;
	}
	ADU_CmdMove *adu = (ADU_CmdMove *) pb;
	targetCmd_ = ntoh(adu->targetCmd);
	raiseTS_.upper = raiseTS_.lower = 0;
	dx_ = ntoh(adu->delta.x);
	dy_ = ntoh(adu->delta.y);
	return TRUE;
}


Byte *
MBv2CmdMove::packetize(Byte *pb)
{
	ADU_CmdMove *adu = (ADU_CmdMove *) pb;
	adu->targetCmd = hton(targetCmd_);
	adu->delta.x = hton(dx_);
	adu->delta.y = hton(dy_);
	return (Byte*)(adu+1);
}


Bool
MBv2CmdMove::incomplete(MBv2Page *page, MBv2Dependency *dep)
{
	if (isComplete_) return 0;
	MBv2Cmd *cmd=NULL;
	// only depends on the targetCmd_
	cmd = page->cmd(targetCmd_);
	if (!cmd) {
		if (dep) {
			dep->cid = page->cid();
			dep->cmd = targetCmd_;
			dep->degree = 1;
		}
		return 1;
	}

	if (cmd->incomplete(page, dep)) {
		if (dep) dep->degree++;
		return 1;
	}
	isComplete_ = 1;
	return 0;
}


Bool
MBv2CmdMove::apply(MBv2Page* page)
{
	if (page->get_canvas_item(id_) != MBv2InvalidCanvId) {
		// this command seems to already have been applied
		MTrace(trcMB, ("cmd %u seems to have already been applied",
			       id_));
		return FALSE;
	}

	MBv2Canvas *canv = page->canvas();
	if (!canv) {
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	MBv2CanvId canvId = page->get_canvas_item(targetCmd_);
	if (canvId==MBv2InvalidCanvId) {
		// couldn't find the canv item associated with the target
		// command
		MTrace(trcMB, ("could not find any canv item associated with "
		       "command id %u", targetCmd_));
		return FALSE;
	}

	if (canv->move_item(canvId, dx_, dy_)) {
		page->forget_canvas_item(targetCmd_);
		page->associate_canvas_item(id_, canvId);
	}

	/* associate the tag of the target with this cmd */
	page->tag(this, page->cmd(targetCmd_)->tag());
	return TRUE;
}


const MBv2Time &
MBv2CmdMove::raise_timestamp(MBv2Page *page)
{
	if (raiseTS_.upper==0 && raiseTS_.lower==0) {
		MBv2Cmd *cmd = page->cmd(targetCmd_);
		if (cmd) raiseTS_ = cmd->raise_timestamp(page);
	}
	return raiseTS_;
}


MBv2Item *
MBv2CmdMove::make_copy(MBv2Page *page)
{
	MBv2Item *item = page->cmd(targetCmd_)->make_copy(page);
	if (!item) return item;

	MBv2Point *points=item->points();
	for (MBv2Point* point=points; point < points + (item->num_points());
	     point++) {
		point->x += dx_;
		point->y += dy_;
	}
	return item;
}


Bool
MBv2CmdCopy::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_CmdCopy)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_CmdCopy)));
		return FALSE;
	}
	ADU_CmdCopy *adu = (ADU_CmdCopy *) pb;
	targetCmd_ = ntoh(adu->targetCmd);
	targetCid_ = ntoh(adu->targetCid);
	raiseTS_.upper = raiseTS_.lower = 0;
	return TRUE;
}


Byte *
MBv2CmdCopy::packetize(Byte *pb)
{
	ADU_CmdCopy *adu = (ADU_CmdCopy *) pb;
	adu->targetCmd = hton(targetCmd_);
	adu->targetCid = hton(targetCid_);
	return (Byte*)(adu+1);
}


Bool
MBv2CmdCopy::incomplete(MBv2Page *page, MBv2Dependency *dep)
{
	if (isComplete_) return 0;
	MBv2Cmd *cmd=NULL;
	// depends on targetCid_ and targetCmd_
	MBv2Page *targetPage = page->source()->find_page(targetCid_);
	if (!targetPage) {
		if (dep) {
			dep->cid = targetCid_;
			dep->cmd = targetCmd_;
			dep->degree = 1;
		}
		return 1;
	}

	cmd = targetPage->cmd(targetCmd_);
	if (!cmd) {
		if (dep) {
			dep->cid = targetCid_;
			dep->cmd = targetCmd_;
			dep->degree = 1;
		}
		return 1;
	}

	if (cmd->incomplete(targetPage, dep)) {
		if (dep) dep->degree++;
		return 1;
	}
	isComplete_ = 1;
	return 0;
}


Bool
MBv2CmdCopy::apply(MBv2Page *page)
{
	if (page->get_canvas_item(id_) != MBv2InvalidCanvId) {
		// this command seems to already have been applied
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	MBv2Canvas *canv = page->canvas();
	if (!canv) {
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	MBv2Page *targetPage;
	targetPage = page->source()->find_page(targetCid_);
	if (!targetPage) {
		MTrace(trcMB, ("could not find target page for cid %u",
			       targetCid_));
		return FALSE;
	}

	item_ = targetPage->cmd(targetCmd_)->make_copy(targetPage);
	if (!item_) {
		MTrace(trcMB, ("could not make a copy of item associated with "
			       "cmd %u on page with cid %u", targetCmd_,
			       targetCid_));
		return FALSE;
	}

	if (type()==MBv2EditText) {
		// we must first delete the canvas item associated
		// with the original command

		MBv2Canvas *canv = targetPage->canvas();
		if (!canv) {
			MTrace(trcMB, ("could not find canvas for page with "
				       "cid %u", targetPage->cid()));
			return FALSE;
		}
		MBv2CanvId canvId = targetPage->get_canvas_item(targetCmd_);
		if (canvId==MBv2InvalidCanvId) {
			// couldn't find the canv item associated with the
			// command that needs to be deleted
			MTrace(trcMB, ("could not find any canv item "
				       "associated with command id %u on page "
				       "with cid %u", targetCmd_,
				       targetPage->cid()));
			return FALSE;
		}

		if (!canv->delete_item(canvId)) {
			MTrace(trcMB, ("could not delete canvid %u", canvId));
		}
		// remove the mapping from the cmd to the canv item
		targetPage->forget_canvas_item(targetCmd_);
	}

	MBv2CanvId canvId = canv->create_item(item_, page->is_local());
	if (canvId==MBv2InvalidCanvId) return FALSE;
	page->associate_canvas_item(id_, canvId);
	if (type()==MBv2Copy) {
		// create a new tag for this command
		page->tag(this, MBv2InvalidCmdId);
	} else {
		// undelete or edittext
		/* associate the tag of the target with this cmd */
		page->tag(this, targetPage->cmd(targetCmd_)->tag());
	}
	return TRUE;
}


const MBv2Time &
MBv2CmdCopy::raise_timestamp(MBv2Page *page)
{
	if (cmdType_==MBv2Copy) {
		return MBv2Cmd::raise_timestamp(page);
	}

	// this is an undelete or edittext command
	if (raiseTS_.upper==0 && raiseTS_.lower==0) {
		MBv2Page *targetPage;
		targetPage = page->source()->find_page(targetCid_);
		if (targetPage) {
			MBv2Cmd *cmd = targetPage->cmd(targetCmd_);
			if (cmd) raiseTS_ = cmd->raise_timestamp(targetPage);
		}
	}
	return raiseTS_;
}


MBv2Item *
MBv2CmdCopy::make_copy(MBv2Page* /*page*/)
{
	return item_->make_copy();
}


Bool
MBv2CmdDelete::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_CmdDelete)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_CmdDelete)));
		return FALSE;
	}
	ADU_CmdDelete *adu = (ADU_CmdDelete *) pb;
	targetCmd_ = ntoh(adu->targetCmd);
	return TRUE;
}


Byte *
MBv2CmdDelete::packetize(Byte *pb)
{
	ADU_CmdDelete *adu = (ADU_CmdDelete *) pb;
	adu->targetCmd = hton(targetCmd_);
	return (Byte*)(adu+1);
}


Bool
MBv2CmdDelete::incomplete(MBv2Page *page, MBv2Dependency *dep)
{
	if (isComplete_) return 0;
	MBv2Cmd *cmd=NULL;
	// only depends on the targetCmd_
	cmd = page->cmd(targetCmd_);
	if (!cmd) {
		if (dep) {
			dep->cid = page->cid();
			dep->cmd = targetCmd_;
			dep->degree = 1;
		}
		return 1;
	}

	if (cmd->incomplete(page, dep)) {
		if (dep) dep->degree++;
		return 1;
	}
	isComplete_ = 1;
	return 0;
}


Bool
MBv2CmdDelete::apply(MBv2Page *page)
{
	MBv2Canvas *canv = page->canvas();
	if (!canv) {
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	MBv2CanvId canvId = page->get_canvas_item(targetCmd_);
	if (canvId==MBv2InvalidCanvId) {
		// couldn't find the canv item associated with the
		// command that needs to be deleted
		MTrace(trcMB, ("could not find any canv item "
			       "associated with command id %u",
			       targetCmd_));
		return FALSE;
	}

	if (!canv->delete_item(canvId)) {
		MTrace(trcMB, ("could not delete canvid %u", canvId));
	}
	// remove the mapping from the cmd to the canv item
	page->forget_canvas_item(targetCmd_);
	/* associate the tag of the target with this cmd */
	page->tag(this, page->cmd(targetCmd_)->tag());
	return TRUE;
}


Bool
MBv2CmdCharOrChars::extract(const Byte *pb, int len)
{
	if (len < (int)sizeof(ADU_CmdChars)) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need %d)", len,
			       sizeof(ADU_CmdChars)));
		return FALSE;
	}
	ADU_CmdChars *adu = (ADU_CmdChars *) pb;
	createCmd_ = ntoh(adu->createCmd);
	index_ = ntoh(adu->index);
	return TRUE;
}


Byte *
MBv2CmdCharOrChars::packetize(Byte *pb)
{
	ADU_CmdChars *adu = (ADU_CmdChars *) pb;
	adu->createCmd = hton(createCmd_);
	adu->index = hton(index_);
	return (Byte*)(adu+1);
}


Bool
MBv2CmdCharOrChars::incomplete(MBv2Page *page, MBv2Dependency *dep)
{
	if (isComplete_) return 0;
	MBv2Cmd *cmd=NULL;
	// only depends on the createCmd_
	cmd = page->cmd(createCmd_);
	if (!cmd) {
		if (dep) {
			dep->cid = page->cid();
			dep->cmd = createCmd_;
			dep->degree = 1;
		}
		return 1;
	}
	if (cmd->incomplete(page, dep)) {
		if (dep) dep->degree++;
		return 1;
	}
	isComplete_ = 1;
	return 0;
}


Bool
MBv2CmdChar::extract(const Byte *pb, int len)
{
	if (!MBv2CmdCharOrChars::extract(pb, len)) return FALSE;
	pb += sizeof(ADU_CmdChars);
	len-= sizeof(ADU_CmdChars);
	if (len < 1) {
		MTrace(trcMB, ("too small ADU (got %d bytes, need 1)", len));
		return FALSE;
	}

	char_ = *pb;
	return TRUE;
}


Byte *
MBv2CmdChar::packetize(Byte *pb)
{
	pb = MBv2CmdCharOrChars::packetize(pb);
	*pb = char_;
	return pb+1;
}


char *
MBv2CmdCharOrChars::create_text(MBv2Page *page, MBv2CmdId start)
{
	MBv2Cmd *createCmd = page->cmd(createCmd_);
	char *text=NULL;
	MBv2TextItem *item = (MBv2TextItem*)MBv2CmdText::item(createCmd);
	u_int16_t max_chars=0, text_len;

	// add an extra byte to max_chars for the text_len, coz the common
	// case is to insert a character into the text
	text_len = item->max_chars()+1;

	if (createCmd->type()==MBv2EditText && start==MBv2InvalidCmdId) {
		max_chars = strlen(((MBv2TextItem*)item)->get_text());
		if (text_len <= max_chars) text_len = max_chars+1;
		text = new char [text_len+1];
		strcpy(text, ((MBv2TextItem*)item)->get_text());
	} else {
		text = new char [text_len+1];
		*text= '\0';
	}

	MBv2CmdId cmdId=((start==MBv2InvalidCmdId) ? (createCmd_+1):start),
		max=page->max_cmdid();
	MBv2CmdCharOrChars *cmd;
	while (cmdId < max) {
		cmd = (MBv2CmdCharOrChars*)page->cmd(cmdId);
		if (cmd) {
			if (((MBv2Cmd*)cmd)->type()!=MBv2Char &&
			    ((MBv2Cmd*)cmd)->type()!=MBv2Chars) break;
			cmd->insert_chars(text, text_len, max_chars);
		}
		cmdId++;
	}

	item->max_chars(max_chars);
	return text;
}


void
MBv2CmdCharOrChars::grow_text(char *&text, u_int16_t text_len)
{
	char *ntext=NULL;
	ntext = new char [text_len+1];
	strcpy(ntext, text);
	delete [] text;
	text = ntext;
}


void
MBv2CmdChar::insert_chars(char *&text, u_int16_t &text_len,
			  u_int16_t &max_chars)
{
	u_int16_t len = strlen(text);

	if (char_ == '\b') {
		// delete the character before index_
		if (index_ > 0 && len >= index_) {
			memmove(text+index_-1, text+index_, len-index_+1);
		}
	} else {
		// insert char_ at index_
		// we may need to add padding spaces
		if (len < index_) {
			if (text_len <= index_) {
				text_len = index_ + 2;
				grow_text(text, text_len);
			}
			memset(text+len, ' ', index_ - len);
			text[index_] = char_;
			text[index_+1] = '\0';
			if (max_chars <= index_) max_chars = index_+1;
		} else {
			if (text_len <= len) {
				text_len = len + 2;
				grow_text(text, text_len);
			}
			memmove(text+index_+1, text+index_, len-index_+1);
			text[index_] = char_;
			if (max_chars <= len) max_chars = len+1;
		}
	}
}


Bool
MBv2CmdChar::apply(MBv2Page *page)
{
	// FIXME: right now we cannot check if the command has been
	// applied twice in a row!
	MBv2CanvId canvId  = page->get_canvas_item(createCmd_);
	if (canvId==MBv2InvalidCanvId) {
		MTrace(trcMB, ("inserting char into non-existent text item"));
		return FALSE;
	}

	MBv2Canvas *canv = page->canvas();
	if (!canv) {
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	// allocate a new string
	MBv2Item *item = MBv2CmdText::item(page->cmd(createCmd_));
	if (!item || item->type()!=MBv2Text) {
		MTrace(trcMB, ("invalid text item for cmd %u", createCmd_));
		return FALSE;
	}

	char *text = create_text(page);

	/* don't assoc item with this command yet, this will be done
	 * at the grouptext command */
	canv->set_text(canvId, text);
	delete [] text;
	/* associate the tag of the target with this cmd */
	page->tag(this, page->cmd(createCmd_)->tag());
	return TRUE;
}


Bool
MBv2CmdChars::extract(const Byte *pb, int len)
{
	if (!MBv2CmdCharOrChars::extract(pb, len)) return FALSE;
	pb += sizeof(ADU_CmdChars);
	len-= sizeof(ADU_CmdChars);

	// find the terminating '\0'
	int i;
	for (i=0; i<len; i++)
		if (pb[0]=='\0') break;

	chars_ = new char [i+1];
	if (i>0) memcpy(chars_, pb, i);
	chars_[i] = '\0';
	return TRUE;
}


Byte *
MBv2CmdChars::packetize(Byte *pb)
{
	pb = MBv2CmdCharOrChars::packetize(pb);
	strcpy((char*)pb, chars_);
	return pb + strlen(chars_) + 1;
}


void
MBv2CmdChars::insert_chars(char *&text, u_int16_t &text_len,
			   u_int16_t &max_chars)
{
	u_int16_t len, clen;
	clen = strlen(chars_);
	len = strlen(text);

	// insert char_ at index_
	// we may need to add padding spaces
	if (index_ == MBv2CmdChars_ReplaceText) {
		if (text_len < clen) {
			text_len = clen+2;
			grow_text(text, text_len);
		}
		strcpy(text, chars_);
		if (max_chars < clen) max_chars = clen;
	} else if (len <= index_) {
		if (text_len < index_+clen) {
			text_len = index_+clen+2;
			grow_text(text, text_len);
		}
		memset(text+len, ' ', index_-len);
		strcpy(text + index_, chars_);
		if (max_chars < index_+clen) max_chars = index_+clen;
	} else {
		if (text_len < len+clen) {
			text_len = len+clen+2;
			grow_text(text, text_len);
		}
		memmove(text+index_+clen, text+index_, len-index_+1);
		strncpy(text + index_, chars_, clen);
		if (max_chars < len+clen) max_chars = len+clen;
	}
}


Bool
MBv2CmdChars::apply(MBv2Page *page)
{
	// FIXME: right now we cannot check if the command has been
	// applied twice in a row!
	MBv2CanvId canvId  = page->get_canvas_item(createCmd_);
	if (canvId==MBv2InvalidCanvId) {
		MTrace(trcMB, ("inserting chars into non-existent text item"));
		return FALSE;
	}

	MBv2Canvas *canv = page->canvas();
	if (!canv) {
		MTrace(trcMB, ("could not find canvas for page with cid %u",
			       page->cid()));
		return FALSE;
	}

	// allocate a new string
	MBv2Item *item = MBv2CmdText::item(page->cmd(createCmd_));
	if (!item || item->type()!=MBv2Text) {
		MTrace(trcMB, ("invalid text item for cmd %u", createCmd_));
		return FALSE;
	}

	char *text = create_text(page);

	/* don't assoc item with this command yet, this will be done
	 * at the grouptext command */
	canv->set_text(canvId, text);
	delete [] text;
	/* associate the tag of the target with this cmd */
	page->tag(this, page->cmd(createCmd_)->tag());
	return TRUE;
}
