/*
 * mb-appmgr.cc --
 *
 *      MediaBoard Application Manager
 *      Maintains receivers, senders etc. (top most object in mb that ties
 *      things together).
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-appmgr.cc,v 1.25 2002/02/03 03:16:30 lim Exp $
 */

#include "mb/mb-mgr.h"
#include "mb/mb-rcvr.h"

MB_DECLARE_LOG(mbLog);
int cMustByteSwapQuad;

static class MBMgrClass : public TclClass {
public:
	MBMgrClass() : TclClass("MB_Manager") {
#ifndef _MSC_VER
		const unsigned long long ll=0x1;
#else
		const __int64 ll=0x1;
#endif
		if (*((unsigned long*)(&ll))==(unsigned long)ll)  {
			// least significant long first
			MTrace(trcMB|trcExcessive,
			       ("-------- ByteSwapQuad enabled ----\n"));
			cMustByteSwapQuad=TRUE;
		} else cMustByteSwapQuad=FALSE;
	}
	TclObject* create(int argc, const char*const* /*argv*/) {
		if (argc==4) {
			return (new MBManager());
		} else {
			return (TclObject*)NULL;
		}
	}
} mbsm;


/* initialize static variables */
/* TODO: add hooks to update these values from tcl */
int MBManager::sourceActionInterval_ = 200; /* in ms */
int MBManager::pageActionInterval_ = 500; /* in ms */

MBManager::MBManager()
	: MBBaseMgr(), local_sid_(), pLocalRcvr_(NULL),
	  pObjUI_(NULL), pLastActiveCmd_(NULL),
	  lastSourceActiveTime_(0), pLastActivePage_(0),
	  lastPageActiveTime_(0)
{
	trackSrcId_.ss_uid = 0;
	trackSrcId_.ss_addr = 0;
}

/*virtual*/
MBManager::~MBManager()
{
        // note that since pLocalReceiver is part of phtReceivers_, it
	// will be deleted by the base class
	END_LOG();
	if (pObjUI_) Tcl_DecrRefCount(pObjUI_);
}

MBLocalReceiver* MBManager::CreateLocal()
{
	SRM_Source *local =
		session()->source_manager()->first_local_source();
	MTrace(trcMB|trcExcessive, ("Trying to create local source for %s",
				    (const char *)local->id()));
	pLocalRcvr_  = new MBLocalReceiver(this, local->id());
	local_sid_ = local->id();
	INIT_LOG(local_sid_.ss_uid);
	assert(AddReceiver(pLocalRcvr_) && "local receiver created twice?");

	return pLocalRcvr_;
}


int MBManager::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	if (argc==4) {
		/*| $mgr warp_time $pageid $time */
		if (!strcmp(argv[1], "warp_time")) {
			PageId pgId;
			if (!Str2PgId(argv[2], pgId)) {
				tcl.add_error("invalid pageid");
				return TCL_ERROR;
			}
			double targetTime;
			if (!str2double(argv[3], targetTime)) {
				tcl.add_error("invalid time, expected long integer");
				return TCL_ERROR;
			}
			MBTime mbTime;
			tod2mb(targetTime, mbTime);
			return warpTime(pgId, mbTime);
		}
		/*| $mgr track $src $trackIt
		 *=     if $trackIt is 1,  $ui_ notify_currPage $src $pgId
		 *=       will be called when the source's SA is received
		 *=	if $trackIt is 0, these callbacks stops
		 *=
		 *=     <b>NOTE: </b> right now we only track one
		 *=              source at a time
		 */
		if (!strcmp(argv[1], "track")) {
			SRM_Source* pSrc =
				(SRM_Source*)tcl.lookup(argv[2]);
			if (!pSrc) {
				tcl.add_error("invalid source");
				return TCL_ERROR;
			}
			int trackIt;
			if (!str2int(argv[3], trackIt)) {
				tcl.add_error("invalid arg for var trackIt");
				return TCL_ERROR;
			}
			if (!trackIt) {
				trackSrcId_.ss_uid = 0;
				trackSrcId_.ss_addr = 0;
				return TCL_OK;
			} else if (pSrc) {
				trackSrcId_ = pSrc->id();
			}
			return TCL_OK;
		}
	} else if (argc==3) {
		/*| $mgr time_range $pagid */
		if (!strcmp(argv[1], "time_range")) {
			PageId pgId;
			if (!Str2PgId(argv[2], pgId)) {
				tcl.add_error("invalid pageid");
				return TCL_ERROR;
			}
			MBTime minTime, maxTime;
			int retcode = timeRange(pgId, minTime, maxTime);
			if (retcode == TCL_OK && maxTime < minTime) {
				MTrace(trcMB, ("empty page range"));
				tcl.result("");
				return TCL_OK;
			}
			if (retcode != TCL_OK) {
				tcl.add_error("while invoking timeRange");
			} else {
				Tcl_Obj* objv[2];
				double minT, maxT;
				mb2tod(minTime, minT);
				mb2tod(maxTime, maxT);
				objv[0] = Tcl_NewDoubleObj(minT);
				objv[1] = Tcl_NewDoubleObj(maxT);
				tcl.result(Tcl_NewListObj(2, objv));
			}
			return retcode;
		}
		/*| $mgr release_time $pageId
		 *=   releases the page from being constrained to
		 *=    display at certain times */
		if (!strcmp(argv[1], "release_time")) {
			PageId pgId;
			if (!Str2PgId(argv[2], pgId)) {
				tcl.add_error("invalid pageid");
				return TCL_ERROR;
			}
			return warpTime(pgId, cMBTimeAny);
		}
		if (!strcmp(argv[1], "attach_ui")) {
			attachUI(argv[2]);
			return TCL_OK;
		}
		/*|  $mgr intoa $addr
		 *=      changes internet addresses from hex to x.x.x.x
		 *=  NOTE: $addr is base 10!!
		 */
		if (!strcmp(argv[1], "intoa")) {
			ulong ul;
			if (!str2ulong(argv[2],ul)) {
				tcl.add_error("invalid address, expected long integer");
				return TCL_ERROR;
			}
			// note that intoa assumes network byte order
			tcl.result(intoa(ul));
			return TCL_OK;
		}
	} else if (argc==2) {
		if (!strcmp(argv[1], "local_srcid")) {
			const SrcId& srcid = pLocalRcvr_->getSrcId();
			tcl.resultf("%x_%x",srcid.ss_addr,srcid.ss_uid);
			return TCL_OK;
		}
		if (!strcmp(argv[1], "local_src")) {
			SRM_Source* pSrc = pLocalRcvr_->getSrc();
			if (pSrc) {
				tcl.result(
					Tcl_NewStringObj(
						CONST_CAST(char*)(pSrc->name()),
						-1));
			} else {
				tcl.result("");
			}
			return TCL_OK;
		}
		if (!strcmp(argv[1], "abort")) {
			assert(FALSE && "progam aborted");
			END_LOG();
			tcl.eval("exit");
			return TCL_OK;
		}
		if (!strcmp(argv[1], "tod")) {
			timeval tv;
			gettimeofday(&tv, 0);
			double s = tv.tv_sec + 1e-6 * (tv.tv_usec);
			tcl.resultf("%.3f", s);
			return TCL_OK;
		}
	}
	return MBBaseMgr::command(argc, argv);
}

/* virtual */
MBPageObject *MBManager::NewPageObject(const PageId &pageId, Bool /*newPage*/)
{
	Tcl& tcl= Tcl::instance();
	const int nArgs = 3;
	Tcl_Obj* objv[nArgs];
	objv[0] = pObjUI_;
	objv[1] = Tcl_NewStringObj("create_canvas", -1);
	objv[2] = PgId2Obj(pageId);
	if (TCL_OK != tcl.evalObjs(nArgs, objv)) {
		tcl.add_error("create_canvas failed");
		return NULL;
	}

	char *szCanvas = tcl.result();
	Page* pPage = new Page(this, pageId, szCanvas);
	if (!pPage) {
		SignalError(("out of memory"));
		return NULL;        // return null page id
	}
	return pPage;
}


/* virtual */
MBBaseRcvr *MBManager::NewReceiver(const SrcId &srcId, Bool isLocal)
{
	if (isLocal==TRUE) return NULL;
	MTrace(trcMB|trcVerbose,
	       ("Creating new receiver for %s", (const char*)srcId));
	return new MBReceiver(this, srcId);
}


/* virtual */
Bool MBManager::isVisible(const PageId &pgId)
{
	return (pgId == pLocalRcvr_->getCurrPgId());
}


/* virtual */
void MBManager::Rearrange(MBPageObject *pPage, MBCmd *pCmd)
{
	// change the display order of the targeted item to reflect
	// the timestamp
	n_long itemId = pCmd->getItemId();
	if (itemId) {
		CanvItemId cid = FindMostRecent(pPage->getId(),
						pCmd->getTimeStamp());
		if (cid) ((Page*)pPage)->RaiseAfter(cid,itemId);
	}
}

#ifdef MB_DEBUG
static int s_nSourceActionSkip=0;
static int s_nPageActionSkip=0;
#endif

//
// MBManager::activity --
//   notify UI of an activity
//
//   Optimizations:
//      - calls to the same source/page within 200ms
//	intervals are suppressed
//	- also, once commands with newer timestamps has been notified,
//      notification for commands with older timestamps will be suppressed
//      - for local pages, only source actions are notified (not page/cmd)

/* virtual */
void MBManager::activity(MBPageObject *pPageObj, MBCmd* pCmd)
{
	/* skip commands that happened before our last activity
	 * notification */
	if (pLastActiveCmd_ != NULL &&
	    pLastActiveCmd_->getTimeStamp() > pCmd->getTimeStamp()) {
		return;
	}
	if (!pPageObj || !pCmd) {
		assert(FALSE ||
		       !"MBMgr::activity should be called with non-null page and command");
		return;
	}

	long newtime;
	timeval tv;
	gettimeofday(&tv, 0);
	/* FIXME: ignore overflows for now, no big deal */
	newtime = tv.tv_sec*1000 + tv.tv_usec/1000;
	MBBaseRcvr* pRcvr = pCmd->rcvr();
	Page* pPage = DYN_CAST(Page*)(pPageObj);
	if (!pRcvr) {
		assert(pRcvr ||
		       !"MBManager::activity command has null rcvr");
		return;
	}
	Tcl_Obj *pObjSrc = Tcl_NewStringObj("", -1);
	/* note: since pLastActiveRcvr is initialized to zero,
	 * we won't skip the first call */
	MBBaseRcvr* pLastRcvr = (pLastActiveCmd_) ?
		pLastActiveCmd_->rcvr() : ((MBBaseRcvr*)NULL);
	if ((pRcvr == pLastRcvr)
	    && (newtime < lastSourceActiveTime_ + sourceActionInterval_)) {
#ifdef MB_DEBUG
		s_nSourceActionSkip++;
#endif
	} else {
		lastSourceActiveTime_ = newtime;
		pLastActiveCmd_ = pCmd;

		SRM_Source* pSrc = pRcvr->getSrc();
		/* FIXME: decide what to do with receivers without source
		 * objects (in what situation do they occur?) */
		if (pSrc) {
			Tcl_SetStringObj(pObjSrc,
					 CONST_CAST(char*)(pSrc->name()), -1);
		}
	}
	MB_DefTcl(tcl);
	const int maxArgs = 6;
	Tcl_Obj* objv[maxArgs];
	int nArgs = 3;
	objv[0] = pObjUI_;
	objv[1] = Tcl_NewStringObj("activity", -1);
	objv[2] = pObjSrc;

	/* skip the other parameters for local pages */
	if (!pPage->isLocal()) {
	    if ( (pPage == pLastActivePage_)
		 && (newtime < lastPageActiveTime_ + pageActionInterval_)) {
#ifdef MB_DEBUG
		    s_nPageActionSkip++;
#endif
	    } else {
		    MBBaseCanvas* pCanv = pPage->getCanvas();
		    if (pCanv) {
			    nArgs += 3;
			    assert(nArgs<=maxArgs || !"memory corruption");
			    objv[3] = PgId2Obj(pPage->getId());
			    /* FIXME:TclObject should have a nameObj()
			     * function */
			    objv[4] =
				    Tcl_NewStringObj(CONST_CAST(char*)(pCanv->name()), -1);
			    CanvItemId cId =
				    pPage->itemId2canvId(pCmd->getItemId());

			    objv[5] = Tcl_NewIntObj(cId);
		    }
		    pLastActivePage_ = pPage;
		    lastPageActiveTime_ = newtime;
	    }
	}
	/* calls $self activity $src [$pageid $canvId] */
	if (TCL_OK != tcl.evalObjs(nArgs, objv)) {
		dispTclErr("activity failed");
	}
}

/*virtual*/
void
MBManager::notifyCurrPage(const SrcId& srcId, const PageId& pgId)
{
	if (srcId != trackSrcId_) {
		return;
	}
	SRM_Source* pSrc = getSrc(srcId);
	if (!pSrc) return;

	const int maxArgs = 4;
	Tcl_Obj* objv[maxArgs];
	int i=0;
	objv[i++] = pObjUI_;
	objv[i++] = Tcl_NewStringObj("notify_currPage", -1);
	objv[i++] = Tcl_NewStringObj(CONST_CAST(char*)(pSrc->name()),
				     -1);
	objv[i++] = PgId2Obj(pgId);
	assert(i <= maxArgs);

	/* $ui activity $src $pgId */
	if (TCL_OK != Tcl::instance().evalObjs(i, objv)) {
		dispTclErr("notifyCurrPage failed");
	}
}


// finds the most recent item just before the given timestamp
//   - loops thru all receivers and ask each receiver to return
//     the most recent before timestamp
//
CanvItemId MBManager::FindMostRecent(PageId pid, n_long timestamp)
{
	Tcl_HashSearch hsearch;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(getReceiversHT(), &hsearch);
	n_long currMostRecentTS=0;
	CanvItemId cidMostCurr=0, cidTmp;
	while (pEntry) {
		MBReceiver *pRcvr = (MBReceiver *)Tcl_GetHashValue(pEntry);
		// note: currMostRecentTS will be set to the more recent value
		//       if the receiver has a more recent item
		cidTmp=pRcvr->FindMostRecent(pid, currMostRecentTS, timestamp);
		if (cidTmp) cidMostCurr=cidTmp;
		pEntry=Tcl_NextHashEntry(&hsearch);
	}
	return cidMostCurr;
}

int MBManager::warpTime(const PageId &pgId, const MBTime& t)
{
	Tcl_HashSearch hsearch;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(getReceiversHT(),
						   &hsearch);
	int retcode=TCL_OK;
	while (pEntry && retcode == TCL_OK) {
		MBReceiver *pRcvr = (MBReceiver *)Tcl_GetHashValue(pEntry);
		retcode =  pRcvr->warpTime(pgId, t);
		if (retcode != TCL_OK) {
			assert(0); /* want to know why */
		}
		pEntry=Tcl_NextHashEntry(&hsearch);
	}
	return retcode;
}


int MBManager::timeRange(const PageId &pgId, MBTime& start, MBTime& end)
{
	Tcl_HashSearch hsearch;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(getReceiversHT(), &hsearch);
	MBTime minTime=cMBTimePosInf, maxTime=cMBTimeNegInf,
		pagemin=cMBTimePosInf, pagemax=cMBTimeNegInf;
	int retcode;
	for (pEntry = Tcl_FirstHashEntry(getReceiversHT(), &hsearch);
	     pEntry;pEntry=Tcl_NextHashEntry(&hsearch)) {
		MBReceiver *pRcvr = (MBReceiver *)Tcl_GetHashValue(pEntry);
		// note: currMostRecentTS will be set to the more recent value
		//       if the receiver has a more recent item
		retcode=pRcvr->timeRange(pgId, pagemin, pagemax);
		if (retcode==TCL_OK) {
			if (pagemin < minTime && pagemin!=cMBTimeNegInf)
				minTime = pagemin;
			if (pagemax > maxTime && pagemax!=cMBTimeNegInf)
				maxTime = pagemax;
		}
	}
	start = minTime;
	end = maxTime;
	return TCL_OK;
}

// if a receiver has requested a SA, service them first, else
// ask the local receiver to send out the SA
int MBManager::periodic_update(Byte *pb)
{
	int len = MBBaseMgr::periodic_update(pb);
	if (len==0) {
		len = pLocalRcvr_->FillSA(pb, getMTU());
	}
	return len;
}


// virtual
int MBManager::next_ADU(u_char *pb, int len, srm_src &/*sid*/, int &pktType,
			int &nextSize)
{
	nextSize=0;
	pktType = APP_DATA;
	// TODO: should poll all receivers or preq as well
	int outlen=(pLocalRcvr_->NextADU(pb, len, nextSize));
	MTrace(trcMB|trcVerbose,("MBMgr Sending %d bytes",outlen));
	return outlen;
}


// the page is about to be hidden
void MBManager::ChangeStatus(const PageId& pgid, Bool isVisible)
{
	Tcl_HashSearch hsearch;
	Tcl_HashEntry *pEntry = Tcl_FirstHashEntry(getReceiversHT(), &hsearch);
	while (pEntry) {
		MBReceiver *pRcvr = (MBReceiver *)Tcl_GetHashValue(pEntry);
		pRcvr->ChangeStatus(pgid,isVisible);
		pEntry=Tcl_NextHashEntry(&hsearch);
	}
}

/* virtual from MBBaseMgr */
void
MBManager::handle_request(const SrcId& sidRqtSrc, Byte *pb, int len)
{
        if (sidRqtSrc == local_sid_) { // ignore our own request
                MTrace(trcMB|trcVerbose,
		       ("ignoring local rqt from source: %x@%s",
			sidRqtSrc.ss_uid, intoa(sidRqtSrc.ss_addr)));
                return;
        }
        MBBaseMgr::handle_request(sidRqtSrc, pb, len);
}

/* virtual form MBBaseMgr */
void
MBManager::handle_reply(const SrcId& sidRpySrc, Byte *pb, int len)
{
        if (sidRpySrc == local_sid_) {
                return;         // forget about own reply
        }
        MBBaseMgr::handle_reply(sidRpySrc, pb, len);
}

