//////////////////////////////////////////////////////////////////////////////
// Filename:    omniNames.cc
// Description: Lightweight implementation of the naming service for vxWorks
// Author:      Tihomir Sokcevic
// Note:        This module provides 2 public methods :
//               startNamingService(port) and stopNamingService.
//              For normal operations, only use these methods.
// WARNING:     To avoid thread unsafety, always invoke the instance()
//               of the singleton.
//////////////////////////////////////////////////////////////////////////////
// $Log: omniNames.cc,v $
// Revision 1.1.1.1  2002/11/19 14:59:21  sokcevti
// OmniOrb4.0.0 VxWorks port
//
// Revision 1.1.1.1  2002/04/02 10:09:44  sokcevti
// omniORB4 initial realease
//
//
//////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <iostream.h>
#include <omnithread.h>
#include <NamingContext_i.h>

#ifdef __WIN32__
#include <io.h>
#else
#include <unistd.h>
#endif

#ifndef O_SYNC
#ifdef  O_FSYNC              // FreeBSD 3.2 does not have O_SYNC???
#define O_SYNC O_FSYNC
#endif
#endif

// Minimum idle period before we take a checkpoint (15 mins)
#define DEFAULT_IDLE_TIME_BTW_CHKPT  (15*60)


PortableServer::POA_var the_poa;
PortableServer::POA_var the_ins_poa;

//////////////////////////////////////////////////////////////////////////////
//   Local defines
//////////////////////////////////////////////////////////////////////////////
#define DEFAULT_PORT        5000
#define MAX_TEXT_INT_LENGTH   40


//////////////////////////////////////////////////////////////////////////////
// Class :      CNamingService is a thread AND a singleton
// Description: Starts the naming service thread and encapsulate application
//               private data.
// Methods:     start ...... Starts the naming service
//              stop ....... Stop the service
//              instance ... Get the unique singleton naming service instance
//              dispose .... Delete the singleton object safely
//////////////////////////////////////////////////////////////////////////////
class CNamingService : public omni_thread
   {
public:
   CNamingService();

   static CNamingService *instance();
   void dispose();
   int start( int nPort );
   int stop();

protected:
   virtual void *run_undetached(void *arg);

private:
   static omni_mutex m_Lock;
   static CNamingService *m_pInstance;

   char m_pszPort[MAX_TEXT_INT_LENGTH];
   };


//////////////////////////////////////////////////////////////////////////////
//   Static instantiation
//////////////////////////////////////////////////////////////////////////////
omni_mutex CNamingService::m_Lock;
CNamingService *CNamingService::m_pInstance;


//////////////////////////////////////////////////////////////////////////////
// Function:    CNamingService::ctor
// Description: Call base class ctor
// Note:        The cast is vital since this is the only to distinguish btw
//               attached or detached thread.
//////////////////////////////////////////////////////////////////////////////
CNamingService::CNamingService() : omni_thread( (void* (*)(void*))NULL )
   {
   // Initialise base class only
   }


//////////////////////////////////////////////////////////////////////////////
// Function:    CNamingService::instance
// Description: Returns the unique instance of the nameservice
// Note:        The mutex object is a static instance
//////////////////////////////////////////////////////////////////////////////
CNamingService *CNamingService::instance()
   {
   omni_mutex_lock l( m_Lock );

   if ( m_pInstance == 0 )
      {
      m_pInstance = new CNamingService();
      }

   return m_pInstance;
   }


//////////////////////////////////////////////////////////////////////////////
// Function:    CNamingService::dispose
// Description: Delete the object safely
// Note:        The mutex object is a static instance
//////////////////////////////////////////////////////////////////////////////
void CNamingService::dispose()
   {
   omni_mutex_lock l( m_Lock );

   delete m_pInstance;

   m_pInstance = 0;
   }


//////////////////////////////////////////////////////////////////////////////
// Function:    CNamingService::start
// Description: Starts the naming service
// Args:        nPort ... Port number.
// Returns:     0 on success. -1 otherwise
// Note:
//////////////////////////////////////////////////////////////////////////////
int CNamingService::start( int nPort )
   {
   omni_mutex_lock l( m_Lock );

   // Check the port number
   if ( nPort == 0 )
      {
      nPort = DEFAULT_PORT;
      }

   // Cannot proceed if the thread is already running
   if ( state() == STATE_RUNNING )
      {
      printf("Warning: Naming service already started at port %s\n", m_pszPort );

      return -1;
      }

   // Transpose the port to text
   sprintf(m_pszPort, "giop:tcp::%d", nPort);

   // Start the thread undetached (omni_thread will call run_undetached)
   // An undetached thread can be later joined.
   start_undetached();

   // Indicate success
   return 0;
   }


//////////////////////////////////////////////////////////////////////////////
// Function:    CNamingService::stop
// Description: Stops the naming service
// Returns:     0 on success. -1 otherwise
// Note:        The singleton is deleted too
//////////////////////////////////////////////////////////////////////////////
int CNamingService::stop()
   {
   int nRetVal;

   omni_mutex_lock l( m_Lock );

   if ( state() != STATE_RUNNING )
      {
      printf("Warning: Naming service not previously running.\n" );

      return -1;
      }

   // Shut down the boa to allow the thread to come out of its loop
//   m_pBoa->impl_shutdown();

   // Actually wait for the thread to complete
   join( (void**)(&nRetVal) );

   // Irrespective of whether or not stop actually succedded,
   //  dispose of the singleton.
   m_pInstance = 0;

   delete this;

   return nRetVal;
   }


//////////////////////////////////////////////////////////////////////////////
// Function:    CNamingService::run
// Description: Naming service loop
// Args:        None used
// Note:        Ported from omniORB naming service
//////////////////////////////////////////////////////////////////////////////
void *CNamingService::run_undetached( void * )
   {
   int argc = 3;
   char *argv[3] = {""};

   // Fill-in the port option
   argv[1] = "-ORBendPoint";
   argv[2] = m_pszPort;

  //
  // Initialize the ORB and the object adapter.
  //

  CORBA::ORB_ptr orb;

  try {
    orb = CORBA::ORB_init(argc, argv);
  }
  catch (CORBA::INITIALIZE& ex) {
    cerr << "Failed to initialise the ORB." << endl;
    return 0;
  }

  try {
    CORBA::Object_var poaref = orb->resolve_initial_references("RootPOA");
    PortableServer::POA_var poa = PortableServer::POA::_narrow(poaref);

    PortableServer::POAManager_var pman = poa->the_POAManager();

    CORBA::PolicyList pl(1);
    pl.length(1);
    pl[0] = poa->create_lifespan_policy(PortableServer::PERSISTENT);

    the_poa = poa->create_POA("", pman, pl);
    pman->activate();

    // Get the "magic" interoperable naming service POA
    poaref      = orb->resolve_initial_references("omniINSPOA");
    the_ins_poa = PortableServer::POA::_narrow(poaref);
    pman        = the_ins_poa->the_POAManager();
    pman->activate();
  }
  catch (CORBA::INITIALIZE& ex) {
    cerr << "Failed to initialise the POAs. "
	 << "Is omniNames already running?" << endl;
    return 0;
  }

   return 0;
   }


//////////////////////////////////////////////////////////////////////////////
// Function:    startNamingService
// Description: start the naming service agent. This entry points creates a
//               naming service singleton thread object.
//              Therefore, only 1 naming service runs at a time.
// Args:        port ... Port number to run on
// Returns:     0 on successful call. A different value otherwise.
// Note:        The behaviour of what happen if the service already runs is
//               left to the CNamingService object.
//////////////////////////////////////////////////////////////////////////////
int startNamingService( int port = 0 )
   {
   return CNamingService::instance()->start( port );
   }


//////////////////////////////////////////////////////////////////////////////
// Function:    stopNamingService
// Description: stops the naming service agent.
// Returns:     0 on successful call. A different value otherwise.
// Note:        The behaviour of what happen if the service already runs is
//               left to the CNamingService object.
//////////////////////////////////////////////////////////////////////////////
int stopNamingService( void )
   {
   int nRetVal;

   // Stop the service
   nRetVal = CNamingService::instance()->stop();

   return nRetVal;
   }


//////////////////////////////////////////////////////////////////////////////
// Function:    main
// Description: Entry point for non vxWorks application
// Returns:     0 on successful call. A different value otherwise.
// Note:
//////////////////////////////////////////////////////////////////////////////
#ifndef __vxWorks__
int main( void )
   {
   return startNamingService();
   }
#endif

// ------------------------------  End of file  ------------------------------

//
// main
//
/*
int
main(int argc, char **argv)
{
  //
  // Initialize the ORB and the object adapter.
  //

  CORBA::ORB_ptr orb;

  try {
    orb = CORBA::ORB_init(argc, argv);
  }
  catch (CORBA::INITIALIZE& ex) {
    cerr << "Failed to initialise the ORB." << endl;
    return 1;
  }

  try {
    CORBA::Object_var poaref = orb->resolve_initial_references("RootPOA");
    PortableServer::POA_var poa = PortableServer::POA::_narrow(poaref);

    PortableServer::POAManager_var pman = poa->the_POAManager();

    CORBA::PolicyList pl(1);
    pl.length(1);
    pl[0] = poa->create_lifespan_policy(PortableServer::PERSISTENT);

    the_poa = poa->create_POA("", pman, pl);
    pman->activate();

    // Get the "magic" interoperable naming service POA
    poaref      = orb->resolve_initial_references("omniINSPOA");
    the_ins_poa = PortableServer::POA::_narrow(poaref);
    pman        = the_ins_poa->the_POAManager();
    pman->activate();
  }
  catch (CORBA::INITIALIZE& ex) {
    cerr << "Failed to initialise the POAs. "
	 << "Is omniNames already running?" << endl;
    return 1;
  }

  //
  // Now this thread has nothing much to do.  Simply take a checkpoint once
  // every so often.
  //

  int idle_time_btw_chkpt;
  char *itbc = getenv("OMNINAMES_ITBC");
  if (itbc == NULL || sscanf(itbc,"%d",&idle_time_btw_chkpt) != 1)
    idle_time_btw_chkpt = DEFAULT_IDLE_TIME_BTW_CHKPT;

  omni_mutex m;
  omni_condition c(&m);

  m.lock();
  while (1) {
    unsigned long s, n;
    omni_thread::get_time(&s, &n, idle_time_btw_chkpt);
    c.timedwait(s,n);
  }
  m.unlock();

  return 0;
}
*/