/* Copyright (C) 2002-2004  Mark Andrew Aikens <marka@desert.cx>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: IO.h,v 1.7 2004/08/22 14:52:02 marka Exp $
 */
#ifndef __IO_h
#define __IO_h

#include <stdint.h>

/** \file */


/**
 * A base class for lowlevel access methods to the PIC programmer.
 */
class IO {
public:
	/** Creates an instance of an IO class from it's name.
	 * \param name The name of the IO hardware to use.
	 * \param port The port number to pass to the specific subclass
	 *        constructor.
	 * \returns An instance of the IO subclass that implements the
	 *          hardware interface specified by \c name.
	 * \throws runtime_error Contains a textual description of the error.
	 */
	static IO *acquire(char *name, int port = 0);

	/** Frees all the memory and resources associated with this object */
	virtual ~IO();

	/** Sets the state of the clock signal.
	 * \param state The new state of the clock signal.
	 */
	virtual void set_clk(bool state) = 0;

	/** Gets the state of the clock output signal.
	 * \return The boolean state of the clock signal.
	 */
	virtual bool get_clk(void) = 0;

	/** Sets the state of the data signal.
	 * \param state The new state of the data signal.
	 */
	virtual void set_data(bool state) = 0;

	/** Gets the state of the data output signal.
	 * \return The boolean state of the data signal.
	 */
	virtual bool get_data(void) = 0;

	/** Reads the value of the data input signal.
	 * \return The boolean state of the data signal.
	 */
	virtual bool read_data(void) = 0;

	/** Sets the state of the programming voltage.
	 * \param state The new state of the programming voltage.
	 */
	virtual void set_vpp(bool state) = 0;

	/** Gets the state of the programming voltage output.
	 * \return The boolean state of the vpp signal.
	 */
	virtual bool get_vpp(void) = 0;

	/** Sets the state of Vcc to the device.
	 * \param state The new state of supply voltage.
	 */
	virtual void set_pwr(bool state) = 0;

	/** Gets the state of the supply voltage output.
	 * \return The boolean state of the pwr signal.
	 */
	virtual bool get_pwr(void) = 0;

	/** Waits for a specified number of microseconds to pass.
	 * \param us The number of microseconds to delay for.
	 */
	virtual void usleep(unsigned long us);

	/**
	 * Sends a stream of up to 32 bits on the data signal, clocked by the
	 * clock signal. The bits are sent LSB first and it is assumed that
	 * the device will latch the data on the falling edge of the clock.
	 *
	 * \param bits The data bits to send out, starting at the LSB.
	 * \param numbits The number of bits to send out.
	 * \param tset Setup time for the data before the clock's falling
	 *        edge (in microseconds).
	 * \param thold The data hold time after the clock's falling
	 *        edge (in microseconds).
	 */
	virtual void shift_bits_out(uint32_t bits, int numbits, int tset=0, int thold=0);

	/**
	 * Reads a stream of up to 32 bits from the data signal, clocked by
	 * us controlling the clock signal. The bits are received LSB first
	 * and are read tdly microseconds after the rising edge of the clock.
	 *
	 * \param numbits The number of bits to read in and return.
	 * \param tdly The delay time (microseconds) from the clock's rising
	 *        edge to data valid (from the device).
	 * \returns The bits which were read.
	 */
	virtual uint32_t shift_bits_in(int numbits, int tdly=0);

protected:
	/** Constructor
	 * \param port A subclass specific port number.
	 */
	IO(int port);

	/** Additional delay to add to Tdly. This will stretch out the length
	 * of the clock signal by this many microseconds. */
	unsigned long tdly_stretch;

	/** Additional delay to add to Tset. This will increase the data setup
	 * time by this many microseconds. */
	unsigned long tset_stretch;

	/** Additional delay to add to Thld. This will increase the data hold
	 * time by this many microseconds. */
	unsigned long thld_stretch;

	/** General additional delay. This extra time will be added to every
	 * delay that occurs. Use this to slow down the entire programming
	 * operation. */
	unsigned long gen_stretch;
};


#endif
