## Copyright (C) 2001 Paul Kienzle
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
## USA

## x = trisolve(d,e,b)
##
## Solves the tridiagonal system:
##
##   / d1 e1  0  . . .  0    0 \     / b11 b12 . . . b1k \
##   | e1 d2 e2  . . .  0    0 |     | b21 b22 . . . b2k |
##   |  0 e2 d3  . . .  0    0 | x = | b31 b22 . . . b3k |
##   |           . . .         |     |         . . .     |
##   \  0  0  0  . . . en-1 dn /     \ bn1 bn2 . . . bnk /
##
## x = trisolve(l,d,u,b)
##
## Solves the tridiagonal system:
##
##   / d1 u1  0  . . .  0    0 \     / b11 b12 . . . b1k \
##   | l1 d2 u2  . . .  0    0 |     | b21 b22 . . . b2k |
##   |  0 l2 d3  . . .  0    0 | x = | b31 b22 . . . b3k |
##   |           . . .         |     |         . . .     |
##   \  0  0  0  . . . ln-1 dn /     \ bn1 bn2 . . . bnk /
##
## x = trisolve(d,e,b,cl,cu)
##
## Solves the periodic tridiagonal system:
##
##   / d1 e1  0  . . .  0   cu \     / b11 b12 . . . b1k \
##   | e1 d2 e2  . . .  0    0 |     | b21 b22 . . . b2k |
##   |  0 e2 d3  . . .  0    0 | x = | b31 b22 . . . b3k |
##   |           . . .         |     |         . . .     |
##   \ cl  0  0  . . . en-1 dn /     \ bn1 bn2 . . . bnk /
##
## x = trisolve(l,d,u,b,cl,cu)
##
## Solves the periodic tridiagonal system:
##
##   / d1 u1  0  . . .  0   cu \     / b11 b12 . . . b1k \
##   | l1 d2 u2  . . .  0    0 |     | b21 b22 . . . b2k |
##   |  0 l2 d3  . . .  0    0 | x = | b31 b22 . . . b3k |
##   |           . . .         |     |         . . .     |
##   \ cl  0  0  . . . ln-1 dn /     \ bn1 bn2 . . . bnk /
##
## Uses A\b, so is only practical for small systems.  Use the compiled
## form for efficient LAPACK tridiagonal solvers.

function x = trisolve(l, d, u, b, cl, cu)

  if (nargin == 3)
    b = u; u = d; d = l; l = u;
    cl = cu = 0;
  elseif (nargin == 4)
    cl = cu = 0;
  elseif (nargin == 5)
    cu = cl; cl = b; b = u; u = d; d = l; l = u;
  elseif (nargin != 6)
    usage ("x = trisolve ([l, ] d, u, b [, cl, cu])");
  endif

  n = length(d);
  if any ( [ length(l)+1, length(u)+1, rows(b) ] != n )
    error ("trisolve: incorrect arguments");
  else
    A = diag (l, -1) + diag (d) + diag (u, 1);
    A(n,1) = cl; A(1,n) = cu;
    x = A\b;
  endif

endfunction
