/*
  keysearch: extract lines beginning with a given string (key) from
  a NoSQL table.

  Copyright (c) 2003 Carlo Strozzi

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  $Id: keysearch.c,v 1.2 2003/09/23 10:11:37 carlo Exp $

*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/file.h>
#include <string.h>

#ifndef HELPDIR
#define HELPDIR "/usr/local/nosql/help"
#endif
#ifndef HELPFILE
#define HELPFILE (HELPDIR "/keysearch.txt")
#endif

#define MAXKEYLEN 256

/* global variables */

struct {
  unsigned int header : 1;
  unsigned int fold : 1;
  unsigned int partial : 1;
} flags;

static const char *progname = "keysearch";  /* global pointer to argv[0] */


/* Unbuffered input routine */

static int getch(int fd) {
    char c;
    return (read(fd, &c, 1) == 1) ? (unsigned char) c : EOF;
}

static int eusage(void) {
    fprintf(stderr, "Usage: %s [-df] string file\n", progname);
    return 2;
}

static int etoolong(void) {
    fprintf(stderr, "%s: max. search string width (%d) exceeded\n",\
	progname, MAXKEYLEN);
    return 2;
}

int main(int argc, char *argv[]) {

    int i=0, ifd, min_args=3;
    char c;

    char tmpbuf[MAXKEYLEN] = "";		/* local work buffer */

    FILE *ofile = NULL;

    flags.header = 1;
    flags.fold = 0;
    flags.partial = 0;

    while (++i < argc && *argv[i] == '-') {

            min_args++;

            if (!strcmp(argv[i], "-N") ||
                     !strcmp(argv[i], "--no-header")) flags.header = 0;

            if (!strcmp(argv[i], "-p") ||
                     !strcmp(argv[i], "--partial")) flags.partial = 1;

            else if (!strcmp(argv[i], "-f") ||
                     !strcmp(argv[i], "--fold")) flags.fold = 1;

            else if (!strcmp(argv[i], "-o") ||
                     !strcmp(argv[i], "--output")) {

              if (++i >= argc || *argv[i] == '-') exit(eusage());

              min_args++;

              if ((ofile = freopen(argv[i], "w", stdout)) < 0) {
                 perror(argv[0]);
                 exit(2);
	      }
            }

            else if (!strcmp(argv[i], "-h") ||
                     !strcmp(argv[i], "--help")) {

                    execlp("grep","grep","-v","^#",HELPFILE,(char *) 0);
                    perror("grep");
                    exit(0);
            }
        }

        if (argc < min_args) exit(eusage());

	/* account for the append of a TAB, depending on flags.partial */
	if (strlen(argv[i]) >= (MAXKEYLEN-1)) exit(etoolong());

	if ((ifd = open(argv[i+1], O_RDONLY, 0)) < 0) {
	   fprintf(stderr, "%s: unable to open file %s\n",\
					progname, argv[i+1]);
	   exit(2);
	}

        while ((c = getch(ifd)) != EOF) {
              if (flags.header) putchar(c);
              if (c == '\n') {
                 fflush(NULL);   /* Make sure the header is output */
		 close(ifd);		/* close when no longer needed */
		 sprintf(tmpbuf, "LC_ALL=POSIX");
		 putenv(tmpbuf);

		 strcpy(tmpbuf, argv[i]);

		 if (!flags.partial) strcat(tmpbuf, "\t"); 

		 if (flags.fold) execlp("look", "look", "-f", "-t",\
				"\t", tmpbuf, argv[i+1], (char *) 0);

		 else execlp("look", "look", "-t", "\t",\
				tmpbuf, argv[i+1], (char *) 0);

                 perror("look");
                 exit(2);
              }
        }

        exit(0);                        /* Not reached */
}

/* EOF */
