// formantsynth.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <math.h>

#include "application.h"
#include "data.h"
#include "datamodifier.h"
#include "formantsynth.h"
#include "lpcdata.h"
#include "lpcfilter.h"
#include "formantrequester.h"
#include "synthfunction.h"
#include "pitchconv.h"
#include "reson.h"


class FormantSynthRequester : public FormantRequester {
	friend class FormantSynthesizer;
public:
	redefined void configureRequest(Request *);
	redefined boolean confirmValues();
protected:
	FormantSynthRequester(const char* title, FormantSynthesizer *);
private:
	enum TransposeMode { Ratio = 1, OctavePoint = 2, LinearOctave = 4 };
	static double		_savedTransposeFactor;
	static ChoiceValue	_savedTransposeMode;
	static ChoiceValue	_savedUnvoicedOnly;
	FormantSynthesizer* client;	
};

double FormantSynthRequester::_savedTransposeFactor = 1.0;
ChoiceValue FormantSynthRequester::_savedTransposeMode = FormantSynthRequester::Ratio;
ChoiceValue FormantSynthRequester::_savedUnvoicedOnly = Requester::False;

FormantSynthRequester::FormantSynthRequester(
			const char* title, FormantSynthesizer* fs)
	: FormantRequester(title, fs, fs->sampRate(), fs->gain,
	                   fs->warpFactor, fs->interpMode), 
	  client(fs) {}
		                 
void
FormantSynthRequester::configureRequest(Request* request) {
	request->appendLabel("(Leave reson BW at 0 to turn it off.)");
	request->appendValue("Unvoiced threshold:", &client->hiThreshold,
			    Range(0.0, 1.0), true);
	request->appendValue("Voiced threshold:", &client->lowThreshold,
	                    Range(0.0, 1.0), true);
	request->appendValue("Voiced/Unvoiced amp factor:",
	                     &client->voicedFactor,
						 Range(0.0, 10.0), true);
	request->appendValue("Gain threshold:",
	                     &client->gainThresh,
						 client->getLPC()->limits(0), true);
	request->appendChoice("Mute Voiced Frames:",
						  "|No|Yes|",
						  &_savedUnvoicedOnly, true);
    request->appendValue("Pitch Transposition Interval/Ratio:",
						 &_savedTransposeFactor);
	request->appendChoice("Transposition mode:",
						  "|Ratio|Octave Pt. P.C.|Linear Octave P.C.|",
						  &_savedTransposeMode, true);															     
	request->appendValue("Reson center freq (Hz):",
	                     &client->resonCF,
						 Range(0.0, client->sampRate()/2.0), true);
	request->appendValue("Reson bandwidth (Hz):",
	                     &client->resonBW,
						 Range(0.0, client->sampRate()/2.0), true);
	FormantRequester::configureRequest(request);
}

boolean
FormantSynthRequester::confirmValues() {
    if (!FormantRequester::confirmValues())
	    return false;
    double factor = _savedTransposeFactor;
    double ifactor = 0;
    switch(_savedTransposeMode) {
    case Ratio:
        ifactor = factor;
        break;
    case OctavePoint:
        factor = Frequency::octaveFromPitch(factor);
        ifactor = Frequency::cpsFromOctave(10.0+factor) / Frequency::cpsFromOctave(10.0);
        break;
    case LinearOctave:
        ifactor = Frequency::cpsFromOctave(10.0+factor) / Frequency::cpsFromOctave(10.0);
        break;
    }
    if(ifactor <= 0.0) {
        Application::alert("Ratio mode value must be a positive number.");
        return false;
    }
    client->setTransposeFactor(ifactor);
    client->setUnvoicedOnly(_savedUnvoicedOnly == Requester::True);
	return true;
}
//********


static unsigned seed = 3;

double FormantSynthesizer::_savedGain = 1.0;
double FormantSynthesizer::_savedVoicedFactor = 3.0;
double FormantSynthesizer::_savedHiThreshold = 0.1;
double FormantSynthesizer::_savedLowThreshold = 0.0;
double FormantSynthesizer::_savedWarpFactor = 0.0;
double FormantSynthesizer::_savedTransposeFactor = 1.0;
double FormantSynthesizer::_savedResonCF = 0.0;
double FormantSynthesizer::_savedResonBW = 0.0;
double FormantSynthesizer::_savedGainThresh = 0.0;

ChoiceValue FormantSynthesizer::_savedInterpMode = FormantSynthesizer::Linear;

FormantSynthesizer::FormantSynthesizer(Data* output, Data* lpcdata,
	double freq, int nHarms, double amp)
		: LPC_Filter(lpcdata, output->length(), false),
		  Buzz(output, freq, nHarms, amp),
		  noise(seed), pResonFilter(NULL), gain(1.0), voicedFactor(3.0),
		  hiThreshold(1.0), lowThreshold(0.0), warpFactor(0.0),
		  interpMode(Linear), transposeFactor(1.0), resonCF(0.0), resonBW(0.0),
		  gainThresh(0), unvoicedOnly(false)
{
    okIf(LPC_Filter::good());
	if (ok())
		initialize();
}

FormantSynthesizer::FormantSynthesizer(Data* output, Data* lpcdata)
		: LPC_Filter(lpcdata, output->length(), false), Buzz(output),
		  noise(seed), pResonFilter(NULL), gain(_savedGain),
		  voicedFactor(_savedVoicedFactor),
		  hiThreshold(_savedHiThreshold), lowThreshold(_savedLowThreshold),
		  warpFactor(_savedWarpFactor), interpMode(_savedInterpMode),
		  transposeFactor(_savedTransposeFactor),
		  resonCF(_savedResonCF), resonBW(_savedResonBW),
		  gainThresh(_savedGainThresh)
{
    okIf(LPC_Filter::good());
}

FormantSynthesizer::~FormantSynthesizer()
{
	delete pResonFilter;
}

Requester *
FormantSynthesizer::createRequester() {
	return new FormantSynthRequester(
		"Resynthesize LPC Data into Selected Region:", this);
}

void
FormantSynthesizer::initialize() {
	if(warpFactor != 0.0) {
		LPCData *lpd = getLPC();
		if (lpd) {
			Data* warpData = new Data(FloatData, lpd->length() + 1);
			warpData->ref();
			Offset o(warpData, warpFactor);	// set to single value
			o.apply();
			setWarp(warpData);
			Resource::unref(warpData);
		}
	}
	hiThreshold = sqrt(hiThreshold);
	lowThreshold = sqrt(lowThreshold);
	setInterpMode(InterpMode(interpMode));
	LPC_Filter::initialize();
	Buzz::initialize();
	if (resonBW != 0.0)
	{
		pResonFilter = new Reson4(target(), resonCF, resonBW);
	}
	srOverTwo = sampRate()/2.0;
	restoreState();
}

void
FormantSynthesizer::saveConfig() {
    _savedGain = gain;
    _savedVoicedFactor = voicedFactor;
    // square terms since they were sqrt'd in initialize()
    _savedHiThreshold = hiThreshold * hiThreshold;
    _savedLowThreshold = lowThreshold * lowThreshold;
    _savedWarpFactor = warpFactor;
	_savedTransposeFactor = transposeFactor;
	_savedResonCF = resonCF;
	_savedResonBW = resonBW;
    _savedInterpMode = interpMode;
	_savedGainThresh = gainThresh;
}

void
FormantSynthesizer::restoreState() {
	LPC_Filter::reset();
	Buzz::restoreState();
	noise.setSeed(seed);
	if (pResonFilter)
	    pResonFilter->restoreState();
}

Status
FormantSynthesizer::doConfigure(Controller *) {
	setNeedUndo(true);	// Formant synth writes into target
	return Succeed;
}

double
FormantSynthesizer::doSynthesis() {
	RunStatus status;
	float amp = 0.0;
	float error = 0.0;
	float pitch = 0.0;
	float* coeffs = getFrame(&amp, &error, &pitch);
	int nharm = int(srOverTwo/pitch) - 2;
	double sqerr = sqrt(error);
	double noiseAmp = (sqerr - lowThreshold) / (hiThreshold - lowThreshold);
	// 0 <= noiseAmp <= 1.0
	noiseAmp = max(0.0, min(1.0, noiseAmp));	
	// complement of that
	double voicedAmp = unvoicedOnly ? 0.0 : 1.0 - noiseAmp;
	set(pitch * transposeFactor, nharm, voicedAmp * voicedFactor);
	noise.setGain(noiseAmp);
	double signal = unvoiced() + voiced();
	if (pResonFilter)
		(*pResonFilter)(&signal, 1);
	double outGain = (gainThresh > 0.0 && amp < gainThresh) ? 0.0 : gain;
	return filter(coeffs, amp * signal * outGain);
}
