/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* 
  Regex class implementation
 */

#ifdef __GNUG__
#pragma implementation
#endif
#include <ctype.h>
#include <new.h>
#include <builtin.h>
#include <string.h>
#include <stdlib.h>

#include <sys/types.h>
#include "MyRegex.h"

#define NMATCH 8

Regex::~Regex()
{
  regfree(buf);
  delete buf;
  delete [] pmatch;
}

Regex::Regex(const char* t)
{
  int tlen = (t == 0)? 0 : strlen(t);
  buf = new regex_t;
  pmatch = new regmatch_t[NMATCH];
  int status = regcomp(buf, t, REG_EXTENDED);	// flags?
  if (status != 0) {
    char errbuf[256];
    regerror(status, buf, errbuf, 256);
    (*lib_error_handler)("Regex", errbuf);
  }
}

int Regex::match_info(int& start, int& length, int nth) const
{
    if (nth >= NMATCH)
        return 0;
    start = pmatch[nth].rm_so;
    length = pmatch[nth].rm_eo - start;
    return start >= 0 && length >= 0;
}

int Regex::search(const char* s, int len, int& matchlen, int startpos) const
{
  int matchpos;
  int status = regexec(buf, s, NMATCH, pmatch, 0);	// flags?
  if (status == REG_NOERROR) {
    matchlen = pmatch[0].rm_eo - pmatch[0].rm_so;
	matchpos = pmatch[0].rm_so + startpos + 1;
  }
  else {
    matchlen = 0;
	matchpos = len;
  }
  return matchpos;
}

int Regex::match(const char*s, int len, int p) const
{
  if (p < 0)
  {
    p += len;
    if (p > len)
      return -1;
    return (regexec(buf, s, NMATCH, pmatch, 0) == REG_NOERROR) ? 0 : -1;
  }
  else if (p > len)
    return -1;
  else
    return (regexec(buf, s, NMATCH, pmatch, 0) == REG_NOERROR) ? 0 : -1;
}

int Regex::OK() const
{
// can't verify much, since we've lost the original string
  int v = buf != 0;             // have a regex buf
  if (!v) (*lib_error_handler)("Regex", "invariant failure");
  return v;
}

/*
 some built-in Regular expressions
*/

const Regex RXwhite("[ \n\t\r\v\f]+");
const Regex RXint("-?[0-9]+");
const Regex RXdouble("-?(([0-9]+.[0-9]*)|([0-9]+)|(.[0-9]+))([eE][-+]?[0-9]+)?");
const Regex RXalpha("[A-Za-z]+");
const Regex RXlowercase("[a-z]+");
const Regex RXuppercase("[A-Z]+");
const Regex RXalphanum("[0-9A-Za-z]+");
const Regex RXidentifier("[A-Za-z_][A-Za-z0-9_]*");

