/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include "MRI.h"

struct MRISubsampleData {
	MRI_Link *next;
	int	iwidth, owidth;
	int	iheight, oheight;
	int	xoff, yoff;
	int	nrows;
	int	freedata;
	struct MRI_ScanLine *acc;
#if 0
	double	*accR, *accG, *accB;
	unsigned long *accF;
#endif
	int	dx, dy;
};

void
MRISubsampleStart (void *private, int width, int height, int freedata)
{
	struct MRISubsampleData *wd = private;

	wd->iwidth = width;
	wd->owidth = width / wd->dx;
	if ((width % wd->dx) != 0) {
	  wd->xoff = (width - wd->owidth*wd->dx)/2;
	  if (wd->xoff & 1) wd->xoff--;
	}
	wd->iheight = height;
	wd->oheight = height / wd->dy;
	if ((height % wd->dy) != 0) {
	  wd->yoff = (height - wd->oheight*wd->dy)/2;
	  if (wd->yoff & 1) wd->yoff--;
	}
#if 0
	wd->accR = malloc (sizeof (double) * wd->owidth);
	wd->accG = malloc (sizeof (double) * wd->owidth);
	wd->accB = malloc (sizeof (double) * wd->owidth);
	wd->accF = malloc (sizeof (unsigned long) * wd->owidth);
	if (wd->accR == (double*)0 ||
	    wd->accG == (double*)0 ||
	    wd->accB == (double*)0 ||
	    wd->accF == (unsigned long *)0) {
		fprintf (stderr, "MRISubsampleStart: malloc failed\n");
		exit (1);
	}
#endif
	wd->nrows = 0;
	wd->freedata = freedata;
	(*wd->next->start) (wd->next->private, wd->owidth, wd->oheight, TRUE);
}

void
MRISubsampleRow (void *private, void *data)
{
	struct MRISubsampleData *wd = private;
	struct MRI_ScanLine *sl = data;
	int lineType = sl->scanLineType;
	int x, x1;

	if (lineType != LINETYPE_SHORT && lineType != LINETYPE_FLOAT) {
		fprintf (stderr, "MRISubsampleRow: unexpected line type %d\n", lineType);
		exit (1);
	}
	if (wd->yoff > 0) {
		if (wd->freedata) MRI_FreeScanLine (sl);
		wd->yoff--;
		return;
	}

	if (wd->nrows == 0) {
#if 1
		int i;
		wd->acc = MRI_NewScanLine (LINETYPE_DOUBLE, wd->owidth);
		for (i = 0; i < wd->acc->sl_NumChannels; i++) {
		    double *chan = (double*)(wd->acc->sl_Channel[i]);
		    for (x = 0; x < wd->owidth; x++)
		        chan[x] = 0.0;
		}
		for (x = 0; x < wd->owidth; x++)
		    ((unsigned long*)wd->acc->masks)[x] = 0;
#else
		for (x = 0; x < wd->owidth; x++) wd->accR[x] = 0.0;
		for (x = 0; x < wd->owidth; x++) wd->accG[x] = 0.0;
		for (x = 0; x < wd->owidth; x++) wd->accB[x] = 0.0;
		for (x = 0; x < wd->owidth; x++) wd->accF[x] = 0;
#endif
	}

	for (x = 0; x < wd->owidth; x++) {
		double aR, aG, aB;
		unsigned long aF;
		aR = aG = aB = 0.0;
		if (lineType == LINETYPE_SHORT)
		    for (x1 = 0; x1 < wd->dx; x1++) {
			aR += ((unsigned short *)(sl->R))[x*wd->dx+x1+wd->xoff];
			aG += ((unsigned short *)(sl->G))[x*wd->dx+x1+wd->xoff];
			aB += ((unsigned short *)(sl->B))[x*wd->dx+x1+wd->xoff];
		    }
		else
		    for (x1 = 0; x1 < wd->dx; x1++) {
			aR += ((float *)(sl->R))[x*wd->dx+x1+wd->xoff];
			aG += ((float *)(sl->G))[x*wd->dx+x1+wd->xoff];
			aB += ((float *)(sl->B))[x*wd->dx+x1+wd->xoff];
		    }
		((double *)(wd->acc->R))[x] += aR;
		((double *)(wd->acc->G))[x] += aG;
		((double *)(wd->acc->B))[x] += aB;
		aF = 0;
		for (x1 = 0; x1 < wd->dx; x1++)
		    aF |= sl->masks[x*wd->dx+x1+wd->xoff];
		wd->acc->masks[x] |= aF;
	}
	if (wd->freedata) MRI_FreeScanLine (sl);

	wd->nrows++;
	if (wd->nrows == wd->dy) {
		struct MRI_ScanLine *sl = MRI_NewScanLine (lineType, wd->owidth);
		double scaleRB = 4.0/(wd->dx * wd->dy);
		double scaleG = 2.0/(wd->dx * wd->dy);
		if (lineType == LINETYPE_SHORT)
		    for (x = 0; x < wd->owidth; x++) {
			double r;
			r = ((double *)(wd->acc->R))[x] * scaleRB;
			((unsigned short *)(sl->R))[x] = r < 65535.0 ? r : 65535;
			r = ((double *)(wd->acc->G))[x] * scaleG;
			((unsigned short *)(sl->G))[x] = r < 65535.0 ? r : 65535;
			r = ((double *)(wd->acc->B))[x] * scaleRB;
			((unsigned short *)(sl->B))[x] = r < 65535.0 ? r : 65535;

			sl->masks[x] = wd->acc->masks[x];
		    }
		else
		    for (x = 0; x < wd->owidth; x++) {
			double r;
			r = ((double *)(wd->acc->R))[x] * scaleRB;
			((float *)(sl->R))[x] = r;
			r = ((double *)(wd->acc->G))[x] * scaleG;
			((float *)(sl->G))[x] = r;
			r = ((double *)(wd->acc->B))[x] * scaleRB;
			((float *)(sl->B))[x] = r;

			sl->masks[x] = wd->acc->masks[x];
		    }
		(*wd->next->row) (wd->next->private, sl);
		wd->nrows = 0;
	}
}

void
MRISubsampleClose (void *private)
{
	struct MRISubsampleData *wd = private;

	(*wd->next->close) (wd->next->private);
	MRI_FreeScanLine (wd->acc);
	free (wd->next);
	free (wd);
}

MRI_Link *
MRI_GenSubsampler (int dx, int dy, MRI_Link *next)
{
	struct MRISubsampleData *wd = malloc (sizeof (struct MRISubsampleData));
	MRI_Link *ep = malloc (sizeof (*ep));

	if ((dx & 1) || (dy & 1)) {
		fprintf (stderr, "Error: dx and dy must be even\n");
		exit (1);
	}
	if (wd == NULL || ep == NULL) {
		fprintf (stderr, "Error: unable to allocate memory in subsample\n");
		exit (1);
	}
	ep->start = MRISubsampleStart;
	ep->row = MRISubsampleRow;
	ep->close = MRISubsampleClose;
	ep->private = wd;
	wd->next = next;
	wd->dx = dx;
	wd->dy = dy;
	wd->xoff = 0;
	wd->yoff = 0;
	return ep;
}
