#include <unistd.h>

#include "StopWatch.h"
#include "SDLTools.h"
#include "SDLFrameRate.h"
#include "Configuration.h"
#include "Tiles.h"
#include "ObjectBase.h"
#include "MenuManager.h"
#include "SoundMixer.h"
#include "GameBridge.h"
#include "PlayerConfiguration.h"


//----------------------------------------------------------------------------
namespace CL_OPT
{
    bool enableSound = true;
}


//----------------------------------------------------------------------------
void printVersion()
{
    std::cout << "Moagg - Mother of all gravity games" <<
    std::endl << " (c) 2003-2004 Bernhard Trummer, Andreas Granig" <<
    std::endl;
}

//----------------------------------------------------------------------------
void printHelp(const char* me)
{
    printVersion();

    std::cout << 
    std::endl << "Usage: " << me << " [-h] [-v] [-s]" << 
    std::endl << "  -h: prints this message" <<
    std::endl << "  -v: prints version" <<
    std::endl << "  -s: disables sound" <<
    std::endl;
}

//----------------------------------------------------------------------------
void readCommandLine(int argc, char** argv)
{
    int ch;
    char *argString = "hvs";

    while ((ch = getopt(argc, argv, argString)) != EOF)
    {
        switch(ch)
        {
        case 'h':
            printHelp(argv[0]);
            exit(0);
            break;

        case 'v':
            printVersion();
            exit(0);
            break;

        case 's':
            CL_OPT::enableSound = false;
            break;
        }
    }
}


//----------------------------------------------------------------------------
int main(int argc, char** argv)
{
    try
    {
        const char* theme = "moagg2";
        const char* layout = "menu.xml";


        srand(time(NULL));
        readCommandLine(argc, argv);

        Configuration::getInstance()->searchDataDir();

        std::cout << "Initializing configurations ... " << std::flush;
        PlayerConfiguration::create();
        std::cout << "done." << std::endl;

        Uint32 sdlflags = SDL_INIT_VIDEO | SDL_INIT_NOPARACHUTE;
        if (CL_OPT::enableSound) sdlflags |= SDL_INIT_AUDIO;

        std::cout << "Initializing SDL ..." << std::flush;
        SDL_CALLS::Init(sdlflags);
        std::cout << "done." << std::endl;

        if (CL_OPT::enableSound)
        {
            std::cout << "Initializing sound system ..." << std::flush;
            try
            {
                SoundMixer::init();
                std::cout << "done." << std::endl;
            }
            catch (Exception &e)
            {
                CL_OPT::enableSound = false;
                std::cout << "failed." << std::endl;
                std::cout << "  (" << e << ")" << std::endl;
            }
        }

        std::cout << "Initializing tile structures ... " << std::flush;
        Tiles::init();
        std::cout << "done." << std::endl;

        std::cout << "Initializing surfaces ... " << std::flush;
        SurfacesBase::init();
        std::cout << "done." << std::endl;
    
        std::cout << "Initializing GUI ... " << std::flush;
        MenuManager::init(layout, theme);
        std::cout << "done." << std::endl;

        GameInterface::setInstance(GameBridge::getInstance());
    }
    catch (Exception &e)
    {
        std::cout << "Error initializing the game!" << std::endl
                  << e << "." << std::endl;
        return -1;
    }

    SDLFrameRate::setFrameRate(72);  // Always use 72 fps.

    PG_Application::GetApp()->Run();

    MenuManager::destroy();

    PlayerConfiguration::destroy();

    SurfacesBase::destroy();

    Tiles::destroy();

    if (CL_OPT::enableSound)
    {
        SoundMixer::destroy();
    }

    SDL_Quit();

    return 0;
}
