// -*- c++ -*-
#ifndef INCLUDED_MATH3D_M3D_H
#define INCLUDED_MATH3D_M3D_H
/*
 * Math3d - The 3D Computer Graphics Math Library
 * Copyright (C) 1996-2000 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: m3d.h,v 1.6 2000/10/09 12:17:54 jeh Exp $
 */

#ifndef INCLUDED_MATH3D_MATH3DDEF_H
#include <math3d/math3ddef.h>
#endif

namespace Math3d {
  
  class M2d;
  class M4d;
  class MQuat;
  class M4x4;

/** 
 * 3-dimensional vector class.
 */
  class _CCMATH3D M3d {
    private:
      double d_v[3];
    public:
      M3d() {d_v[0]=d_v[1]=d_v[2]=0.0;};
      M3d(double x, double y, double z) {d_v[0]=x; d_v[1]=y; d_v[2]=z;};
      M3d(const M2d& A);
      M3d(const M3d& A);
      M3d(const M4d& A);
      const M3d& operator=(const M3d& A);
      
      void zero();
      void set(double x, double y, double z) {d_v[0]=x; d_v[1]=y; d_v[2]=z;};
      void copy(const M3d& A);

      inline double& operator[](int i);
      operator double*() {return(d_v);}
      double& x() {return(d_v[0]);}
      double& y() {return(d_v[1]);}
      double& z() {return(d_v[2]);}
      double& get(int i);

      M3d operator+(const M3d& A);
      M3d operator+();
      const M3d& operator+=(const M3d& A);
      M3d operator-(const M3d& A);
      M3d operator-();
      const M3d& operator-=(const M3d& A);
      M3d operator*(double k);
      const M3d& operator*=(double k);
      void neg();
      void abs();
      void add(const M3d& A, const M3d& B);
      void sub(const M3d& A, const M3d& B);
      void scalar(double k);
      void normalize();
      void cartesianize();
      M2d cartesianized();
      void rationalize();
      void homogenize();
      void cross(const M3d& A, const M3d& B);
      void lerp(const M3d& A, const M3d& B, double t);
      void normal(const M3d& A, const M3d& B, const M3d& C);
      void min(const M3d& m);
      void max(const M3d& m);
      void cubic(const M3d& A, const M3d& TA, const M3d& TB, 
        const M3d& B, double t);

      inline double operator[](int i) const;
      operator const double*() const {return(d_v);}
      double x() const {return(d_v[0]);}
      double y() const {return(d_v[1]);}
      double z() const {return(d_v[2]);}
      double get(int i) const;
      
      double operator*(const M3d& A)  const;  //dot product!
      bool operator==(const M3d& A) const;
      bool operator!=(const M3d& A) const;
      double dot(const M3d& A) const;
      bool cmp(const M3d& A, double epsilon=EPSILON) const;
      double squared() const;
      double length() const;

      friend M2d;
      friend M4d;
      friend MQuat;
      friend M4x4;
  };

  inline double&
  M3d::operator[](int i)
  {
    ASSERT(i>=0 && i<3);
    return(d_v[i]);
  }
  inline double
  M3d::operator[](int i) const
  {
    ASSERT(i>=0 && i<3);
    return(d_v[i]);
  }
  
  extern _CCMATH3D ostream& operator << (ostream& co, const M3d& v);

}
#endif // INCLUDED_MATH3D_M3D_H







