/***************************************************************************
  $RCSfile: ddvcard.cpp,v $
  -------------------
  cvs         : $Id: ddvcard.cpp,v 1.8 2003/06/24 16:50:26 aquamaniac Exp $
  begin       : Mon Apr 14 2003
  copyright   : (C) 2003 by Martin Preuss
  email       : martin@libchipcard.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif



#include "ddvcard.h"
#include "mediumddv.h"
#include <string.h>


namespace HBCI {



DDVCardPlugin::DDVCardPlugin(const API *api)
    :MediumPlugin()
,_api(api)
{

}


DDVCardPlugin::~DDVCardPlugin(){
}


Error DDVCardPlugin::mediumCheck(string &name) const {
#if (k_CHIPCARD_VERSION_MAJOR>0) || (k_CHIPCARD_VERSION_MINOR>8)
  Pointer<DDVCardTrader> trader;
  Pointer<DDVCard> hcard;
  CTCard *cardp;
  CTError err;
  Error herr;
  bool havecard;

  trader=new DDVCardTrader(_api,
			   false,
			   0,
			   0,
			   CHIPCARD_STATUS_INSERTED,
			   CHIPCARD_STATUS_INSERTED |
			   CHIPCARD_STATUS_LOCKED_BY_OTHER,
			   CHIPCARD_STATUS_INSERTED);
  err=trader.ref().start();
  if (!err.isOk()) {
    fprintf(stderr, "DDVCardPlugin: Could not initialize trader");
    return Error("DDVCardPlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_MEDIUM,
		 ERROR_ADVISE_ABORT,
		 "could not start trader");
  }

  havecard=false;
  while (!havecard) {
    // try for 5 seconds to get a card
    err=trader.ref().getNext(cardp, 5);
    if (!err.isOk()) {
      if (err.code()==k_CTERROR_API &&
	  (err.subcode1()==CHIPCARD_ERROR_NO_TRANSPORT ||
	   err.subcode1()==CHIPCARD_ERROR_NO_REQUEST)) {
	fprintf(stderr,
		"DDVCardPlugin: Service unreachable, maybe \"chipcardd\" is not running?\n");
	trader.ref().stop();
	return Error("DDVCardPlugin::mediumCheck",
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_NO_CARD,
		     ERROR_ADVISE_ABORT,
		     "service unreachable");
      }

      if (!_api->interactor().ref().
	  msgInsertMediumOrAbort(0, MediumTypeCard)) {
	if (Hbci::debugLevel()>1)
	  fprintf(stderr,"DDVCardPlugin: User aborted.\n");
	trader.ref().stop();
	return Error("DDVCardPlugin::mediumCheck",
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_USER_ABORT,
		     ERROR_ADVISE_ABORT,
		     "No card, user aborted.");
      }
    } // if err is not ok
    else
      havecard=true;
  } // while !havecard

  trader.ref().stop();

  // ok, we have a card, now check it
  hcard=new DDVCard(*cardp, _api);
  err=hcard.ref().openCard();
  if (!err.isOk()) {
    fprintf(stderr,"DDVCardPlugin: Error opening card: %s\n",
	    err.errorString().c_str());
    return Error("DDVCardPlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_BAD_MEDIUM,
		 ERROR_ADVISE_ABORT,
		 "This card is not my type.");
  }
  if (name.empty()) {
    // no name given, so create one
    name=hcard.ref().cardNumber();
  }
  else {
    // name given, so compare against that stored on the card (serial number)
    if (name!=hcard.ref().cardNumber()) {
      hcard.ref().closeCard();
      return Error("DDVCardPlugin::mediumCheck",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_WRONG_MEDIUM,
		   ERROR_ADVISE_ABORT,
		   "Serial number does not match.");
    }
  }

  // close the card
  err=hcard.ref().closeCard();
  if (!err.isOk()) {
    fprintf(stderr,"DDVCardPlugin: Error closing card: %s\n",
	    err.errorString().c_str());
    return Error("DDVCardPlugin::mediumCheck",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_MEDIUM,
		 ERROR_ADVISE_ABORT,
		 "Card error.");
  }

  return Error();
#else
  return Error("DDVCardPlugin::mediumCheck",
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_UNSUPPORTED,
	       ERROR_ADVISE_ABORT,
	       "Need newer Libchipcard version for this to work");
#endif
}


Pointer<Medium> DDVCardPlugin::mediumFactory(const string &name){
  Pointer<Medium> m;

  m=_api->findMedium(name);
  if (m.isValid())
    return m;
  m=new MediumDDV(_api, name);

  return m;
}


Pointer<Medium> DDVCardPlugin::createNewMedium(bool readonly,
					       int country,
					       const string &bankId,
					       const string &userid,
					       const string &name){
  Pointer<Medium> m;

  m=new MediumDDV(_api, name);
  return m;
}


Pointer<Medium> DDVCardPlugin::mediumFromConfig(SimpleConfig &cfg,
						cfgPtr group){
  string typ;
  string name;
  Pointer<Medium> m;
  Error err;
  cfgPtr pgroup;

  typ=cfg.getVariable("mediumtype","",group);
  if (strcasecmp(typ.c_str(), mediumTypeName().c_str())!=0) {
    throw Error("DDVCardPlugin::mediumFromConfig",
		ERROR_LEVEL_NORMAL,
		HBCI_ERROR_CODE_INVALID,
		ERROR_ADVISE_DONTKNOW,
		"bad medium type");
  }
  name=cfg.getVariable("mediumname",
		       cfg.getVariable("cardnumber","",group),
		       group);
  m=new MediumDDV(_api, name);

  // load properties
  pgroup=cfg.findGroup("properties", group);
  if (pgroup.isValid()) {
    // set property "loglevel"
    err=m.ref().setProperty("loglevel",
			    cfg.getVariable("loglevel",
					    "error",
					    pgroup));
    if (!err.isOk())
      throw Error("DDVCardPlugin::mediumFromConfig", err);

    // set property "usekeypad"
    err=m.ref().setProperty("usekeypad",
			    cfg.getVariable("usekeypad",
					    "yes",
					    pgroup));
    if (!err.isOk())
      throw Error("DDVCardPlugin::mediumFromConfig", err);
  } // if properties

  return m;
}


Error DDVCardPlugin::mediumToConfig(Pointer<Medium> m,
				    SimpleConfig &cfg,
				    cfgPtr group){
  Pointer<MediumDDV> mddv;
  string s;
  Error err;
  cfgPtr pgroup;

  try {
    mddv=m.cast<MediumDDV>();
  }
  catch (Error xerr) {
    return Error("DDVCardPlugin::mediumToConfig",xerr);
  }
  cfg.setVariable("mediumtype",m.ref().mediumTypeName(),group);
  cfg.setVariable("mediumname",mddv.ref().mediumName(),group);

  // save properties
  pgroup=cfg.createGroup("properties", group);
  if (!pgroup.isValid()) {
    return HBCI::Error("DDVCardPlugin::mediumToConfig",
		       ERROR_LEVEL_NORMAL,
		       HBCI_ERROR_CODE_MEDIUM,
		       ERROR_ADVISE_DONTKNOW,
		       "could not create properties group");
  }

  // save property "loglevel"
  err=m.ref().getProperty("loglevel", s);
  if (!err.isOk())
    return Error("DDVCardPlugin::mediumToConfig", err);
  cfg.setVariable("loglevel", s, pgroup);

  // save property "usekeypad"
  err=m.ref().getProperty("usekeypad", s);
  if (!err.isOk())
    return Error("DDVCardPlugin::mediumToConfig", err);
  cfg.setVariable("usekeypad", s, pgroup);

  return Error();
}


Error DDVCardPlugin::checkVersion(const API *api) {
  int vmajor, vminor, vpl, vbuild;

  /* do some version checking
   * Please note that the constants used below are defined upon
   * compilation of this plugin. If you compile this plugin using
   * OpenHBCI 0.9.10 and try to use it on a system where OpenHBCI 0.9.9
   * is installed, this module will refuse to work. This plugin only
   * works with versions of OpenHBCI which are the same or newer than
   * the version used to compile this plugin.
   * If the major version differs then this module believes that the
   * version of OpenHBCI used is totally incompatible with the version
   * used to build this plugin.
   */
  api->libraryVersion(vmajor, vminor, vpl, vbuild);
  if (vmajor!=OPENHBCI_VERSION_MAJOR) {
    /* this is the major compatibility check. The major version is
     * incremented if a newer version is incompatible with older ones */
    fprintf(stderr,
	    " Different major versions, "
	    "please recompile DDVCard plugin.\n");
    return HBCI::Error("DDVCard Plugin",
		       ERROR_LEVEL_NORMAL,
		       HBCI_ERROR_CODE_MEDIUM,
		       ERROR_ADVISE_DONTKNOW,
		       "Major version does not match");
  }
  if (!(
	((vminor==OPENHBCI_VERSION_MINOR)&&(vpl>OPENHBCI_VERSION_PATCHLEVEL))
	||
	((vminor==OPENHBCI_VERSION_MINOR) &&
	 (vpl==OPENHBCI_VERSION_PATCHLEVEL) &&
	 (vbuild>=OPENHBCI_VERSION_BUILD))
       )
     ) {
    fprintf(stderr,
	    "This plugin needs OpenHBCI v"OPENHBCI_VERSION_FULL_STRING
	    " or newer.\n");
    return HBCI::Error("RDHFile Plugin",
		       ERROR_LEVEL_NORMAL,
		       HBCI_ERROR_CODE_MEDIUM,
		       ERROR_ADVISE_DONTKNOW,
		       "need OpenHBCI v"OPENHBCI_VERSION_FULL_STRING
		       " or newer");
  }

  return Error();
}



}; // namespace


extern "C" {

  int ddvcard_openhbci_plugin_interface_version = OPENHBCI_PLUGIN_INTERFACE_VERSION;

  int ddvcard_openhbci_module_current=DDVCARD_MODULE_CURRENT;
  int ddvcard_openhbci_module_revision=DDVCARD_MODULE_REVISION;
  int ddvcard_openhbci_module_age=DDVCARD_MODULE_AGE;


  HBCI::Error ddvcard_registerYourSelf(HBCI::API *api) {
    HBCI::Pointer<HBCI::MediumPlugin> mp;
    HBCI::Error err;

    err=HBCI::DDVCardPlugin::checkVersion(api);
    if (!err.isOk())
      return HBCI::Error("DDVCard Plugin",err);

    // version is ok, now register plugin
    mp=new HBCI::DDVCardPlugin(api);
    return api->registerMediumPlugin(mp);
  }

  /*
   * This function simply creates a medium plugin without registering it
   */
  HBCI::Pointer<HBCI::MediumPlugin> ddvcard_createPlugin(const HBCI::API *api){
    HBCI::Pointer<HBCI::MediumPlugin> mp;
    HBCI::Error err;

    err=HBCI::DDVCardPlugin::checkVersion(api);
    if (!err.isOk())
      throw HBCI::Error("DDVCard Plugin",err);

    mp=new HBCI::DDVCardPlugin(api);
    mp.setDescription("DDVCardPlugin");
    return mp;
  }


}; // extern "C"



