/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.heartbeat;

import java.rmi.RemoteException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;

import javax.management.ObjectName;

import mx4j.tools.connector.RemoteMBeanServer;
import mx4j.log.Log;
import mx4j.log.Logger;

/**
 *
 * @author <a href="mailto:mgore@users.sourceforge.net">Michael Gorelik</a>
 * @version $Revision: 1.6 $
 */
public class HeartBeat implements HeartBeatMBean
{
	private HashMap m_listeners = new HashMap();
	private String m_name = HeartBeatMBean.HEARTBEAT_OBJECT_NAME;
	private int m_period = HeartBeatMBean.DEFAULT_PERIOD;
	private int m_retries = HeartBeatMBean.DEFAULT_RETRIES;

	private ObjectName m_objectName;
	private HeartBeatThread m_hbThread = new HeartBeatThread();

	private Thread m_t;
	private boolean m_started[] = new boolean[1];

	public HeartBeat(String name)
	{
		m_name = name;
		try
		{
			m_objectName = new ObjectName(m_name);
		}
		catch (Exception ex)
		{
		}

		Properties env = System.getProperties();
		String p = env.getProperty(TRIES_PROP);
		if (p != null) m_retries = (new Integer(p)).intValue();

		p = env.getProperty(PERIOD_PROP);
		if (p != null) m_period = (new Integer(p)).intValue();

		m_started[0] = false;
	}

	// When this method returns it guarantees that heartbeat thread has started
	public void start()
	{
		// start heartbeat thread
		m_t = new Thread(m_hbThread, m_name);
		m_t.start();

		// synchronize with the thread
		synchronized (m_started)
		{
			while (!m_started[0])
			{
				try
				{
					m_started.wait();
				}
				catch (Exception e)
				{
				}
			}
		}
	}

	public void stop()
	{
		Logger logger = getLogger();

		if (logger.isEnabledFor(Logger.TRACE)) logger.trace(getClass().getName() + ".stop");

		// prepare to stop
		m_hbThread.stop();

		// interrupt HeartBeat thread
		m_t.interrupt();

		Thread.currentThread().yield();
	}

	/**
	 * Adds the specified heartbeat listener to receive heartbeat notifications from
	 * this HeartBeatMBean.
	 */
	public void addHeartBeatListener(String listenerName, Object connectorType, Object listenerAddress)
	{
		Logger logger = getLogger();
		if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("HeartBeat.addHeartBeatListener: connType=" + connectorType);

		try
		{
			RemoteMBeanServer conn = HeartBeatConnectorFactory.getFactory().getConnector(connectorType, listenerAddress);

			HeartBeatSession session = new HeartBeatSession(conn);

			synchronized (m_listeners)
			{
				m_listeners.put(listenerName, session);
			}
		}
		catch (ConnectorException ex)
		{
			// FIXME:
			if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("HeartBeat.addHeartBeatListener exception", ex);
		}
	}


	/**
	 * Removes the specified heartbeat listener so that it no longer receives
	 * heartbeat notifications from this HeartBeatMBean.
	 */
	public void removeHeartBeatListener(String heartBeatListenerName)
	{
		synchronized (m_listeners)
		{
			m_listeners.remove(heartBeatListenerName);
		}
	}

	// Getter/Setter for the heartbeat period in milliseconds.
	public int getPeriod()
	{
		return m_period;
	}

	public void setPeriod(int period)
	{
		m_period = period;
	}

	// Getter/Setter for the number of retries.
	public int getRetries()
	{
		return m_retries;
	}

	public void setRetries(int nretries)
	{
		this.m_retries = nretries;
	}

	private Logger getLogger()
	{
		return Log.getLogger(getClass().getName());
	}

	private class HeartBeatThread implements Runnable
	{
		private boolean m_stop = false;

		public void run()
		{
			// notify creator that this thread is running
			synchronized (m_started)
			{
				m_started[0] = true;
				m_started.notifyAll();
			}

			Logger logger = getLogger();

			while (true)
			{
				HeartBeatSession sess = null;
				String listener = null;
				RemoteMBeanServer conn;
				Object[] params = new Object[1];
				String[] signature = new String[1];
				params[0] = m_objectName.getCanonicalName();
				signature[0] = "java.lang.String";

				try
				{
					HashMap listeners;
					synchronized (m_listeners)
					{
						listeners = (HashMap)m_listeners.clone();
					}

					// loop thru currently registered listeners
					Set keys = listeners.keySet();
					for (Iterator it = keys.iterator(); it.hasNext();)
					{
						listener = (String)it.next();
						sess = (HeartBeatSession)listeners.get(listener);
						conn = sess.getConnection();
						if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("HeartBeatThread.run: calling " + listener);
						try
						{
							conn.invoke(new ObjectName(listener), "processHeartBeat", params, signature);
							sess.reset();
						}
						catch (RemoteException ex)
						{
							// remote invocation has failed. Should we continue?
							if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("HeartBeatThread.run: ConnectorException listnr=" + listener);

							if (!sess.shouldContinue())
							{
								// remove this listener
								synchronized (m_listeners)
								{
									m_listeners.remove(listener);
									if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("HeartBeatThread.run: removed listenr=" + listener);
								}

								listeners.remove(listener);
							}
						} // catch
					} // for (Iterator it = keys.iterator(); it.hasNext();)
				}
				catch (Exception ex)
				{
					if (logger.isEnabledFor(Logger.DEBUG)) logger.debug("HeartBeatThread.run exception", ex);
				}

				try
				{
					Thread.sleep(1000 * m_period);
				}
				catch (Exception ex)
				{
				}

				if (m_stop)
				{
					return;
				}
			} // while(true)
		} // run

		public void stop()
		{
			m_stop = true;
		}
	} // HeartBeatThread

	// utility class to hold info for heartbeat session
	private class HeartBeatSession
	{
		private RemoteMBeanServer m_conn;
		private int m_failCount = 0;

		public HeartBeatSession(RemoteMBeanServer conn)
		{
			m_conn = conn;
		}

		public void reset()
		{
			m_failCount = 0;
		}

		public boolean shouldContinue()
		{
			m_failCount++;

			if (m_failCount > m_retries)
			{
				return false;
			}
			return true;
		}

		public RemoteMBeanServer getConnection()
		{
			return m_conn;
		}
	} // private class HeartBeatSession
}
