/*
 *  client/functions-write.c
 *		The functions to actually execute the commands
 *		for the client.  Commands for writing texts.
 *
 *
 *  Copyright (C) 1990	Lysator Computer Club,
 *			Linkoping University,  Sweden
 *
 *  Everyone is granted permission to copy, modify and redistribute
 *  this code, provided the people they give it to can.
 *
 *
 *  Author:	Thomas Bellman
 *		Lysator Computer Club
 *		Linkoping University
 *		Sweden
 *
 *  email:	Bellman@Lysator.LiU.SE
 *
 *
 *  Any opinions expressed in this code are the author's PERSONAL opinions,
 *  and does NOT, repeat NOT, represent any official standpoint of Lysator,
 *  even if so stated.
 */

#include <config.h>

#include <stdio.h>
/* #include <time.h> included from kom-types.h */
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#if STDC_HEADERS || HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include <errno.h>

#ifdef __vax__
#define __SIZE_T included from string.h
#endif

#include <libintl.h>

#include <kom-types.h>	/* Before zmalloc.h */
#include <zmalloc.h>
#include <s-string.h>
#include <s-collat.h>

#include <kom-errno.h>
#include <services.h>

#include "conf.h"

#include "copy.h"
#include "xoutput.h"
#include "read-line.h"
#include "commands.h"
#include "parser.h"
#include "edit.h"
#include "error.h"
#include "cache.h"
#include "quit.h"

#include "internal.h"
#include "cmds.h"
#include "offline.h"

#include "defaults.h"

/* This is to emphasize that a function is exported. */
#define Export


#define CHECK_ACTIVE_TEXT() if (active_text < 0) { \
   xprintf(gettext("Du hller inte p att skriva ngot inlgg\n")); RETURN(FAILURE); }


/*  Write a new text with the current conference as recipient  */
Export  Success
cmd_write_new_text (String   argument)
{
#define CLEAN_UP()				\
    {						\
	s_clear (&argument);			\
    }

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    if (!ccn)
    {
	xprintf(gettext("Du r inte nrvarande i ngot mte just nu.\n"));
	RETURN (FAILURE);
    }

    /*  Create a new text-struct and find a slot for it in the table  */

    active_text = edit_create_new_text();
    /* BUG! Check for -1 in return */
    edit_add_recipient (active_text, ccn);
    /* BUG! Check for FAILURE! */

    newline();
    xprintf (gettext("Tryck Ctrl-D fr att spara eller Esc fr att anvnda andra kommandon\n"));
    xprintf (gettext("rende: "));

    RETURN (do_continue_writing (active_text));

#undef CLEAN_UP
}   /* END: write_new_text() */



/*  Write a comment to a text  */
Export  Success
cmd_comment_text (String   argument)
{
    Text_no		text_to_comment = (Text_no) 0;

    String_size		error_check;
    Text_stat	      * text_stat_commented = NULL;
    String		text = EMPTY_STRING_i;
    String		subject = EMPTY_STRING_i;

    Kom_err		err;

#define CLEAN_UP()					\
    {							\
	s_clear (&argument);				\
	if (text_stat_commented)			\
	{						\
	    release_text_stat(text_stat_commented);	\
	    zfree(text_stat_commented);			\
	}						\
	s_clear (&text);				\
	s_clear (&subject);				\
    }

    TOPLOOP_SETJMP();

    if (s_empty(argument)) 
    {
	text_to_comment = reading_state.last_viewed;
    }
    else
    {
	argument = s_strip_trailing(argument, command_separators);
    
	/* This should be a number, and nothing else. */
	text_to_comment = (Text_no) s_strtol(argument, &error_check,
					     DEFAULT_NUMBER_BASE);
	if (error_check < s_strlen(argument))
        {
	    /* Wasn't a number. This is an error. */
	    /*xgettext:c-format*/
	    xprintf (gettext("Oknt argument: %S\n"),argument);
	    RETURN (FAILURE);
	}
    }

    if (text_to_comment
	&& (text_stat_commented = text_stat (text_to_comment)) != NULL
	&& (text = get_text(text_to_comment,
			    0, LONGEST_SUBJECT_NOT_TRUNCATED, &err)).len > 0
	&& (err == KOM_NO_ERROR))
    {
	Misc_info     * misc;
	int		r;
	String_size	pos;

	active_text = edit_create_new_text();

	for(misc = text_stat_commented->misc_items, r = 0;
	    r < (int)text_stat_commented->no_of_misc;
	    r++, misc++) 
	{
	    if (misc->type == recpt) 
	    {
		edit_add_recipient(active_text, misc->datum.recipient);
	    }
	}
	edit_add_commenting (active_text, text_to_comment);

	if ((pos = s_strchr(text, '\n', 0)) == -1) 
	{
	    pos = END_OF_STRING;
	}
	s_substr(&subject,text, 0, pos);
	edit_add_string(active_text, subject);
    }
    else
    {
	xprintf(gettext("Jag frstr inte vilken text du vill kommentera.\n"));
	RETURN (FAILURE);
    }
	
    newline();
    xprintf(gettext("Tryck Ctrl-D fr att spara eller Esc fr att anvnda andra kommandon\n"));
    xprintf(gettext("rende: %S"), *edit_text(active_text));

    RETURN (do_continue_writing (active_text));
#undef CLEAN_UP
}


/*  Write a footnote to a text  */
Export  Success
cmd_footnote_text (String   argument)
{
    Text_no		text_to_comment = (Text_no) 0;

    String_size		error_check;
    Text_stat	        text_stat_commented = EMPTY_TEXT_STAT;
    String		text = EMPTY_STRING_i;
    String		subject = EMPTY_STRING_i;

#define CLEAN_UP()				\
    {						\
	s_clear (&argument);			\
	release_text_stat(&text_stat_commented);		\
	s_clear (&text);			\
	s_clear (&subject);			\
    }

    TOPLOOP_SETJMP();

    OFFLINE();

    if (s_empty(argument)) 
    {
	text_to_comment = reading_state.last_viewed;
    } else {
	argument = s_strip_trailing (argument, command_separators);
    
	/* This should be a number, and nothing else. */
	text_to_comment = (Text_no) s_strtol (argument, &error_check,
					      DEFAULT_NUMBER_BASE);
	if (error_check < s_strlen (argument))
        {
	    /* Wasn't a number. This is an error. */
	    /*xgettext:c-format*/
	    xprintf (gettext("Oknt argument: %S\n"),argument);
	    RETURN (FAILURE);
	}
    }

    if (text_to_comment &&
	kom_get_text_stat (text_to_comment,
			   &text_stat_commented) != FAILURE &&
	kom_get_text (text_to_comment,
		      0, LONGEST_SUBJECT_NOT_TRUNCATED, &text) != FAILURE
	/* BUG: No error-check. */
	)
    {
	Misc_info     * misc;
	int		r;
	String_size	pos;

	if (text_stat_commented.author != cpn)
	{
	  xprintf(gettext("Du fr bara skriva en fotnot till texter du har skrivit\n"));
	  RETURN (FAILURE);
	}

	active_text = edit_create_new_text();

	for(misc = text_stat_commented.misc_items , r = 0;
	    r < (int)text_stat_commented.no_of_misc;
	    r++, misc++) 
	{
	    if (misc->type == recpt) 
	    {
		edit_add_recipient (active_text, misc->datum.recipient);
	    }
	}
	edit_add_footnote (active_text, text_to_comment);

	if ((pos = s_strchr(text, '\n', 0)) == -1) 
	{
	    pos = END_OF_STRING;
	}
	s_substr(&subject,text, 0, pos);
	edit_add_string(active_text, subject);
    } else {
	xprintf (gettext("Jag frstr inte vilken text du vill kommentera.\n"));
	RETURN (FAILURE);
    }
	
    newline();
    xprintf (gettext("Tryck Ctrl-D fr att spara eller Esc fr att anvnda andra kommandon\n"));
    xprintf (gettext("rende: %S"), *edit_text(active_text));

    RETURN (do_continue_writing (active_text));
#undef CLEAN_UP
}


/*  Reply directly (and privately) to the author of a text  */
Export  Success
cmd_private_comment (String   argument)
{
    Text_no		text_to_comment = (Text_no) 0;

    String_size		error_check;
    Text_stat	        text_stat_commented = EMPTY_TEXT_STAT;
    String		text = EMPTY_STRING_i;
    String		subject = EMPTY_STRING_i;

#define CLEAN_UP()				\
    {						\
	s_clear (&argument);			\
	release_text_stat(&text_stat_commented);		\
	s_clear (&text);			\
	s_clear (&subject); 			\
    }

    TOPLOOP_SETJMP();

    OFFLINE();

    if (s_empty(argument)) 
    {
	text_to_comment = reading_state.last_viewed;
    } else {
	argument = s_strip_trailing (argument, command_separators);
    
	/* This should be a number, and nothing else. */
	text_to_comment = (Text_no) s_strtol (argument, &error_check,
					      DEFAULT_NUMBER_BASE);
	if (error_check < s_strlen (argument))
        {
	    /* Wasn't a number. This is an error. */
	    /*xgettext:c-format*/
	    xprintf (gettext("Oknt argument: %S\n"),argument);
	    RETURN (FAILURE);
	}
    }

    if (text_to_comment &&
	kom_get_text_stat (text_to_comment,
			   &text_stat_commented) != FAILURE &&
	kom_get_text (text_to_comment,
		      0, LONGEST_SUBJECT_NOT_TRUNCATED, &text) != FAILURE
	/* BUG: No error-check. */
	)
    {
	String_size	pos;

	active_text = edit_create_new_text();

	edit_add_recipient (active_text, text_stat_commented.author);
	edit_add_recipient (active_text, cpn);
	edit_add_commenting (active_text, text_to_comment);

	if ((pos = s_strchr(text, '\n', 0)) == -1) 
	{
	    pos = END_OF_STRING;
	}
	s_substr(&subject,text, 0, pos);
	edit_add_string(active_text, subject);
    } else {
	xprintf (gettext("Jag frstr inte vilken text du vill kommentera.\n"));
	RETURN (FAILURE);
    }
	
    newline();

    xprintf (gettext("rende: %S"), *edit_text(active_text));
    xprintf(gettext("Tryck Ctrl-D fr att spara eller Esc fr att anvnda andra kommandon\n"));
    RETURN (do_continue_writing (active_text));
#undef CLEAN_UP
}




/*  Write a letter to a person (or a conference)  */
Export  Success
cmd_write_letter (String   recipient_name)
{
    Conf_z_info		  recipient_conf = EMPTY_CONF_Z_INFO;

#define CLEAN_UP()				\
    {						\
	s_clear (&recipient_name); 		\
	release_conf_z_info(&recipient_conf);	\
    }
    

    TOPLOOP_SETJMP();

    OFFLINE();

    /* BUG:  Should be able to specify more than one recipient */
    recipient_conf = ask_for_conf (&recipient_name, 
				   gettext("Till vem eller vilket mte vill du skicka brevet?\n"));

    if (recipient_conf.conf_no == 0)
    {
	RETURN (FAILURE);
    }
    active_text = edit_create_new_text();

    edit_add_recipient (active_text, recipient_conf.conf_no);
    edit_add_recipient (active_text, cpn);
    
    newline();
    xprintf(gettext("Tryck Ctrl-D fr att spara eller Esc fr att anvnda andra kommandon\n"));
    xprintf (gettext("rende: "));

    RETURN (do_continue_writing (active_text));

#undef CLEAN_UP
}


/*
 *  Start a raw editing of a text.
 *  No recipients are added, no nothing.
 *  This is the one used by the "redigera med editor" command.
 */
Export  Success
cmd_plain_write_new_text (String argument)
{
#define CLEAN_UP()				\
    {						\
	s_clear(&argument);			\
    }

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    active_text = edit_create_new_text();

    newline();
    xprintf(gettext("Tryck Ctrl-D fr att spara eller Esc fr att anvnda andra kommandon\n"));
    xprintf (gettext("rende: "));

    RETURN (do_continue_writing (active_text));

#undef CLEAN_UP
}


/*
 *  Remove the current edited text, i e the user didn't really
 *  want to write the text.
 */
Export  Success
cmd_remove_text (String	argument)
{
#define CLEAN_UP()				\
    {						\
	s_clear(&argument);			\
    }

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    CHECK_ACTIVE_TEXT();

    active_text = edit_dealloc_text (active_text);

    RETURN (OK);

#undef  CLEAN_UP
}



Export  Success
cmd_enter_text (String	argument)
{
#define FUNCTION "cmd_enter_text()"

    Text_no		  number_of_new_text;
    Text_stat		* stat = NULL;

#define CLEAN_UP()     				\
    {						\
	s_clear (&argument);			\
	zfree (stat);				\
    }


    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    CHECK_ACTIVE_TEXT();

    if (offline)
    {
	if (store_created_text(s_strip_trailing(edited_texts[active_text].text,
						s_fcrea_str(" \t\n\r")),
			       edited_texts[active_text].no_of_miscs,
			       edited_texts[active_text].miscs)
	    != OK)
	{
	    xprintf(gettext("\nTexten kunde inte sparas.\n"));
	    return FAILURE;
	}

	xprintf("\nTexten sparad fr senare uppladdning.\n");
	active_text = edit_dealloc_text(active_text);
    }
    else
    {
	number_of_new_text =
	    kom_create_text(s_strip_trailing(edited_texts[active_text].text,
					     s_fcrea_str(" \t\n\r")),
			    edited_texts[active_text].no_of_miscs,
			    edited_texts[active_text].miscs);
	if (number_of_new_text == 0)
	{
	    xprintf(gettext("\nTexten kunde inte lggas in. Felet var\n"));
	    /* BUG! More error codes must be checked for */
	    switch (kom_errno)
	    {
	    case KOM_NO_CONNECT:
		fatal1(CLIENT_SERVER_ERROR,
		       "create_text() failed");
		break;

	    case KOM_INDEX_OUT_OF_RANGE:
		xprintf(gettext("Du har inte rttigheter att skriva hr"));
		RETURN (FAILURE);

	    default:
		fatal3(CLIENT_UNEXPECTED_SERVER_ERROR,
		       "create_text() failed. Lyskom-errno: %d",
		       kom_errno, kom_errno_string());
		break;
	    }

	    RETURN (FAILURE);
	}

	/* Mark the text as read. */
	mark_global_as_read(number_of_new_text, 0);

	xprintf(gettext("\nText %d inlagd.\n"), number_of_new_text);
	active_text = edit_dealloc_text(active_text);

	reading_state.last_viewed = number_of_new_text;
    }

    RETURN (OK);

#undef CLEAN_UP
#undef FUNCTION
}



/* Continue writing a text */
Export  Success
cmd_continue_writing (String	  argument)

{
    int			  pos, start_of_line;
    String		  temp;
    Success		  retval;

#define CLEAN_UP()	do {	s_clear (&argument);	} while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    CHECK_ACTIVE_TEXT();

    newline();

    /* Print the line before the insertion point */
    pos = edit_insertion_point (active_text);
    temp = *edit_text (active_text);
    start_of_line = s_strrchr (temp, '\n', pos - 1);
    if (start_of_line < 0)
    {
        xprintf (gettext("Tryck Ctrl-D fr att spara eller Esc fr att anvnda andra kommandon\n"));
	xprintf (gettext("rende: "));
	start_of_line = -1;	/* Not really necessary */
    }
    s_xputs (s_fsubstr (temp, start_of_line + 1, pos - 1));

    retval = do_continue_writing (active_text);

    RETURN (retval);
#undef CLEAN_UP
}


/* Shows the header and the text body sofar. */
Export  Success
cmd_look_at_text_and_header(String argument)
{
    Edited_text		  text;

#define CLEAN_UP()	do { s_clear (&argument); } while (0)

    TOPLOOP_SETJMP();

    CHECK_NO_ARGUMENTS();

    CHECK_ACTIVE_TEXT();

    newline();
    text = edited_texts[active_text];
    print_misc_list( text.miscs, text.no_of_miscs, PRINT_ALL_MISC );
    xprintf (gettext("rende: %S"), *edit_text (active_text));

    RETURN (OK);
#undef CLEAN_UP
}
    

/* Add another conference as recipient to the edited text. */
Export	Success
cmd_add_recipient_to_edited_text(String argument)
{
    Conf_z_info		recipient_conf = EMPTY_CONF_Z_INFO;
    String		name = EMPTY_STRING;

#define CLEAN_UP()     				\
    {						\
	s_clear (&argument);			\
	(void)s_release(&name); 		\
	release_conf_z_info(&recipient_conf);	\
    }

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_ACTIVE_TEXT();

    recipient_conf = ask_for_conf (&argument,
				   gettext("Vilket mte vill du addera som mottagare"));

    newline();

    if (recipient_conf.conf_no == 0)
    {
	RETURN (FAILURE);
    }

    if (edit_add_recipient (active_text, recipient_conf.conf_no) == FAILURE)
    {
        /*xgettext:c-format*/
	xprintf(gettext("%S kunde inte adderas som mottagare. "
			"Den var nog redan mottagare.\n"), 
		recipient_conf.name);
	RETURN (FAILURE);
    }

    /*xgettext:c-format*/
    xprintf(gettext("%S adderad som mottagare.\n"), recipient_conf.name);

    RETURN (OK);
#undef CLEAN_UP
}

/* Add another text as commented by the edited text. */
Export	Success
cmd_add_comment_to_to_edited_text(String argument)
{
    Text_no		text_to_comment;
    String_size		error_check;
    Text_stat	      * textstatP = NULL;

#define CLEAN_UP()     				\
    {						\
	s_clear (&argument);			\
    }

    TOPLOOP_SETJMP();

    OFFLINE();

    CHECK_ACTIVE_TEXT();

    text_to_comment = (Text_no) s_strtol(argument, &error_check, 
					 DEFAULT_NUMBER_BASE);

    /* Ignore error_check. There is normally garbage after the number. */

    if (text_to_comment == 0)
    {
	xprintf(gettext("Oknt argument: "));
	s_puts (argument);
	newline();
	RETURN (FAILURE);
    }

    /* Verify that there exists such a text and that we are allowed 
       to fetch it. */
    if ((textstatP = text_stat(text_to_comment)))
    {
	/* Ok. It exists. */
	release_text_stat(textstatP);
	zfree(textstatP);
    }
    else
    {
	/* There exists no such text. */
	xprintf(gettext("Texten %u existerar inte och kan drfr inte lggas till.\n"), 
		(unsigned int)text_to_comment);
	RETURN (FAILURE);
    }

    if (edit_add_commenting(active_text, text_to_comment) == FAILURE)
    {
	xprintf(gettext("Texten %u kunde inte lggas till.\n"), 
		(unsigned int)text_to_comment);
	RETURN (FAILURE);
    }

    xprintf(gettext("Texten %u kommenterad.\n"), text_to_comment);

    RETURN (OK);
#undef CLEAN_UP
}


/*
 * Edit using an external editor
 * This is the function to edit a text with an external editor.
 * The idea is to:
 * 1. Create a file with some magick contents (mainly the text sofar)
 * 2. Remove the creation of the text (much like cmd_remove_text)
 * 3. Invoke the editor on the new file.
 * 4. When the editor is done, read in the contents in the brutally simple way
 * 5. Enter writing mode again.
 */
Export  Success
cmd_edit_using_editor (String argument)
{
#define FUNCTION "cmd_edit_using_editor()"
#define TMPFILENAMELEN 30
    char 	file_name_area[TMPFILENAMELEN];
    int 	r;
    XO_stream  	xo_file;

    Kom_err	err;
    String	name = EMPTY_STRING_i;
    String	line = EMPTY_STRING_i;
    String	arguments = EMPTY_STRING_i;
    Text_stat * volatile textstatP = NULL;

    char      * command_string;
    char      * editor_string;

    int		lineno;
    int		retval;

    FILE * volatile inpfile = NULL;

#define CLEAN_UP()	do {					\
	s_clear(&argument);					\
	s_clear(&name);						\
	s_clear(&line);						\
        if (xo_file.f_stream != NULL) fclose(xo_file.f_stream);	\
	if (inpfile != NULL) fclose(inpfile);			\
	if (file_name_area[0] != '\0') unlink(file_name_area);		\
	if (textstatP != NULL)					\
	{							\
	    release_text_stat(textstatP); 			\
	    zfree(textstatP); 					\
	}							\
    } while (0)

    TOPLOOP_SETJMP();

    file_name_area[0] = '\0';

    xo_file.stream_type = XO_file;
    xo_file.f_stream = NULL;

    CHECK_NO_ARGUMENTS();

    CHECK_ACTIVE_TEXT();

    newline();

    sprintf(file_name_area, "/tmp/LysKOMText%d.txt", (int) getpid());
    xo_file.f_stream = fopen(file_name_area, "w");

    if (xo_file.f_stream == NULL)
    {
#ifdef HAVE_STRERROR
	xprintf(gettext("Kunde inte skriva till filen: %s\n"), 
		strerror(errno));
#else
	xprintf(gettext("Kunde inte skriva till filen.\n"));
#endif
	RETURN (do_continue_writing(active_text));
    }

    lineno = 1;

    for (r = 0; r < edited_texts[active_text].no_of_miscs; r++)
    {
	switch (edited_texts[active_text].miscs[r].type)
	{
	case recpt:
	    name = conf_name(edited_texts[active_text].miscs[r] 
			     . datum.recipient,
			     &err);
	    if (err != KOM_NO_ERROR)
	    {
		fatal2(CLIENT_SHOULDNT_HAPPEN,
		       "conf_name(%d) failed on conf that was added as "
		       "recipient",	
		       edited_texts[active_text].miscs[r].datum.recipient);
	    }
	    xfprintf(&xo_file, "!%S %S\n", 
		     command_name(cmd_add_recipient_to_edited_text),
		     name);
	    lineno++;
	    s_clear(&name);
	    name = EMPTY_STRING;
	    break;

	case comm_to:
	    textstatP = text_stat(edited_texts[active_text].miscs[r]
				  . datum.comment_to);
	    if (err != KOM_NO_ERROR)
	    {
		fatal2(CLIENT_SHOULDNT_HAPPEN,
		       "text_stat(%ld) failed on text I was commenting.",
		       edited_texts[active_text].miscs[r].datum.comment_to);
	    }
	    if (textstatP != NULL)
	    {
		name = conf_name(textstatP->author, &err);
		if (s_strlen(name) == 0)
		{
		    xfprintf(&xo_file, "!%S %d\n", 
			     command_name(cmd_add_comment_to_to_edited_text),
			     edited_texts[active_text].miscs[r]
			     . datum.comment_to);
		}
		else
		{
		    xfprintf(&xo_file, "!%S %d (%S)\n", 
			     command_name(cmd_add_comment_to_to_edited_text),
			     edited_texts[active_text].miscs[r]
			     .datum.comment_to,
			     name);
		    s_clear(&name);
		    name = EMPTY_STRING;
		}
		lineno++;

		release_text_stat(textstatP);
		zfree(textstatP);
		textstatP = NULL;
	    }
	    break;

	default:
	    xprintf("Not implemented yet.\n");
	    break;
	}
    }

    /* Now we have written the header. Lets write a magick command 
       and then the text.
     */
    xfprintf(&xo_file, "!%S %S\n", 
	     command_name(cmd_new_subject),
	     edited_texts[active_text].text);
    lineno++;

    /* Close the file. */
    fclose(xo_file.f_stream);
    xo_file.f_stream = NULL;

    /* No the text resides safely in the file. Lets remove it */
    active_text = edit_dealloc_text(active_text);

    /* Invoke the favorite editor */
    editor_string = getenv("VISUAL");
    if (!editor_string)
	editor_string = getenv("EDITOR");
    if (!editor_string)
        editor_string = EDITOR;

    command_string = (char *)zmalloc(strlen(editor_string) 
				     + 10 + TMPFILENAMELEN);
    sprintf(command_string, "%s +%d %s", editor_string, lineno, file_name_area);

    xprintf(gettext("Startar extern editor med kommandot: %s\nVntar..."),
	    command_string);
    system(command_string);
    xprintf(gettext("klar.\nLser in texten frn den externa editorn:\n"));

    zfree(command_string);

    /* Setup an empty text. */
    /* Read lines from the file. If the line begins with !, treat it as a 
       command.
       */
    active_text = edit_create_new_text();
    
    inpfile = fopen(file_name_area, "r");
    if (inpfile == NULL)
    {
#ifdef HAVE_STRERROR
	xprintf(gettext("Kunde inte lsa frn filen: %s\n"),
		strerror(errno));
#else
	xprintf(gettext("Kunde inte lsa frn filen.\n"));
#endif
	RETURN (do_continue_writing(active_text));
    }
    while ((retval = input_line(inpfile, stdout, &line, -1, "\n", "", EOF))
	   != EOF)
    {
	if (retval == 0)
	{
	    fatal2(CLIENT_SHOULDNT_HAPPEN,
		   "cannot read lines from the file %s",
		   file_name_area);
	    /*NOTREACHED*/
	}

	line = s_strip_trailing(line, 
				s_fcrea_str((unsigned char *) " \t\r\n"));
	newline();

	if (s_strhead(s_fcrea_str("!"), line))
	{
	    /* This is a command. */
	    s_strdel(&line, 0, 0);

	    /* Only five of the commands are accepted:
	       cmd_add_recipient_to_edited_text
	       cmd_add_cc_recipient_to_edited_text (NYI)
	       cmd_new_subject
	       cmd_add_footn_to (NYI)
	       */
#define TEST_AND_PERFORM_COMMAND(command)				\
	    if (s_strhead(command_name(command), line))			\
	    {								\
		arguments = EMPTY_STRING;				\
		s_strcpy(&arguments, line);				\
		s_strdel(&arguments, 0, 				\
			 s_strlen(command_name(command)));		\
									\
		command(arguments);					\
	    }

	    TEST_AND_PERFORM_COMMAND(cmd_add_recipient_to_edited_text)
	    else TEST_AND_PERFORM_COMMAND(cmd_add_comment_to_to_edited_text)
	    else TEST_AND_PERFORM_COMMAND(cmd_new_subject)
	    else {
		xprintf(gettext("Oknt kommando %S.\n"), line);
	    }
	}
	else
	{
	    /* This is a line to be appended to the text. */
	    edit_add_string(active_text, line);
	    edit_add_string(active_text, s_fcrea_str("\n"));
	}
    }

    RETURN(OK);

#undef CLEAN_UP
#undef FUNCTION
}



/*  ==================================================================	*/
/*			    Internal functions				*/


/*
 *  Continue the writing of a text that was started earlier.
 */
Export  Success
do_continue_writing (int  edit_no)

{
    Edit_result		  edit_return;


#define CLEAN_UP()	do {	} while (0)

    TOPLOOP_SETJMP();

    /* BUG!  Check if any text is active */
    edit_return = edit_enter_text (edit_no);

    switch (edit_return)
    {
    case  enter_this_text:
	cmd_enter_text (EMPTY_STRING);
	break;

    case  kill_this_text:
	active_text = edit_dealloc_text (edit_no);
	break;

    case  to_be_continued:
	break;
    }

    RETURN (OK);

#undef CLEAN_UP
}
