/***************************************************************************
                          mimesources.cpp  -  description
                             -------------------
    begin                : Son Sep 14 2003
    copyright            : (C) 2003 by Dominik Seichter
    email                : domseichter@web.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "mimesources.h"
#include "mycanvasitem.h"
#include "mybarcode.h"
#include "xmlutils.h"

#include <qdom.h>
#include <qpixmap.h>

DSTextDrag::DSTextDrag( QWidget *dragSource, const char *name )
    : QTextDrag( dragSource, name )
{
}

QByteArray DSTextDrag::encodedData( const char *mime ) const
{
    if ( strcmp( selectionMimeType(), mime ) == 0)
        return kbarcode;
    else if( strcmp( "application/x-kbarcode-textobjectnumber", mime ) == 0)
    {
        QByteArray a;
        QCString s (  QString::number(m_textObjectNumber).local8Bit() );
        a.resize( s.length() + 1 ); // trailing zero
        memcpy( a.data(), s.data(), s.length() + 1 );
        return a;
    }
    else
        return QTextDrag::encodedData(mime);
}

bool DSTextDrag::canDecode( QMimeSource* e )
{
    if ( e->provides( selectionMimeType() ) )
        return true;
    return QTextDrag::canDecode(e);
}

const char* DSTextDrag::format( int i ) const
{
    if ( i < 4 ) // HACK, but how to do otherwise ??
        return QTextDrag::format(i);
    else if ( i == 4 )
        return selectionMimeType();
    else if ( i == 5 )
        return "application/x-kbarcode-textobjectnumber";
    else return 0;
}

const char * DSTextDrag::selectionMimeType()
{
    return "application/x-kbarcode-textselection";
}

void DSTextDrag::setTextObjectNumber( int number )
{
    m_textObjectNumber = number;
}

int DSTextDrag::decodeTextObjectNumber( QMimeSource *e )
{
    QByteArray a =  e->encodedData("application/x-kbarcode-textobjectnumber");
    if(!a.isEmpty())
        return QCString(a).toInt();
    else
        return -1;
}

RectDrag::RectDrag()
    : QStoredDrag( "application/x-kbarcode-rect" )
{
}

void RectDrag::setRect( const DrawingRect* rect )
{
    QByteArray data;
    int values[ 11 ];
    values[ 0 ] = rect->isCircle();
    values[ 1 ] = rect->borderStyle();
    values[ 2 ] = rect->borderWidth();
    values[ 3 ] = rect->borderColor().red();
    values[ 4 ] = rect->borderColor().green();
    values[ 5 ] = rect->borderColor().blue();
    values[ 6 ] = rect->color().red();
    values[ 7 ] = rect->color().green();
    values[ 8 ] = rect->color().blue();
    values[ 9 ] = rect->width();
    values[ 10 ] = rect->height();

    data.resize( sizeof( values ) );
    memcpy( data.data(), values, sizeof( values ) );
        
    setEncodedData( data );
}

bool RectDrag::canDecode( QMimeSource* e )
{
   return e->provides( "application/x-kbarcode-rect" );
}

bool RectDrag::decode( QMimeSource* e, DrawingRect* rect )
{
    QByteArray data = e->encodedData( "application/x-kbarcode-rect" );
    int values[ 11 ];
    if( data.size() != sizeof( values ) )
        return false;

    memcpy( values, data.data(), sizeof( values ) );

    rect->setCircle( values[ 0 ] );
    rect->setSize( values[ 9 ], values[ 10 ] );
    rect ->setBorderStyle( values[ 1 ] );
    rect->setBorderWidth( values[ 2 ] );
    rect->setBorderColor( QColor( values[ 3 ], values[ 4 ], values[ 5 ] ) );
    rect->setColor( QColor( values[ 6 ], values[ 7 ], values[ 8 ] ) );
    
    return true;
}

LineDrag::LineDrag()
    : QStoredDrag( "application/x-kbarcode-line" )
{
}

void LineDrag::setLine( const MyCanvasLine* line )
{
    QByteArray data;
    int values[ 9 ];

    values[0] = line->startPoint().x();
    values[1] = line->startPoint().y();
    values[2] = line->endPoint().x();
    values[3] = line->endPoint().y();
    values[4] = line->pen().width();
    values[5] = line->pen().style();
    values[6] = line->pen().color().red();
    values[7] = line->pen().color().green();
    values[8] = line->pen().color().blue();
    
    data.resize( sizeof( values ) );
    memcpy( data.data(), values, sizeof( values ) );

    setEncodedData( data );
}

bool LineDrag::canDecode( QMimeSource* e )
{
   return e->provides( "application/x-kbarcode-line" );
}

bool LineDrag::decode( QMimeSource* e, MyCanvasLine* line )
{
    QByteArray data = e->encodedData( "application/x-kbarcode-line" );
    int values[ 9 ];
    if( data.size() != sizeof( values ) )
        return false;

    memcpy( values, data.data(), sizeof( values ) );

    line->setPoints( values[0], values[1], values[2], values[3] );
    line->setPen( QPen( QColor( values[6], values[7], values[8] ), values[4], (QPen::PenStyle)values[5] ) );

    return true;
}

BarcodeDrag::BarcodeDrag()
    : QDragObject() //  "application/x-kbarcode-barcode" )
{
}

const char* BarcodeDrag::format( int i ) const
{
    if( i == 0 )
        return "application/x-kbarcode-barcode";
    else if( i == 1 )
        return "image/png";
    else return 0;
}

bool BarcodeDrag::provides( const char* format ) const
{
    if( QString( format ) == "image/png" )
        return true;
    else if( QString( format ) == "application/x-kbarcode-barcode" )
        return true;

    return false;
}

void BarcodeDrag::setBarcode( const barcodeData* barcode )
{
    QDomDocument doc("barcodedata");

    QDomElement tag = doc.createElement( "barcode" );
    XMLUtils xml;
    xml.writeBarcode( &tag, barcode, 0, 0, false );    
    doc.appendChild( tag );
    
    QCString string = doc.toCString();

    data = string;

    BarCode b( barcode );
    setPixmap( b.pixmap() );
}

void BarcodeDrag::setPixmap( const QPixmap & pix )
{
    QBuffer buffer( pixdata );
    buffer.open( IO_WriteOnly );

    QImageIO io( &buffer, "PNG" );
    io.setImage( pix.convertToImage() );
    io.write();
    buffer.close();
}

bool BarcodeDrag::canDecode( QMimeSource* e, const char* format ) 
{
   return e->provides( format );
}

bool BarcodeDrag::decode( QMimeSource* e, QPixmap* barcode )
{
    QByteArray data = e->encodedData( "image/png" );
    return barcode->load( data, "PNG" );
}

bool BarcodeDrag::decode( QMimeSource* e, CanvasBarcode* barcode )
{
    QByteArray data = e->encodedData( "application/x-kbarcode-barcode" );

    QDomDocument doc("barcodedata");
    if ( doc.setContent( QString( data ) ) ) {
        QDomNode n = doc.documentElement();
        if( !n.isNull() ) {
            QDomElement e = n.toElement(); // try to convert the node to an element.
            if( !e.isNull() && e.tagName() == "barcode" ) {
                XMLUtils xml;
                barcodeData d = xml.readBarcode( &e );
                barcode->setData( &d );
                barcode->show();
                barcode->updateBarcode();
                return true;
            } 
        }
    }

    return false;
}

QByteArray BarcodeDrag::encodedData ( const char* format ) const
{
    QByteArray b;
    if( QString( format ) == "application/x-kbarcode-barcode" ) {
        return data;
    } else if( QString( format ) == "image/png" ) {
        return pixdata;
    }

    return b;
}
