#****************************************************************************
#*                          WaveWidgetPrefs.tcl
#*
#* Author: Matthew Ballance
#* Desc:   Implements a simple wave-widget preferences dialog
#* <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
#*
#*    This source code is free software; you can redistribute it
#*    and/or modify it in source code form under the terms of the GNU
#*    General Public License as published by the Free Software
#*    Foundation; either version 2 of the License, or (at your option)
#*    any later version.
#*
#*    This program is distributed in the hope that it will be useful,
#*    but WITHOUT ANY WARRANTY; without even the implied warranty of
#*    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#*    GNU General Public License for more details.
#*
#*    You should have received a copy of the GNU General Public License
#*    along with this program; if not, write to the Free Software
#*    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
#*
#* </Copyright>
#****************************************************************************

#********************************************************************
#* namespace WaveWidgetPrefs
#********************************************************************
namespace eval WaveWidgetPrefs {

    variable configspec {
        { -trace_height        22     }
        { -pix_per_time        30     }
        { -wave_widget         ""     }
        { -color_scheme        ""     }
        { -num_cursors         4      }
        { -vertbar_sep         60     }
        { -sigpane_width       50     }
    }

    variable d_configitems {
        { "Signal-list Background Color" "siglist_bg"                "grey50" }
        { "Signal-list Foreground Color" "siglist_fg"                "white"  }
        { "Wave Background Color"        "wavewin_bg"                "black"  }
        { "Vector expanded 1 color"      "vector_exp_1_color"        "green"  }
        { "Vector expanded 0 color"      "vector_exp_0_color"        "green"  }
        { "Vector expanded x color"      "vector_exp_x_color"        "red"    }
        { "Vector expanded z color"      "vector_exp_z_color"        "blue"   }
        { "Vector expanded change color" "vector_exp_chg_color"      "green"  }
        { "Vector binary color"          "vector_bin_color"          "green"  }
        { "Vector change color"          "vector_chg_color"          "green"  }
        { "Vector x color"               "vector_x_color"            "red"    }
        { "Vector z color"               "vector_z_color"            "blue"   }
        { "Scalar 1 color"               "scalar_1_color"            "green"  }
        { "Scalar 0 color"               "scalar_0_color"            "green"  }
        { "Scalar x color"               "scalar_x_color"            "red"    }
        { "Scalar z color"               "scalar_z_color"            "blue"   }
        { "Scalar change color"          "scalar_chg_color"          "green"  }
        { "Cursor unselected color"      "cursor_unsel_color"        "blue"   }
        { "Cursor selected color"        "cursor_sel_color"          "blue"   }
        { "Cursor locked unsel color"    "cursor_lock_unsel_color"   "red"    }
        { "Cursor locked sel color"      "cursor_lock_sel_color"     "red"    }
        { "Scale Color"                  "scale_color"               "white"  }
        { "Timescale text color"         "timescale_text_color"      "white"  }
        { "Separator color"              "separator_color"           "white"  }
        { "Selection Box Color"          "selbox_color"              "green"  }
        { "Cursor Measurement Color"     "cursor_measure_color"      "grey95" }
    }
}

#********************************************************************
#* WaveWidgetPrefs
#********************************************************************
proc WaveWidgetPrefs::WaveWidgetPrefs {w args} {
    variable configspec

    set w [frame $w]

    array set $w {_dummy _dummy}
    upvar #0 $w data

    BaseClass::config_init $w $configspec

    eval BaseClass::configure $w $args

    #**** Construct top pane
    set top_pane [frame $w.top_pane]
    set preview [ConstructPreviewPane $w $top_pane.c]
    set items   [ConstructItemConfigPane $w $top_pane.ic]

    #**** Construct bottom pane
    set bottom_pane [ConstructBottomPane $w $w.bottom_frame]

    pack $items -side left -fill y
    pack $preview -side left -expand yes -fill both

    pack $bottom_pane -side bottom -fill x
    pack $top_pane -side bottom -expand yes -fill both

    if {$data(-wave_widget) != ""} {
        set data(-color_scheme) [$data(-wave_widget) cget -color_scheme]

        if {[regexp {^WaveWidget.ColorSchemes.} $data(-color_scheme)] != 1} {
            puts "ERROR: ColorScheme must be located in WaveWidget.ColorSchemes"
            set data(-color_scheme) ""
        }
    }

    InitSchemeSel $w

    set leafname [lindex [split $data(-color_scheme) .] end]
    SetupScheme $w $leafname


    UpdateWindow $w

    return $w
}

#********************************************************************
#* ActivateSettings
#********************************************************************
proc WaveWidgetPrefs::ActivateSettings {w} {
    upvar #0 $w data

    #*** First off, propagate the cached changes to the config-db
    set idx 0
    foreach scm_name $data(color_schemes) {
        set scm_data [lindex $data(color_scheme_data) $idx]

        set full_db_name "WaveWidget.ColorSchemes.$scm_name"

        #*** Go through each elem of the scheme looking for differences
        foreach lelem $scm_data {
            set cv_name [lindex $lelem 1]
            set cvalue  [lindex $lelem 2]
            set dbvalue [IviConfig::current $full_db_name.$cv_name]

            if {$dbvalue != $cvalue} {
                IviConfig::current $full_db_name.$cv_name $cvalue -level user
            }
        }

        incr idx
    }

    if {$data(-wave_widget) != ""} {
        set this_scheme [lindex $data(color_schemes) \
            [$data(w:this_window_scheme) getvalue]]
        $data(-wave_widget) configure \
            -color_scheme "WaveWidget.ColorSchemes.$this_scheme"
    }

    set sel_dflt [lindex $data(color_schemes) \
        [$data(w:widget_default_scheme) getvalue]]

    if {[IviConfig::current WaveWidget.ColorScheme] != $sel_dflt} {
        IviConfig::current WaveWidget.ColorScheme $sel_dflt -level user
    }
}


#********************************************************************
#* ConstructBottomPane
#********************************************************************
proc WaveWidgetPrefs::ConstructBottomPane {w path} {
    upvar #0 $w data

    set path [frame $path]

    set sep_pane [frame $path.sep -height 2 -borderwidth 2 -relief sunken]

    #*** First, divide things a bit further with a schemes pane
    #*** and a pick-scheme pane
    set spane [ConstructSchemesPane $w $path.sp]
    set sspane [ConstructSchemeSelPane $w $path.ssp]
    pack $sep_pane -side top -fill x
#    pack $spane -side left
    pack $spane -side left
    pack $sspane -expand yes -fill both -side right

    return $path
}

#********************************************************************
#* ConstructSchemesPane
#********************************************************************
proc WaveWidgetPrefs::ConstructSchemesPane {w path} {
    upvar #0 $w data

    set path [frame $path -relief groove -borderwidth 2]

    #*** Construct the scroll widget
    set sltitle [TitleFrame::TitleFrame $path.sltitle -text "Schemes"]
    set swin [ScrollContainer::ScrollContainer [$sltitle getframe].sc]
    set lframe [$swin subwidget frame]

    set data(w:schemelist) [listbox $lframe.schemelist -exportselection false \
        -height 4]
    bind $data(w:schemelist) <<ListboxSelect>> \
        "WaveWidgetPrefs::SchemeListSelect $w"

    set vsb [$swin subwidget vsb]
    set hsb [$swin subwidget hsb]

    $data(w:schemelist) configure -yscrollcommand "$vsb set"
    $data(w:schemelist) configure -xscrollcommand "$hsb set"

    $vsb configure -command "$data(w:schemelist) yview"
    $hsb configure -command "$data(w:schemelist) xview"

    #*** Construct the scheme creation buttons
    set sop_frame [frame $path.sopf]
    set entry_name [TitleFrame::TitleFrame $sop_frame.nameframe \
        -text "Scheme Name"]
    set sentry   [entry [$entry_name getframe].entry -validate key \
        -validatecommand "WaveWidgetPrefs::SchemeNameValidate $w %P"]
    set newnamef [frame [$entry_name getframe].newnamef]

    set data(w:schemename) $sentry
    set data(w:newscheme) [button $newnamef.new -text "New" \
        -state disabled -command "WaveWidgetPrefs::NewScheme $w"]
#    set data(w:namescheme) [button $newnamef.name -text "Rename" \
#        -state disabled -command "WaveWidgetPrefs::RenameScheme $w"]

    pack $sentry -side top
    grid $data(w:newscheme) 
    # $data(w:namescheme)
    pack $newnamef -side top -expand yes -fill x
    pack $entry_name -side top

    pack $data(w:schemelist) -expand yes -fill both
    pack $sltitle -side left
    pack $swin -side left
    pack $sop_frame -side left -fill y

    return $path
}

#********************************************************************
#* ConstructSchemeSelPane
#********************************************************************
proc WaveWidgetPrefs::ConstructSchemeSelPane {w path} {
    upvar #0 $w data

    set path [frame $path -relief groove -borderwidth 2]

    #*** Widget-default Scheme
    set widdef_title [TitleFrame::TitleFrame $path.widdef_title \
        -text "Widget Default Scheme"]
    set data(w:widget_default_scheme) [ComboBox \
        [$widdef_title getframe].wd_scheme]

    pack $data(w:widget_default_scheme) 

    #*** Scheme for this window
    set winsch_title [TitleFrame::TitleFrame $path.winsch_title \
        -text "This Window Scheme"]
    set data(w:this_window_scheme) [ComboBox \
        [$winsch_title getframe].tw_scheme]
    pack $data(w:this_window_scheme)

    pack $widdef_title $winsch_title

    return $path
}

#********************************************************************
#* ConstructPreviewPane
#********************************************************************
proc WaveWidgetPrefs::ConstructPreviewPane {w path} {
    upvar #0 $w data

    set width  380
    set height 250
    set data(w:c) [canvas $path -width $width -height $height -bg grey95]
    set data(width) $width
    set data(height) $height

    bind $data(w:c) <Configure> "WaveWidgetPrefs::UpdateWindow $w"

    return $path
}

#********************************************************************
#* ConstructItemConfigPane
#********************************************************************
proc WaveWidgetPrefs::ConstructItemConfigPane {w path} {
    upvar #0 $w data
    variable d_configitems
    set img_data {
        #define img_data_width 16
        #define img_data_height 16
        static unsigned char img_data_bits[] = {
            0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff,
            0xff, 0xff, 0xff, 0xff
        };
    }

    set data(configitems) $d_configitems

    set data(w:button_img) [image create bitmap -data $img_data]

    set path [frame $path]

    set scrollc [ScrollContainer::ScrollContainer $path.sc]
    set lframe [$scrollc subwidget frame]

    set data(w:itemlist)  [listbox $lframe.itemlist \
        -exportselection false -height 4]
    bind $data(w:itemlist) <<ListboxSelect>> \
        "WaveWidgetPrefs::ListBoxSelect $w"

    set vsb [$scrollc subwidget vsb]
    set hsb [$scrollc subwidget hsb]

    $data(w:itemlist) configure -yscrollcommand "$vsb set"
    $data(w:itemlist) configure -xscrollcommand "$hsb set"

    $vsb configure -command "$data(w:itemlist) yview"
    $hsb configure -command "$data(w:itemlist) xview"

    #**** Add all config items to the listbox ****
    foreach spec $d_configitems {
        $data(w:itemlist) insert end [lindex $spec 0]
    }

    set value_pane [frame $path.value_pane]


    set data(w:itemvalue)   [entry $value_pane.itemvalue -width 8]

    bind $data(w:itemvalue) <Return> "WaveWidgetPrefs::SetEntryColor $w"

    set data(w:valuebutton) [button $value_pane.valuebutton \
        -image $data(w:button_img) -command "WaveWidgetPrefs::SelectColor $w"]

    grid $data(w:itemvalue) $data(w:valuebutton)

    pack $data(w:itemlist) -expand yes -fill both
    pack $value_pane -side bottom -fill x
    pack $scrollc -expand yes -fill both

    #**** Set initial values...
    $data(w:itemlist) selection set 0
    ListBoxSelect $w

    return $path
}

#********************************************************************
#* ListBoxSelect
#*
#* act on user selection of items in the config-items listbox
#********************************************************************
proc WaveWidgetPrefs::ListBoxSelect {w} {
    upvar #0 $w data

    set sel [$data(w:itemlist) curselection]

    set lelem [lindex $data(configitems) $sel]

    $data(w:button_img) configure -foreground [lindex $lelem 2]

    $data(w:itemvalue) delete 0 end
    $data(w:itemvalue) insert 0 [lindex $lelem 2]
}

#********************************************************************
#* SelectColor
#*
#* Called when the user presses the color-select button
#********************************************************************
proc WaveWidgetPrefs::SelectColor {w} {
    upvar #0 $w data

    set init_color [$data(w:button_img) cget -foreground]
    set color [tk_chooseColor -initialcolor $init_color]

    if {$color != ""} {
        $data(w:button_img) configure -foreground $color
        $data(w:itemvalue) delete 0 end
        $data(w:itemvalue) insert 0 $color

        set sel [$data(w:itemlist) curselection]

        SetConfigValue $w $sel $color
    }
}

#********************************************************************
#* SetEntryColor
#*
#* Called in response to a <Return> on the entry... Tries to update
#* the color based on what the user typed...
#********************************************************************
proc WaveWidgetPrefs::SetEntryColor {w} {
    upvar #0 $w data

    set text [$data(w:itemvalue) get]

    if {$text != ""} {
        set old_color [$data(w:button_img) cget -foreground]

        set sel [$data(w:itemlist) curselection]

        set stat [catch {$data(w:valuebutton) configure -foreground $text} res]

        if {$stat} {
            set new_color $old_color
        } else {
            set new_color $text
        }

        $data(w:itemvalue) delete 0 end
        $data(w:itemvalue) insert 0 $new_color

        if {$new_color == $old_color} {
            return
        }

        $data(w:button_img) configure -foreground $new_color

        SetConfigValue $w $sel $new_color
    }
}

#********************************************************************
#* SetConfigValue
#********************************************************************
proc WaveWidgetPrefs::SetConfigValue {w idx value} {
    upvar #0 $w data

    set lelem [lindex $data(configitems) $idx]

    set lelem [lreplace $lelem 2 2 $value]
    set data(configitems) [lreplace $data(configitems) $idx $idx $lelem]

    #*** Update the global-data array
    set leafname [lindex [split $data(-color_scheme) .] end]
    set idx [lsearch -exact $data(color_schemes) $leafname]

    set data(color_scheme_data) [lreplace $data(color_scheme_data) \
        $idx $idx $data(configitems)]

    UpdateWindow $w
}

#********************************************************************
#* GetConfigValue
#********************************************************************
proc WaveWidgetPrefs::GetConfigValue {w name} {
    upvar #0 $w data

    foreach lelem $data(configitems) {
        if {[lindex $lelem 1] == $name} {
            return [lindex $lelem 2]
        }
    }

    return -error "no value \"$name\""
}

#********************************************************************
#* InitSchemeSel
#********************************************************************
proc WaveWidgetPrefs::InitSchemeSel {w} {
    upvar #0 $w data

    set data(color_schemes) [IviConfig::section_names WaveWidget.ColorSchemes]

    set data(color_scheme_data) ""
    #*** Set list of schemes in the schemes list
    foreach scheme $data(color_schemes) {
        $data(w:schemelist) insert end $scheme
        lappend data(color_scheme_data) null
    }

    #*** Setup the defaults combo boxes
    $data(w:widget_default_scheme) configure -values $data(color_schemes)
    $data(w:this_window_scheme)    configure -values $data(color_schemes)

    #*** Now, set the selection mark on the current scheme
    set leafname [lindex [split $data(-color_scheme) .] end]
    set idx [lsearch -exact $data(color_schemes) $leafname]

    $data(w:this_window_scheme) setvalue "@$idx"

    #*** Find the widget-default...
    set widget_default [IviConfig::current WaveWidget.ColorScheme]
    set wd_idx [lsearch -exact $data(color_schemes) $widget_default]
    if {$wd_idx >= 0} {
        $data(w:widget_default_scheme) setvalue "@$wd_idx"
    } else {
        $data(w:widget_default_scheme) setvalue "@$idx"
    }

    if {$idx >= 0} {
        $data(w:schemelist) selection set $idx
    } else {
        puts "Internal Error: Color scheme \"$data(-color_scheme)\" isn't on the color scheme list"
    }
}

#********************************************************************
#* SchemeListSelect
#*
#* Act on a user selection of an item in the scheme list.
#* Set the current scheme based on this selection
#********************************************************************
proc WaveWidgetPrefs::SchemeListSelect {w} {
    upvar #0 $w data

    set sel [$data(w:schemelist) curselection]

    set leafname [lindex $data(color_schemes) $sel]

    SetupScheme $w $leafname
    UpdateWindow $w
}

#********************************************************************
#* SetupScheme
#********************************************************************
proc WaveWidgetPrefs::SetupScheme {w scheme_name} {
    upvar #0 $w data
    variable d_configitems

    #*** First, if a scheme is already set, re-insert that scheme
    #*** back into the data list
    set leafname [lindex [split $data(-color_scheme) .] end]
    set old_idx  [lsearch -exact $data(color_schemes) $leafname]

    set old_data [lindex $data(color_scheme_data) $old_idx]

    #*** Only re-insert the data if the old name exists...
    if {$old_idx >= 0 && $old_data != "null"} {
        set data(color_scheme_data) [lreplace $data(color_scheme_data) \
            $old_idx $old_idx $data(configitems)]
    }

    #*** Find index of scheme_name in name list
    set idx [lsearch -exact $data(color_schemes) $scheme_name]

    if {$idx < 0} {
        puts "ERROR: no name 'scheme_name'"
        return
    }

    #*** If the scheme data is null, initialize the scheme from
    #*** the ConfigDB
    set cdata [lindex $data(color_scheme_data) $idx]
    if {$cdata == "null"} {
        set cdata $d_configitems

        set full_db_name "WaveWidget.ColorSchemes.$scheme_name"

        set cidx 0
        foreach lelem $cdata {
            set cv_name [lindex $lelem 1]
            set cvalue  [IviConfig::current $full_db_name.$cv_name]

            if {$cvalue != ""} {
                set lelem [lreplace $lelem 2 2 $cvalue]
                set cdata [lreplace $cdata $cidx $cidx $lelem]
            }

            incr cidx
        }

        #*** insert the new data in the config_data list
        set data(color_scheme_data) [lreplace $data(color_scheme_data) \
            $idx $idx $cdata]
    }

    #*** Set the data from the data-list
    set data(configitems) [lindex $data(color_scheme_data) $idx]
    set data(-color_scheme) "WaveWidget.ColorSchemes.$scheme_name"

    #*** Ensure that the correct value is shown on the color-select button
    ListBoxSelect $w
}

#********************************************************************
#* DrawScalarTrace
#********************************************************************
proc WaveWidgetPrefs::DrawScalarTrace {w x y vals scalar} {
    upvar #0 $w data

    if {$scalar == 1} {
        set colors {scalar_0_color scalar_1_color \
                    scalar_x_color scalar_z_color}
        set chg_color [GetConfigValue $w scalar_chg_color]
    } else {
        set colors {vector_exp_0_color vector_exp_1_color \
                    vector_exp_x_color vector_exp_z_color}
        set chg_color [GetConfigValue $w vector_exp_chg_color]
    }

#    set c [expr $y + ($data(-trace_height) / 2)]
    set c $y
    set d [expr (($data(-trace_height) * 7) / 8) / 2]

    foreach val $vals {
        switch $val {
            0 {
                set yn [expr $c - $d]
                set color [GetConfigValue $w [lindex $colors 0]]
            }

            1 {
                set yn [expr $c + $d]
                set color [GetConfigValue $w [lindex $colors 1]]
            }

            x {
                set yn $c
                set color [GetConfigValue $w [lindex $colors 2]]
            }

            z {
                set yn $c
                set color [GetConfigValue $w [lindex $colors 3]]
            }
        }

        set xn [expr $x + $data(-pix_per_time)]

        if {$yn != $y} {
            $data(w:c) create line $x $y $x $yn -fill $chg_color
        }

        $data(w:c) create line $x $yn $xn $yn -fill $color

        set x $xn
        set y $yn
    }
}

#********************************************************************
#* DrawExpVectorTrace
#********************************************************************
proc WaveWidgetPrefs::DrawExpVectorTrace {w x y vals} {
    upvar #0 $w data

    DrawVectorTrace $w $x $y $vals

    set val [lindex $vals 0]
    set vlen [string length $val]

    #*** Draw the expanded traces 
    for {set i 0} {$i < $vlen} {incr i} {

        #*** Create the value list
        set svals ""
        foreach val $vals {
            lappend svals [string index $val $i]
        }

        DrawScalarTrace $w $x [expr $y + $data(-trace_height)*($i+1)] $svals 0
    }
}

#********************************************************************
#* GetVectorColor
#********************************************************************
proc WaveWidgetPrefs::GetVectorColor {w vals} {
    upvar #0 $w data

    set nx 0
    set nz 0
    set nb 0

    set len [string length $vals]
    for {set i 0} {$i < $len} {incr i} {
        set val [string index $vals $i]
        switch $val {
            x {
                incr nx
            }

            z {
                incr nz
            }

            0 {
                incr nb
            }

            1 {
                incr nb
            }

            default {
                puts "ERROR: Unknown value $val"
            }
        }
    }

    if {$nz == 0 && $nx == 0} {
        return "b"
    } elseif {$nz != 0 && $nx == 0 && $nb == 0} {
        return "z"
    } else {
        return "x"
    }
}

#********************************************************************
#* DrawVectorTrace
#********************************************************************
proc WaveWidgetPrefs::DrawVectorTrace {w x y vals} {
    upvar #0 $w data

    set c $y
    set d [expr (($data(-trace_height) * 7) / 8) / 2]
    set last ""
    set last_vtype ""
    set vtype ""

    set y [expr $c + $d]

    set chg_color [GetConfigValue $w vector_chg_color]

    foreach val $vals {
        #*** Figure out how to display the vector portion
        set last_vtype $vtype
        set vtype [GetVectorColor $w $val]
        switch $vtype {
            x {
                set color [GetConfigValue $w vector_x_color]
            }

            z {
                set color [GetConfigValue $w vector_z_color]
            }

            default {
                set color [GetConfigValue $w vector_bin_color]
            }
        }

        set xn [expr $x + $data(-pix_per_time)]

        if {$val != $last && $last != ""} {

            #*** If the transition goes from binary to binary,
            #*** Then draw the full transition
            if {$last_vtype == "b" && $vtype == "b"} {
                #*** Draw change...
                $data(w:c) create line [expr $x - 2] [expr $c - $d] \
                    [expr $x + 2] [expr $c + $d] -fill $chg_color
                $data(w:c) create line [expr $x - 2] [expr $c + $d] \
                    [expr $x + 2] [expr $c - $d] -fill $chg_color

                #*** Draw next vector segment...
                $data(w:c) create line [expr $x + 2] [expr $c - $d] \
                    $xn [expr $c - $d] -fill $color
                $data(w:c) create line [expr $x + 2] [expr $c + $d] \
                    $xn [expr $c + $d] -fill $color

            #*** Otherwise, if the left half of the change is x or z,
            #*** and the next color is binary, draw the right transition...
            } elseif {$last_vtype != "b" && $vtype == "b"} {
                $data(w:c) create line $x $c \
                    [expr $x + 2] [expr $c + $d] -fill $chg_color
                $data(w:c) create line $x $c \
                    [expr $x + 2] [expr $c - $d] -fill $chg_color

                #*** Draw next vector segment...
                $data(w:c) create line [expr $x + 2] [expr $c - $d] \
                    $xn [expr $c - $d] -fill $color
                $data(w:c) create line [expr $x + 2] [expr $c + $d] \
                    $xn [expr $c + $d] -fill $color

            #*** Otherwise, if the left half is binary, and the next change
            #*** is x/z, then draw the left transition
            } elseif {$last_vtype == "b" && $vtype != "b"} {
                $data(w:c) create line [expr $x - 2] [expr $c + $d] \
                    $x $c -fill $chg_color
                $data(w:c) create line [expr $x - 2] [expr $c - $d] \
                    $x $c -fill $chg_color
             
                #*** Draw next vector segment...
                $data(w:c) create line $x $c $xn $c -fill $color
            } else {
                #*** Otherwise, both transitions are x/z

                $data(w:c) create line $x $c $xn $c -fill $color
            }
        } else {
            if {$vtype == "b"} {
                $data(w:c) create line $x [expr $c - $d] \
                    $xn [expr $c - $d] -fill $color
                $data(w:c) create line $x [expr $c + $d] \
                    $xn [expr $c + $d] -fill $color
            } else {
                $data(w:c) create line $x $c $xn $c -fill $color
            }
        }

        set x $xn
        set last $val
    }
}

#********************************************************************
#* DrawScale
#********************************************************************
proc WaveWidgetPrefs::DrawScale {w} {
    upvar #0 $w data

}

#********************************************************************
#* DrawBackdrop
#********************************************************************
proc WaveWidgetPrefs::DrawBackdrop {w} {
    upvar #0 $w data

    set x_base [expr $data(-sigpane_width) * 2]
    set x_max  $data(width)

    #**** Figure out where the floor is
    set vert_floor [expr $data(height) - \
            ($data(-trace_height) * (1 + $data(-num_cursors)))]

    #**** First, draw vertical bars
    set num_vert_bars [expr $data(width) / $data(-vertbar_sep)]

    set scale_color [GetConfigValue $w scale_color]
    for {set i 0} {$i < $num_vert_bars} {incr i} {
        set x [expr ($i * ($data(-vertbar_sep) + 1)) + $x_base]
        $data(w:c) create line $x 0 $x $vert_floor \
            -fill $scale_color -width 1.0
    }

    #**** Next, draw the cursor slots
    set num_slots [expr $data(-num_cursors) + 1]
    for {set i 0} {$i < $num_slots} {incr i} {
        $data(w:c) create line $x_base \
            [expr $vert_floor + ($data(-trace_height) * $i)] \
            $x_max \
            [expr $vert_floor + ($data(-trace_height) * $i)] \
            -fill $scale_color
    }

    #**** Finally, draw the timescale text... 
    set text_color [GetConfigValue $w timescale_text_color]
    set yc [expr $data(-trace_height)*$data(-num_cursors) + \
                ($data(-trace_height)/2) + $vert_floor]
    for {set i 0} {$i < $num_vert_bars} {incr i} {
        set x [expr $x_base + (($i+1) * $data(-vertbar_sep))]
        $data(w:c) create text $x $yc -text "[expr 10 + 10*$i] ns" \
            -fill $text_color
    }
}

#********************************************************************
#* DrawSigPanes
#********************************************************************
proc WaveWidgetPrefs::DrawSigPanes {w} {
    upvar #0 $w data

    set siglist_bg  [GetConfigValue $w siglist_bg]
    set siglist_fg  [GetConfigValue $w siglist_fg]

    set two_panew [expr $data(-sigpane_width) * 2]
    set one_panew $data(-sigpane_width)

    $data(w:c) create rectangle 0 0 $one_panew $data(height) -fill $siglist_bg
    $data(w:c) create rectangle $one_panew 0 $two_panew $data(height) \
        -fill $siglist_bg
    $data(w:c) create line $one_panew 0 $one_panew $data(height) \
        -width 2.0 -fill grey95
}

#********************************************************************
#* DrawCursors
#*
#* Draw one of each type of cursor in sequence...
#********************************************************************
proc WaveWidgetPrefs::DrawCursors {w} {
    upvar #0 $w data
    set color_names  {cursor_unsel_color cursor_sel_color \
                      cursor_lock_unsel_color cursor_lock_sel_color}

    set x_base [expr $data(-sigpane_width) * 2]
    #**** Figure out where the floor is
    set vert_floor [expr $data(height) - \
            ($data(-trace_height) * (1 + $data(-num_cursors)))]


    for {set i 0} {$i < $data(-num_cursors)} {incr i} {

        if {($i & 1) == 1} {
            set width 2.0
        } else {
            set width 1.0
        }

        #*** Descender goes to floor - (height)*i
        set cx [expr $x_base + ($data(-vertbar_sep)*$i + $data(-vertbar_sep)/2)]

        set cval [GetConfigValue $w [lindex $color_names $i]]

        $data(w:c) create line $cx 0 $cx \
            [expr $vert_floor + $data(-trace_height)*$i + 2] -fill $cval \
            -width $width

        set ttext "[expr 10*$i + 5] ns"

        set twidth 40

        if {$twidth % 2} {
            incr twidth
        }

        $data(w:c) create rect \
            [expr $cx - $twidth/2] \
            [expr $vert_floor + $data(-trace_height)*$i + 2] \
            [expr $cx + $twidth/2] \
            [expr $vert_floor + $data(-trace_height)*($i+1) - 2] \
            -outline $cval -width $width

        $data(w:c) create text $cx \
        [expr $vert_floor + $data(-trace_height)*$i + $data(-trace_height)/2] \
        -text $ttext -fill $cval
    }
}

#********************************************************************
#* TraceCenter
#********************************************************************
proc WaveWidgetPrefs::TraceCenter {w tno} {
    upvar #0 $w data

    return [expr $data(-trace_height)*$tno + $data(-trace_height)/2 + 5]
}

#********************************************************************
#* DrawTraces
#********************************************************************
proc WaveWidgetPrefs::DrawTraces {w} {
    upvar #0 $w data

    set x_base [expr $data(-sigpane_width) * 2]
    set x_max  $data(width)

    #*** Draw the scalar trace first... Must show all possible values
    DrawScalarTrace $w $x_base [TraceCenter $w 0] {0 1 0 x z x 0 1} 1

    #*** Next, draw the separator
    $data(w:c) create line $x_base [TraceCenter $w 1] $x_max \
        [TraceCenter $w 1] -fill [GetConfigValue $w separator_color]

    #*** Finally, draw the expanded vector trace
    DrawExpVectorTrace $w $x_base [TraceCenter $w 2] {00 01 10 1z zz zx xx 00}
}

#********************************************************************
#* UpdateWindow
#********************************************************************
proc WaveWidgetPrefs::UpdateWindow {w} {
    upvar #0 $w data

    set siglist_fg  [GetConfigValue $w siglist_fg]

    set reqwidth  [winfo reqwidth $data(w:c)]
    set reqheight [winfo reqheight $data(w:c)]

    set width  [winfo width $data(w:c)]
    set height [winfo height $data(w:c)]

    if {$width > $reqwidth} {
        set data(width) $width
    } else {
        set data(width) $reqwidth
    }

    if {$height > $reqheight} {
        set data(height) $height
    } else {
        set data(height) $reqheight
    }

    $data(w:c) delete all

    set wavewin_bg  [GetConfigValue $w wavewin_bg]
    set scale_color [GetConfigValue $w scale_color]

    #*** First, draw the sig, val, and wave-window panes
    DrawSigPanes $w

    set two_panew [expr $data(-sigpane_width) * 2]
    $data(w:c) create rectangle $two_panew 0 $data(width) $data(height) \
         -fill $wavewin_bg

    #*** Next, draw the backdrop lines...
    DrawBackdrop $w

    DrawTraces $w

    if {0} {
    #*** Draw traces...
    DrawScalarTrace $w $two_panew 2 {0 1 0 1 1 0 0}
    DrawScalarTrace $w $two_panew [expr 4 + $data(-trace_height)] \
            {x z z z x x x}
    DrawVectorTrace $w $two_panew [expr 6 + $data(-trace_height)*2] \
            {0 1 0 1 0 1 0}
    DrawVectorTrace $w $two_panew [expr 8 + $data(-trace_height)*3] \
            {z z z x x z x}
    }

    #*** Draw cursors
    DrawCursors $w

    $data(w:c) create line $two_panew 0 $two_panew $data(height) \
            -width 2.0 -fill grey95

    set strs {"a" "b" "c\[3:0\]" "d\[7:0\]"}

    for {set i 0} {$i < [llength $strs]} {incr i} {
        $data(w:c) create text 5 [expr 10 + 2*$i + $data(-trace_height)*$i] \
        -fill $siglist_fg -anchor nw -text [lindex $strs $i]
    }
}

#********************************************************************
#* SchemeNameValidate
#********************************************************************
proc WaveWidgetPrefs::SchemeNameValidate {w text} {
    upvar #0 $w data

    set ret [regexp {^[a-zA-Z0-9][a-zA-Z0-9_\-]*$} $text]

    if {$ret == 1 && [string length $text] > 0} {
        #*** Activate buttons
        $data(w:newscheme) configure -state normal
#        $data(w:namescheme) configure -state normal
    } elseif {[string length $text] == 0} {
        #*** Deactivate buttons
        $data(w:newscheme) configure -state disabled
#        $data(w:namescheme) configure -state disabled
        set ret 1
    }

    return $ret
}

#********************************************************************
#* NewScheme
#********************************************************************
proc WaveWidgetPrefs::NewScheme {w} {
    upvar #0 $w data

    set scheme_name [$data(w:schemename) get]

    #*** Check to ensure that name doesn't exist...


    #*** Add new scheme, based on current one
    set new_scheme  $data(configitems)
    lappend data(color_schemes) $scheme_name
    $data(w:schemelist) insert end $scheme_name
    lappend data(color_scheme_data) $new_scheme

    $data(w:schemelist) selection clear 0 end
    $data(w:schemelist) selection set end

    $data(w:schemename) delete 0 end

    $data(w:widget_default_scheme) configure -values $data(color_schemes)
    $data(w:this_window_scheme) configure -values $data(color_schemes)

    #*** Switch to new scheme
    SchemeListSelect $w
}

#********************************************************************
#* RenameScheme
#********************************************************************
proc WaveWidgetPrefs::RenameScheme {w} {
    upvar #0 $w data

    
}

