/****************************************************************************
 *                             ivi_wrapper.c
 *
 * Author: Matthew Ballance
 * Desc:   Implements an executable wrapper around the _real_ IVI executable.
 *         This script may be used on platforms without scripting langauges.
 *
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include "config.h"

static char tmpbuf[16384];
static int  verbose = 0;
static int  run_gdb = 0;
static int  run_valgrind = 0;

#ifndef PROGRAM
#ifdef __MINGW32__
#define PROGRAM "ivi_bin.exe"
#else 
#define PROGRAM "ivi_bin"
#endif
#endif

#define TK_VER        "tk8.4"
#define TCL_VER       "tcl8.4"

/********************************************************************
 * file_exists()
 ********************************************************************/
static int file_exists(char *file)
{
    FILE *fp = 0;

    if ((fp = fopen(file, "r"))) {
        fclose(fp);
        return 1;
    } 

    return 0;
}

/********************************************************************
 * process_relative_path()
 ********************************************************************/
static void process_relative_path(char *full_name, char **real_path_p)
{
    int       idx = 0;
    int       end_ptr;
    char     *save_fullname = (char *)malloc(strlen(full_name)+1);
    char     *real_path;
    char      sep_char = SEP_CHAR;

    if (strchr(full_name, '/')) {
        sep_char = '/';
    }

    strcpy(save_fullname, full_name);

    if (!strncmp(full_name, "..", 2)) {
        getcwd(tmpbuf, 16384);

        real_path = (char *)malloc(strlen(full_name)+strlen(tmpbuf)+16);
        sprintf(real_path, "%s%c%s", tmpbuf, sep_char, full_name);
    } else if (full_name[0] == '.' && full_name[1] == sep_char) {
        getcwd(tmpbuf, 16384);
        real_path = (char *)malloc(
                strlen(full_name)+strlen(tmpbuf));
        sprintf(real_path, "%s%s", tmpbuf, &full_name[1]);
    } else {
        fprintf(stderr, "incomprehensible path - %s\n", full_name);
    }

    *real_path_p = real_path;

    free(save_fullname);
}

/********************************************************************
 * find_fullpath()
 ********************************************************************/
static int find_fullpath(char *exec_name, char **full_execname)
{
    char *path_env, *end_ptr;
    char  sep_char = SEP_CHAR;

    fprintf(stderr, "find_fullpath(%s)\n", exec_name);

    if (file_exists(exec_name)) {
        if (exec_name[0] == '.') {
            process_relative_path(exec_name, full_execname);
        } else {
            *full_execname = exec_name;
        }
        return 1;
    }

    if (strchr(exec_name, '/')) {
        sep_char = '/';
    }

    /**** Otherwise, must search the path... ****/
    if ((path_env = getenv("PATH"))) {

        while ((end_ptr = strchr(path_env, PATHSEP_CHAR))) {
            memcpy(tmpbuf, path_env, end_ptr-path_env);

            tmpbuf[end_ptr-path_env] = sep_char;
            tmpbuf[end_ptr-path_env+1] = 0;

            strcat(tmpbuf, exec_name);
            if (file_exists(tmpbuf)) {
                *full_execname = strdup(tmpbuf);
                return 1;
            }

            path_env = end_ptr+1;
        }

        /**** Now, we have one last element in the path... ****/
        strcpy(tmpbuf, path_env);
        strcat(tmpbuf, exec_name);

        if (file_exists(tmpbuf)) {
            *full_execname = strdup(tmpbuf);
            return 1;
        }
    }

    return 0;
}

/********************************************************************
 * get_ivihome()
 ********************************************************************/
static void get_ivihome(char *full_path, char **ivi_home_p)
{
     char *ivi_home = (char *)malloc(strlen(full_path)+1);
     char *end_ptr;
     char  sep_char = SEP_CHAR;

     if (strchr(full_path, '/')) {
         sep_char = '/';
     }

     strcpy(ivi_home, full_path);

     end_ptr = &ivi_home[strlen(full_path)];

     end_ptr--;
     if (*end_ptr == sep_char) {
         end_ptr--;
     }

     while ((*end_ptr != sep_char) && (end_ptr > ivi_home)) { end_ptr--; }
     end_ptr--;

     while ((*end_ptr != sep_char) && (end_ptr > ivi_home)) { end_ptr--; }

     *end_ptr = 0;

     *ivi_home_p = ivi_home;
}

/********************************************************************
 * put()
 ********************************************************************/
static void put(const char *var, const char *val)
{
    char *pval = (char *)malloc(strlen(var)+strlen(val)+16);

    sprintf(pval, "%s=%s", var, val);
    putenv(pval);
}

/********************************************************************
 * append_path()
 ********************************************************************/
static void append_path(const char *var, const char *add)
{
    char *curr_val;
    char *new_val;

    if (!(curr_val = getenv(var))) {
        put(var, add);
    } else {
        new_val = (char *)malloc(strlen(curr_val)+strlen(add)+strlen(var)+4);
        sprintf(new_val, "%s=%s%c%s", var, curr_val, PATHSEP_CHAR, add);
        putenv(new_val);
    }
}

/********************************************************************
 * add_lib_paths
 ********************************************************************/
static void add_lib_paths(char *ivi_home)
{
    char sep_char = SEP_CHAR;

    if (strchr(ivi_home, '/')) {
        sep_char = '/';
    }

    sprintf(tmpbuf, "%s%clib", ivi_home, sep_char);
    append_path(LIBPATH_VAR, tmpbuf);

    sprintf(tmpbuf, "%s%cmodules%civi_common", ivi_home, sep_char, sep_char); 

    append_path(LIBPATH_VAR, tmpbuf);

    /*
    sprintf(tmpbuf, "%s%clib%civi", ivi_home, SEP_CHAR, SEP_CHAR); 
    append_path(LIBPATH_VAR, tmpbuf);
     */
}

/********************************************************************
 * main()
 ********************************************************************/
int main(int argc_i, char **argv_i)
{
    int     ret, i;
    int     argc = 0;
    char   *argv[256], *full_cmd, *full_exec;
    char   *argv2[256];
    char   *ivi_home;
    char    sep_char = SEP_CHAR;

    /**** Pre-parse the options for stuff like -v ****/
    for (i=0; i<argc_i; i++) {
        if (!strcmp(argv_i[i], "-v")) {
            verbose = 1;
        } else 
        if (!strcmp(argv_i[i], "--gdb")) {
            run_gdb = 1;
        } else 
        if (!strcmp(argv_i[i], "--valgrind")) {
            run_valgrind = 1;
        } else {
            argv[argc++] = argv_i[i];
        }
    }

    /**** First, we must find the full pathname of this executable ****/
    if (find_fullpath(argv[0], &full_exec)) {
        if (verbose) fprintf(stderr, "full_exec = %s\n", full_exec);
    } else {
        fprintf(stderr, "ERROR: Cannot find exec-path of %s\n", argv[0]);
    }

    get_ivihome(full_exec, &ivi_home);

    if (verbose) fprintf(stderr, "ivi_home = %s\n", ivi_home);

    put("IVI_HOME", ivi_home);
    if (verbose) fprintf(stderr, "IVI_HOME = %s\n", ivi_home);

    /**** Append locations to LD_LIBRARY_PATH/PATH... ****/
    add_lib_paths(ivi_home);

    if (strchr(ivi_home, '/')) {
        sep_char = '/';
    }

    /**** Set TK/TCL_LIBRARY ****/
    sprintf(tmpbuf, "%s%clib%c%s", ivi_home, sep_char, sep_char,
            TCL_VER);
    put("TCL_LIBRARY", tmpbuf);

    sprintf(tmpbuf, "%s%clib%c%s", ivi_home, sep_char, sep_char,
            TK_VER);
    put("TK_LIBRARY", tmpbuf);


    full_cmd = (char *)malloc(strlen(ivi_home)+strlen(PROGRAM)+16);
    sprintf(full_cmd, "%s%cbin%c%s", ivi_home, sep_char, sep_char, PROGRAM);

    argv[0] = full_cmd;
    argv[argc] = 0;

    if (run_gdb) {
        char *gdb_path;

        if (!find_fullpath(
#ifndef __MINGW32__
                    "gdb", 
#else
                    "gdb.exe", 
#endif
                    &gdb_path)) {
            fprintf(stderr, "Cannot find gdb\n");
            exit(1);
        }

        /**** This keeps gdb from re-scanning our shell's startup file
         ****/
        put("SHELL", "sh");

        argv2[0] = gdb_path;
        for (i=0; i<argc; i++) {
            argv2[i+1] = argv[i];
        }
        argv2[argc+1] = 0;

        ret = execv(gdb_path, argv2);
        perror("execv");
    } else if (run_valgrind) {
        char *vg_path;

        if (!find_fullpath("valgrind", &vg_path)) {
            fprintf(stderr, "Cannot find valgrind\n");
            exit(1);
        }

        argc++;
        for (i=0; i<argc; i++) {
            argv2[argc-i-1] = argv[argc-i-2];
        }
        argv2[0] = vg_path;
        argv2[argc] = 0;

        ret = execv(vg_path, argv2);
    } else {
        if (verbose) fprintf(stderr, "full_cmd = %s\n", full_cmd);
        ret = execv(full_cmd, argv);
    }

    fprintf(stderr, "ERROR: Exec failed (%d)\n", ret);
    return ret;
}

