/****************************************************************************
 *                             StimTraceRdr.cc
 *
 * Author: Matthew Ballance
 * Desc:   Generates stimulus on a design signal based on a DFIO trace...
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "StimTraceRdr.h"
#include "StimulusMgr.h"
#include <string.h>
#include "vector.h"
#include "BitVector.h"
#include "DFIOTrace.h"
#include "IviSim.h"

/********************************************************************
 * StimTraceRdr()
 ********************************************************************/
StimTraceRdr::StimTraceRdr(
        vpiHandle        vpiHndl,
        DFIOTrace       *trace_data,
        StimulusMgr     *mgr) : 
        d_trace(trace_data), d_mgr(mgr), d_vpiHndl(vpiHndl),
        d_maxTime(0), d_maxTrans(0), d_offset(0), d_startTime(0),
        d_traceTime(0)
{
    vpiHandle        tHndl;
    s_vpi_value      v_val;
    Uint32           msb, lsb;

    cb_time.type = vpiSimTime;
    cb_time.low  = 0;
    cb_time.high = 0;

    sim = mgr->getSim();

    memset(&cb_data, 0, sizeof(cb_data));
    cb_data.reason     = cbReadWriteSynch;
    cb_data.time       = &cb_time;
    cb_data.cb_rtn     = StimTraceRdr::DataCB;
    cb_data.user_data  = (char *)this;

    v_val.format = vpiIntVal;
    tHndl = sim->vpi_handle(vpiRightRange, vpiHndl);
    sim->vpi_get_value(tHndl, &v_val);
    msb = v_val.value.integer;
    tHndl = sim->vpi_handle(vpiLeftRange, vpiHndl);
    sim->vpi_get_value(tHndl, &v_val);
    lsb = v_val.value.integer;
    d_sigwidth = (msb>lsb)?(msb-lsb+1):(lsb-msb+1);

    mgr->addTraceInst(this);
}

/********************************************************************
 * ~StimTraceRdr()
 ********************************************************************/
StimTraceRdr::~StimTraceRdr()
{
    d_mgr->delTraceInst(this);
}

/********************************************************************
 * Start()
 ********************************************************************/
void StimTraceRdr::Start()
{
    if (d_offset > 0) {
        /**** Must wait for 'offset' before starting the initial delay...
         ****/

        /**** Leave... ****/
    } else if (d_offset < 0) {
        /**** Need to scan ahead in the trace until we reach???
         ****/
    }

    StimTraceRdr::DataCB(&cb_data);
}

/********************************************************************
 * DataCB()
 ********************************************************************/
int StimTraceRdr::DataCB(p_cb_data  cb_data_p)
{
    StimTraceRdr          *rdr = (StimTraceRdr *)cb_data_p->user_data;
    Vector<DFIOValChg>    *vals;
    Uint32                 t_delta;
    s_vpi_time             time_s;
    s_vpi_value            value_s;

    /**** Grab the change we're after and the one after... ****/
    vals = rdr->d_trace->getValue(rdr->d_traceTime, 
            rdr->d_traceTime, DFIOTrace::GVF_OnePlus);

    if (!vals->length()) {
        fprintf(stderr, "StimTraceRdr Internal Error :: 0-len val-vect\n");
        return 0;
    }

    /**** Drive out the current value... ****/
    BitVector::get_vecval(
            vals->idx(0)->bitVal,
            rdr->d_sigwidth, /* requested len */
            0,
            &value_s.value.vector);

    value_s.format = vpiVectorVal;
    rdr->sim->vpi_put_value(rdr->d_vpiHndl, &value_s, NULL, vpiNoDelay);

    /**** If length > 1, then there's a new time to go to... ****/
    if (vals->length() > 1) {
        time_s.type = vpiSimTime;
        rdr->sim->vpi_get_time(0, &time_s);

        /**** Find the delta between this value's time and the time of
         **** the next value. The callback goes this_time+delta
         ****/
        t_delta = vals->idx(1)->changeTime-vals->idx(0)->changeTime;

        rdr->d_traceTime += t_delta;
        rdr->cb_time.high = 0;
        rdr->cb_time.low  = time_s.low + t_delta;

        rdr->sim->vpi_register_cb(&rdr->cb_data);
    } else {
        /**** Okay, we're done... ****/


    }

    return 0;
}

