/*****************************************************************************
 *                                Stack.h
 *
 * Author: Matthew Ballance
 * Desc:   Implements a templatized stack class
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 *****************************************************************************/
#ifndef INCLUDED_STACK_H
#define INCLUDED_STACK_H
#include "types.h"

template <class eType> class Stack {

    public:
        Stack(void);
        virtual ~Stack(void);

        void push(eType   *newEl);
        eType *pop(void);
        eType *top(void);

        inline Uint32 depth(void) const { return numEl; }

    private:
        struct StackEl {
            StackEl() : element(0), next(0) { ; }
            eType           *element;
            struct StackEl  *next;
        };

        struct StackEl      *storage;
        struct StackEl      *free_store;
        Uint32               numEl;
};


/*******************************************************************
 * Stack()
 *******************************************************************/
template <class eType> Stack<eType>::Stack(void)
{
    numEl       = 0;
    storage     = 0;
    free_store  = 0;
}

/*******************************************************************
 * ~Stack()
 *******************************************************************/
template <class eType> Stack<eType>::~Stack(void)
{
    struct StackEl   *ptr;

    while (storage) {
        ptr = storage;
        storage = storage->next;
        delete ptr;
    }

    while (free_store) {
        ptr = free_store;
        free_store = free_store->next;
        delete ptr;
    }
}

/*******************************************************************
 * push()
 *******************************************************************/
template <class eType> void Stack<eType>::push(eType *newEl)
{
    struct StackEl   *container;

    if (free_store) {
        container = free_store;
        free_store = free_store->next;
    } else {
        container = new StackEl;
    }

    container->element = newEl;
    container->next = storage;
    storage = container;
    numEl++;
}

/*******************************************************************
 * pop()
 *******************************************************************/
template <class eType> eType *Stack<eType>::pop(void)
{
    struct StackEl   *container;
    eType *ret;

    if (!storage) {
        return 0;
    }

    container = storage;
    storage = storage->next;
    ret = container->element;

    /**** Put the container on the free store ****/
    container->next = free_store;
    free_store = container;

    numEl--;

    return ret;
}

/*******************************************************************
 * top()
 *******************************************************************/
template <class eType> eType *Stack<eType>::top(void)
{
    struct StackEl   *container;

    if (!storage) {
        return 0;
    }
    container = storage;
    return container->element;
}


#endif /* INCLUDED_STACK_H */
