/****************************************************************************
 *                                IviSim.cc
 *
 * Author: Matthew Ballance
 * Desc:   Description of the simulator-interface object used by IVI
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "IviSim.h"
#include "CmdSwitcher.h"
#include "WidgetManager.h"
#include "BitVector.h"
#include "DesignDB.h"
#include "vpi_user.h"
#include "Fifo.h"
#include <string.h>
#include <stdlib.h>
#include <tk.h>
#include "tclOldConfig.h"
#include "TclChannelObj.h"

/********************************************************************
 * IviSim_TclCmd()
 ********************************************************************/
static int IviSim_TclCmd(
        ClientData        clientData,
        Tcl_Interp       *interp,
        int               argc,
        char            **argv)
{
    IviSim   *sim = (IviSim *)clientData;
    int       ret;

    Tcl_Preserve(clientData);

    if (sim) {
        ret = sim->InstCmd(argc-1, &argv[1]);
    } else {
        Tcl_AppendResult(interp, "NULL sim handle", 0);
        ret = TCL_ERROR;
    }

    Tcl_Release(clientData);
    return ret;
}

/********************************************************************
 * IviSim()
 ********************************************************************/
IviSim::IviSim(Tcl_Interp *interp, int argc, char **argv) : 
    instName(argv[1])
{
    /**** Create an instance command... ****/
    d_command = Tcl_CreateCommand(interp, argv[1], 
            (Tcl_CmdProc *)IviSim_TclCmd, this, 0);

    Tcl_SetResult(interp, argv[1], TCL_VOLATILE);

    this->interp = interp;
    finished = 0;

    ddb         = 0;
    ddbName     = 0;

    d_simResolution = -9;
    d_resMultiplier = 1;

    WidgetMgr_AddInst(interp, WIDGET_TYPE_IVI_SIM, argv[1], this);

    Configure(argc-2, &argv[2], 0);
}

/*****************************************************************
 * ~IviSim()
 *****************************************************************/
IviSim::~IviSim(void)
{
    Tcl_DeleteCommandFromToken(interp, d_command);
}

/********************************************************************
 * IviSim_McdCB()
 *
 * Callback activated by the simulator when a MCD channel has output
 * This callback must reflect the string output to the specified 
 * TCL channel...
 ********************************************************************/
static void IviSim_McdCB(voidPtr userData, Char *msg)
{
    TclChannelObj *chan = (TclChannelObj *)userData;

    chan->write(msg);
}

static Char *prv_ResStrings[] = {
/*    0   1    2     3     4     5     6     7     8     9     10    11    12 */
    "s", "s", "ms", "ms", "us", "us", "us", "ns", "ns", "ns", "ps", "ps", "ps",
     ""
};

typedef enum {
    IVC_LoadDesign = 1,
    IVC_Run,
    IVC_Time,
    IVC_FormatTime,
    IVC_OpenMcd,
    IVC_GetStdout,
    IVC_Close,
    IVC_Finished,
    IVC_Resolution,
    IVC_ResString,
    IVC_Inspect,
    IVC_SetVal,
    IVC_Configure,
    IVC_Cget,
    IVC_NumCmds
} IviSimCmds;

static CmdSwitchStruct  ivi_sim_cmds[] = {
    {"load_design",           IVC_LoadDesign},
    {"run",                   IVC_Run       },
    {"time",                  IVC_Time      },
    {"format_time",           IVC_FormatTime  },
    {"open_mcd",              IVC_OpenMcd   },
    {"get_stdout",            IVC_GetStdout },
    {"close",                 IVC_Close     },
    {"finished",              IVC_Finished  },
    {"resolution",            IVC_Resolution},
    {"res_string",            IVC_ResString },
    {"inspect",               IVC_Inspect   },
    {"ins",                   IVC_Inspect   },
    {"set_val",               IVC_SetVal    },
    {"configure",             IVC_Configure },
    {"config",                IVC_Configure },
    {"cget",                  IVC_Cget      },
    { "",                     0             }
};

/********************************************************************
 * InstCmd()
 ********************************************************************/
int IviSim::InstCmd(int argc, char **argv)
{
    Int32         cmd, ret, res;
    s_vpi_time    tt;
    TclChannelObj *chan = 0;

    if (argc < 1) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    cmd = CmdSwitch(ivi_sim_cmds, argv[0]);

    switch (cmd) {
        case IVC_LoadDesign:
            if (argc < 2) {
                Tcl_AppendResult(interp, "wrong # args", 0);
                return TCL_ERROR;
            }

            if (LoadDesign(argv[1]) < 0) {
                fprintf(stderr, "ERROR :: while loading design\n");
                return TCL_ERROR;
            } else {
                fprintf(stderr, "NOTE :: Loaded design \"%s\"\n", argv[1]);
            }

            break;

        case IVC_Run:
            ret = Run(strtoul(argv[1], 0, 0));
            if (ret == 1) {
                finished = 1;
            }
            tt.type = vpiSimTime;
            vpi_get_time(0, &tt);
            Tcl_SetObjResult(interp, Tcl_NewIntObj(tt.low));
            break;

        case IVC_Finished:
            Tcl_SetObjResult(interp, Tcl_NewIntObj((finished)?1:0));
            break;

        case IVC_Time:
            tt.type = vpiSimTime;
            vpi_get_time(0, &tt);
            Tcl_SetObjResult(interp, Tcl_NewIntObj(tt.low));
            break;

        case IVC_FormatTime: {
            FormatTimeCmd();
            } break;

        case IVC_GetStdout:
            {
               const char *cname = GetStdoutChannel();

               Tcl_SetResult(interp, (cname)?(char *)cname:(char *)"", 0);
            }
            break;

        /**** Open a TCL channel for output from a particular MCD channel ****/
        /**** open_mcd <ChannelName> <ChannelId>
         ****/
        case IVC_OpenMcd:
            if (argc < 3) {
                Tcl_AppendResult(interp, "too few args", 0);
                return TCL_ERROR;
            }
            chan = new TclChannelObj(interp, argv[1], 
                    "McdChannel", TCL_READABLE);
            if (!chan->ok) {
                Tcl_AppendResult(interp, "problem creating channel ",
                        argv[1], 0);
                return TCL_ERROR;
            }
            RegisterMsgCB((1 << strtoul(argv[2], 0, 0)),
                    IviSim_McdCB, chan, 0);
            Tcl_SetObjResult(interp, Tcl_NewStringObj(argv[1], -1));
            break;

        case IVC_Close:
            if (Close() != TCL_OK) {
                return TCL_ERROR;
            }
            break;

        case IVC_Resolution:
            res = getSimResolution();
            Tcl_SetObjResult(interp, Tcl_NewIntObj(res));
            break;

        case IVC_ResString:
            if (argc > 1) {
                res = atoi(argv[1]);
            } else {
                res = getSimResolution();
            }
            if (res >= 0) {
                fprintf(stderr, "ERROR :: time precision is >= 0 (%d)\n", res);
                return TCL_ERROR;
            } else {
                res = (-res);
            }

            Tcl_SetObjResult(interp, Tcl_NewStringObj(prv_ResStrings[res], -1));
            break;

        case IVC_Configure:
            if (Configure(argc-1, &argv[1], TK_CONFIG_ARGV_ONLY) != TCL_OK) {
                return TCL_ERROR;
            }
            break;

        case IVC_Cget:
            return Tcl_ConfigureValue(interp, 0, getConfigSpec(),
                    (char *)this, argv[1], 0);
            break;

        case IVC_Inspect:
            return Inspect(argc-1, &argv[1]);
            break;

        case IVC_SetVal:
            break;

        default:
            Tcl_AppendResult(interp, "unknown command ", argv[0], 0);
            return TCL_ERROR;
            break;
    }

    return TCL_OK;
}

/********************************************************************
 * Configure()
 ********************************************************************/
int IviSim::Configure(int argc, char **argv, int flags)
{
    Int32 ret;

    ret = Tcl_ConfigureWidget(interp, 0, getConfigSpec(), argc, argv,
            (char *)this, flags);

    if (ret != TCL_OK) {
        return ret;
    }

    if (optionSpecified(OPT_DDB) && ddbName) {
        ddb = (DesignDB *)WidgetMgr_GetObjHandle(interp,
                WIDGET_TYPE_DDB, ddbName);

        if (!ddb) {
            fprintf(stderr, "ERROR :: Cannot locate DDB \"%s\"\n", ddbName);
            Tcl_AppendResult(interp, "no DDB named ", ddbName, 0);
            return TCL_ERROR;
        }
    }

    return TCL_OK;
}

/********************************************************************
 * Inspect()
 ********************************************************************/
int IviSim::Inspect(int argc, char **argv)
{
    Uint32               radix = BitVector::Radix_Bin;
    Uint32               i;
    Char                *sig;
    DesignDB            *ddb;
    Vector<TreeNode>    *sigs;
    vpiHandle            vpiHndl;
    s_vpi_value          value_s;

    if (argc < 1) {
        Tcl_AppendResult(interp, "too few args", 0);
        return TCL_ERROR;
    }

    sig = argv[0];

    for (i=1; i<argc; i++) {
        if (argv[i][0] == '-') {
            if (String::equal(argv[i], "-hex")) {
                radix = BitVector::Radix_Hex;
            } else if (String::equal(argv[i], "-dec")) {
                radix = BitVector::Radix_Dec;
            } else if (String::equal(argv[i], "-oct")) {
                radix = BitVector::Radix_Oct;
            } else if (String::equal(argv[i], "-bin")) {
                radix = BitVector::Radix_Bin;
            } else {
                Tcl_AppendResult(interp, "unknown option ", argv[i], 0);
                return TCL_ERROR;
            }
        }
    }

    sigs = ddb->globElems(NULL, sig, "@signals@", 0);
    if (!sigs->length()) {
        Tcl_AppendResult(interp, sig, "doesn't match any signals", 0);
        return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * Delete()
 ********************************************************************/
void IviSim::Delete()
{
    Tcl_EventuallyFree(this, &IviSim::Delete);
//    Tcl_DoWhenIdle(&IviSim::Delete, this);
}

/********************************************************************
 * Delete()
 ********************************************************************/
void IviSim::Delete(char *userData)
{
    IviSim *sim = (IviSim *)userData;

    delete sim;
}

/********************************************************************
 * FormatTimeCmd()
 ********************************************************************/
void IviSim::FormatTimeCmd()
{
    char         buf[128];
    s_vpi_time   tt;
    int          res = -(getSimResolution());

    tt.type = vpiSimTime;
    vpi_get_time(0, &tt);

    while ((tt.low > 0) && !(tt.low % 1000)) {
        if (res < 3) {
            break;
        } else {
            tt.low /= 1000;
            res -= 3;
        }
    }

    sprintf(buf, "%d%s", tt.low, prv_ResStrings[res]);

    Tcl_SetObjResult(interp, Tcl_NewStringObj(buf, -1));
}

/********************************************************************
 * setSimResolution()
 ********************************************************************/
void IviSim::setSimResolution(Int32 resolution)
{
    static Int32 mult_table[] = {
        /* s  100ms  10ms  ms  100us 10us  us  100ns 10ns 1ns 100ps 10ps      */
           1,  100,   10,  1,  100,   10,   1,  100,  10,  1,  100, 10, 
        /* 1ps 100fs 10fs  1fs                                                */
           1,   100,  10,   1,  100, 10, 1
    };
    static Int32 res_table[] = {
        /* s  .1s  .01s  ms  100us 10us  us  100ns 10ns 1ns  100ps 100ns      */
           0,  -3,  -3,  -3,  -6,   -6,  -6,  -9,   -9,  -9,  -12,  -12,
        /* 1ps  100fs  10fs  1fs  */
           -12,  -15,  -15,  -15, -18, -18, -18
    };
    Int32 m_idx;
   
    m_idx = resolution;

    if (m_idx < 0) {
        m_idx = -m_idx;
    }

    d_resMultiplier = mult_table[m_idx];
    d_simResolution = res_table[m_idx];
}

/********************************************************************
 * getSimResolution()
 ********************************************************************/
Int32 IviSim::getSimResolution()
{
    return d_simResolution;
}

/********************************************************************
 * setSimMultiplier()
 ********************************************************************/
void IviSim::setResMultiplier(Int32 mult)
{
    d_resMultiplier = mult;
}

/********************************************************************
 * getResMultiplier()
 ********************************************************************/
Int32 IviSim::getResMultiplier()
{
    return d_resMultiplier;
}

/********************************************************************
 * getConfigSpec()
 ********************************************************************/
Tk_ConfigSpec *IviSim::getConfigSpec()
{
    static Tk_ConfigSpec   sdbConfigSpec[] = {
        {TK_CONFIG_STRING, "-ddb", "ddb", "DDB",
            (char *)NULL, Tk_Offset(IviSim, ddbName), 0},
        {TK_CONFIG_END, (char *)NULL, (char *)NULL, (char *)NULL,
            (char *)NULL, 0, 0}
    };

    return sdbConfigSpec;
}

/********************************************************************
 * GetVpiObjString()
 *
 * Returns the string description of the object type
 ********************************************************************/
char *IviSim::GetVpiObjString(Uint32 objCode)
{
    char *ret = "unknown";

    switch (objCode) {
        case vpiConstant: ret = "vpiConstant"; break;
        case vpiFunction: ret = "vpiFunction"; break;
        case vpiIntegerVar: ret = "vpiIntegerVar";   break;
        case vpiIterator: ret = "vpiIterator"; break;
        case vpiMemory:   ret = "vpiMemory";   break;
        case vpiMemoryWord: ret = "vpiMemoryWord"; break;
        case vpiModule:   ret = "vpiModule";   break;
        case vpiNamedBegin: ret = "vpiNamedBegin"; break;
        case vpiNamedFork: ret = "vpiNamedFork";     break;
        case vpiNet:      ret = "vpiNet";            break;
        case vpiReg:      ret = "vpiReg";            break;
        case vpiSysFuncCall: ret = "vpiSysFuncCall"; break;
        case vpiSysTaskCall: ret = "vpiSysTaskCall"; break;
        case vpiTask:        ret = "vpiTask";        break;
        case vpiTimeVar:     ret = "vpiTimeVar";     break;
        default: break;
    }


    return ret;
}

/********************************************************************
 * RegisterMsgCB()
 *
 * Registers a message-callback function on a specified set 
 * of mcd channels. The given function will be called 
 * whenever data is sent to the specified channel.
 *
 * Parameters:
 * - mcd
 *
 * - cbFunc
 *
 * - userData
 *
 * - clear_others
 *   If this flag is set, any existing files or functions to 
 *   which the output is directed will be closed or removed.
 ********************************************************************/
int IviSim::RegisterMsgCB(
        Uint32                   mcd,
        mcd_cb_func              cbFunc,
        void                    *userData,
        Uint32                   clear_others)
{
    return -1;
}

/************************************************************
 * Run()
 *
 * Run the simulation ahead for some number of time units.
 ************************************************************/
int IviSim::Run(Uint32 max_time)
{
    return -1;
}

/************************************************************
 * ErrorStr()
 *
 * Returns the most recent error, or NULL if there are no
 * errors.
 ************************************************************/
char *IviSim::ErrorStr()
{
    return 0;
}

/************************************************************
 * LoadDesign()
 *
 * Loads a design into this instance of the VVPSim.
 ************************************************************/
int IviSim::LoadDesign(char *design_path)
{
    return -1;
}

/************************************************************
 * AppendModulePath()
 *
 * Appends an path to the module-path
 ************************************************************/
int IviSim::AppendModulePath(char *path)
{
    return -1;
}

/************************************************************
 * LoadVPIModule()
 *
 * Loads a VPI module into this instance of VVPSim
 ************************************************************/
int IviSim::LoadVPIModule(char *vpi_module)
{
    return -1;
}

/************************************************************
 * GetStdoutChannel()
 ************************************************************/
const char *IviSim::GetStdoutChannel()
{
    return 0;
}

/************************************************************
 * Close()
 ************************************************************/
int IviSim::Close()
{
    return TCL_OK;
}

/********************************************************************
 * vpi_register_systf
 ********************************************************************/
void IviSim::vpi_register_systf(const struct t_vpi_systf_data *ss)
{
    /**** Ignore... ****/
}

/********************************************************************
 * vpi_printf
 ********************************************************************/
void IviSim::vpi_printf(const char*fmt, ...)
{
}

/********************************************************************
 * vpi_mcd_close()
 ********************************************************************/
Uint32 IviSim::vpi_mcd_close(Uint32 mcd)
{
    return 0;
}

/********************************************************************
 * vpi_mcd_name
 ********************************************************************/
Char *IviSim::vpi_mcd_name(Uint32 mcd)
{
    return 0;
}

/********************************************************************
 * vpi_mcd_open()
 ********************************************************************/
Uint32 IviSim::vpi_mcd_open(char *name)
{
    return 0;
}

/********************************************************************
 * vpi_mcd_open_x()
 ********************************************************************/
Uint32 IviSim::vpi_mcd_open_x(char *name, char *mode)
{
    return 0;
}

/********************************************************************
 * vpi_mcd_printf()
 ********************************************************************/
Int32 IviSim::vpi_mcd_printf(Uint32 mcd, const char*fmt, ...)
{
    return -1;
}

/********************************************************************
 * mcd_vprintf()
 ********************************************************************/
Int32 IviSim::vpi_mcd_vprintf(Uint32 mcd, const char*fmt, va_list ap)
{
    return -1;
}

/********************************************************************
 * vpi_mcd_fputc()
 ********************************************************************/
Int32 IviSim::vpi_mcd_fputc(Uint32 mcd, UChar x)
{
    return -1;
}

/********************************************************************
 * vpi_mcd_fgetc()
 ********************************************************************/
Int32 IviSim::vpi_mcd_fgetc(Uint32 mcd)
{
    return -1;
}

/********************************************************************
 * vpi_register_cb()
 ********************************************************************/
vpiHandle IviSim::vpi_register_cb(p_cb_data data)
{
    return 0;
}

/********************************************************************
 * vpi_remove_cb()
 ********************************************************************/
Int32 IviSim::vpi_remove_cb(vpiHandle ref)
{
    return -1;
}

/********************************************************************
 * vpi_sim_control()
 ********************************************************************/
void IviSim::vpi_sim_control(Int32 operation, ...)
{
}

/********************************************************************
 * vpi_handle()
 ********************************************************************/
vpiHandle IviSim::vpi_handle(Int32 type, vpiHandle ref)
{
    return 0;
}

/********************************************************************
 * vpi_iterate()
 ********************************************************************/
vpiHandle IviSim::vpi_iterate(Int32 type, vpiHandle ref)
{
    return 0;
}

/********************************************************************
 * vpi_scan()
 ********************************************************************/
vpiHandle IviSim::vpi_scan(vpiHandle iter)
{
    return 0;
}

/********************************************************************
 * vpi_handle_by_index()
 ********************************************************************/
vpiHandle IviSim::vpi_handle_by_index(vpiHandle ref, Int32 index)
{
    return 0;
}

/********************************************************************
 * vpi_get_time()
 ********************************************************************/
void IviSim::vpi_get_time(vpiHandle obj, s_vpi_time*t)
{
}
 
/********************************************************************
 * vpi_get()
 ********************************************************************/
Int32 IviSim::vpi_get(Int32 property, vpiHandle ref)
{
    return -1;
}

/********************************************************************
 * vpi_get_str()
 ********************************************************************/
Char* IviSim::vpi_get_str(Int32 property, vpiHandle ref)
{
    return 0;
}

/********************************************************************
 * vpi_get_value()
 ********************************************************************/
void IviSim::vpi_get_value(vpiHandle expr, p_vpi_value value)
{
}

/********************************************************************
 * vpi_put_value()
 ********************************************************************/
vpiHandle IviSim::vpi_put_value(vpiHandle obj, p_vpi_value value,
				     p_vpi_time when, Int32 flags)
{
    return 0;
}

/********************************************************************
 * vpi_free_object()
 ********************************************************************/
Int32 IviSim::vpi_free_object(vpiHandle ref)
{
    return -1;
}

/********************************************************************
 * vpi_get_vlog_info()
 ********************************************************************/
Int32 IviSim::vpi_get_vlog_info(p_vpi_vlog_info vlog_info_p)
{
    return -1;
}


