/****************************************************************************
 *                             IviRemoteProc.cc
 *
 * Author: Matthew Ballance
 * Desc:   Implements a remote process interface
 * <Copyright> (c) 2001-2003 Matthew Ballance (mballance@users.sourceforge.net)
 *
 *    This source code is free software; you can redistribute it
 *    and/or modify it in source code form under the terms of the GNU
 *    General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * </Copyright>
 ****************************************************************************/
#include "IviRemoteProc.h"
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>

#ifndef __MINGW32__
#include <sys/wait.h>
#endif

#include <fcntl.h>
#include <errno.h>
#include "TclChannelObj.h"
#include "ivi_String.h"
#include <ctype.h>
#include <tcl.h>
#include "ivi_print.h"

/********************************************************************
 * IviRemoteProc()
 ********************************************************************/
IviRemoteProc::IviRemoteProc(Tcl_Interp *interp, int argc, char **argv) : 
    d_interp(interp), d_pid(0), d_invokeStr(0)
{
    String tmpName;

    d_mergeStd = false;

    d_cmdToken = 0;

    if (!strcmp(argv[0], "ivi_proc")) {
        /**** Create a command for this proc object ****/
        d_cmdToken = Tcl_CreateObjCommand(d_interp, argv[1],
                &IviRemoteProc::InstCmd, this, 0);
    }

    /**** Create stdout pipe ****/
    tmpName = argv[1];
    tmpName += ".stdout";
    d_stdout = new TclChannelObj(d_interp, tmpName.value(),
            "ProcStdout", TCL_READABLE);

    d_running = false;

    d_procIdx = d_numProcs;

    d_procList[d_numProcs++] = this;
    d_conn = 0;
}

/********************************************************************
 * ~IviRemoteProc()
 ********************************************************************/
IviRemoteProc::~IviRemoteProc()
{
    if (isRunning()) {
        kill();
    }

    delete d_stdout;

    deleteProc(this);

    if (d_waitTimer) {
        Tcl_DeleteTimerHandler(d_waitTimer);
    }

    if (d_procOutput) {
        Tcl_UnregisterChannel(d_interp, d_procOutput);
//        Tcl_Close(d_interp, d_procOutput);
    }
}

/********************************************************************
 * setInvokeString()
 ********************************************************************/
void IviRemoteProc::setInvokeString(const char *invokeStr)
{
    char *is;

    if (d_invokeStr) {
        delete [] d_invokeStr;
    }

    is = new char[strlen(invokeStr)+1];
    strcpy(is, invokeStr);

    d_invokeStr = is;

    char *p = d_invokeStr;
    char *e;

    d_procArgs.setLength(0);

    while (*p) {
        /**** skip whitespace... ****/
        while (*p && isspace(*p)) { p++; }

        e = p;
        /**** Now move to the end of the token ****/
        while (*e && !isspace(*e)) { e++; }

        if (*p) {
            d_procArgs.append(p);
            if (*e) {
                p = e+1;
            } else {
                p = e;
            }
            *e = 0;
        }
    }

    d_procArgs.append((char *)0);
}

/********************************************************************
 * setInvoke()
 ********************************************************************/
void IviRemoteProc::setInvoke(int argc, char **argv)
{
    d_procArgs.setLength(0);
    for (int i=0; i<argc; i++) {
        char *tmp = new char [strlen(argv[i])+1];
        strcpy(tmp, argv[i]);
        d_procArgs.append(tmp);
    }

    d_procArgs.append((char *)0);
}

/********************************************************************
 * setInvoke()
 ********************************************************************/
void IviRemoteProc::setInvoke(Tcl_Obj *obj)
{
    int length = 0;

    d_procArgs.setLength(0);
    Tcl_ListObjLength(d_interp, obj, &length);

    for (int i=0; i<length; i++) {
        Tcl_Obj    *arg;
        Tcl_ListObjIndex(d_interp, obj, i, &arg);

        char *tmp = Tcl_GetString(arg);
        char *tmp2 = new char [strlen(tmp)+1];
        strcpy(tmp2, tmp);

        d_procArgs.append(tmp2);
    }

    d_procArgs.append((char *)0);
}

/********************************************************************
 * InputProc
 ********************************************************************/
void IviRemoteProc::InputProc(
        ClientData        clientData,
        int               mask)
{
#if 0
    int  bytes;
    char buf[256];
    IviRemoteProc *irp = (IviRemoteProc *)clientData;

    if (mask & TCL_READABLE) {
        bytes = read(irp->d_outFd[0], buf, 256);

        if (bytes > 0) {
            buf[bytes] = 0;
            irp->d_stdout->write(buf);
        } else if (bytes == 0) {
            /**** Delete event now... ****/
            fprintf(stderr, "EOF on IviRemoteProc\n");
#ifndef __MINGW32__
            Tcl_DeleteFileHandler(irp->d_outFd[0]);
#else
            fprintf(stderr, "TODO: implement Tcl_DeleteFileHandler()\n");
#endif
            irp->d_channelActive = false;
        } else {
            ivi_print("ERROR: IviRemoteProc::InputProc() received error\n");

#ifndef __MINGW32__
            Tcl_DeleteFileHandler(irp->d_outFd[0]);
#else
            fprintf(stderr, "TODO: implement Tcl_DeleteFileHandler()\n");
#endif
            irp->d_channelActive = false;
        }
    } else if (mask & TCL_EXCEPTION) {
        fprintf(stderr, "Exception\n");
    } else {
        fprintf(stderr, "Unknown Tcl Event\n");
    }
#endif /* 0 */
}

/********************************************************************
 * WaitHandler()
 ********************************************************************/
void IviRemoteProc::WaitHandler(ClientData cdata)
{
    IviRemoteProc *irp = (IviRemoteProc *)cdata;

    if (irp->isRunning()) {
        irp->d_waitTimer = Tcl_CreateTimerHandler(150,
            &IviRemoteProc::WaitHandler, irp);
    } else {
        irp->d_waitTimer = 0;
//        fprintf(stderr, "Proc exited...\n");
        irp->d_channelActive = false;
    }
}

/********************************************************************
 * InstCmd()
 ********************************************************************/
int IviRemoteProc::InstCmd(int objc, Tcl_Obj *const objv[])
{

    if (objc < 2) {
        Tcl_AppendResult(d_interp, "too few args", 0);
        return TCL_ERROR;
    }

    const char *cmd = Tcl_GetString(objv[1]);

    if (!strcmp(cmd, "invoke")) {
        int ret = invoke();
        Tcl_SetObjResult(d_interp, Tcl_NewIntObj(ret));
    } else if (!strcmp(cmd, "set_invoke")) {
        setInvoke(objv[2]);
    } else if (!strcmp(cmd, "set_option")) {
        /**** expect at least 3 args ****/
        if (objc < 3) {
            Tcl_AppendResult(d_interp, "too few args - expect at least 3", 0);
            return TCL_ERROR;
        }
        
        const char *opt = Tcl_GetString(objv[2]);

        if (!strcmp(opt, "merge_std")) {
            if (objc < 4) {
                Tcl_AppendResult(d_interp, "too few args - expect "
                        "at least 4", 0);
                return TCL_ERROR;
            }

            const char *val = Tcl_GetString(objv[3]);

            if (!strcmp(val, "true")) {
                d_mergeStd = true;
            } else {
                d_mergeStd = false;
            }
        }
    } else if (!strcmp(cmd, "destroy")) {
        Destroy();
    } else if (!strcmp(cmd, "wait")) {

        /**** Process events while process is running ****/
        while (isRunning()) {
            Tcl_DoOneEvent(0);
        }

        /**** Process events until channel is flushed or we notice
         **** that the process is gone...
         ****/

        Tcl_Flush(d_procOutput);
        Tcl_DoOneEvent(TCL_DONT_WAIT);

        fprintf(stderr, "Done flushing\n");
        fflush(stderr);

#if 0
        while (d_channelActive) {
            Tcl_DoOneEvent(0);
        }
#endif

        Tcl_SetObjResult(d_interp, Tcl_NewIntObj(d_procExitCode));
    } else if (!strcmp(cmd, "get_stdout")) {
        if (!d_procOutput) {
            Tcl_AppendResult(d_interp, "process not running", 0);
            return TCL_ERROR;
        }
        Tcl_SetObjResult(d_interp,
            Tcl_NewStringObj(Tcl_GetChannelName(d_procOutput), -1));
    } else {
        Tcl_AppendResult(d_interp, "unknown process sub-cmd ", cmd, 0);
        return TCL_ERROR;
    }

    return TCL_OK;
}

/********************************************************************
 * invoke()
 *
 * TODO: 
 *  - add flags to specify whether stderr should be captured as well
 ********************************************************************/
#ifndef __MINGW32__
int IviRemoteProc::invoke()
{
    if (pipe(d_outFd) == -1) {
        perror("cannot open pipe");
    }

    if ((d_pid = fork())) {
        /**** Original Process ****/

        d_procOutput = Tcl_MakeFileChannel((ClientData)d_outFd[0],
                TCL_READABLE);

        if (d_procOutput) {
            Tcl_SetChannelOption(d_interp, d_procOutput, "-blocking", "off");
            Tcl_RegisterChannel(d_interp, d_procOutput);
        }

        d_waitTimer = Tcl_CreateTimerHandler(150, 
                &IviRemoteProc::WaitHandler, this);
        d_running = true;
        d_channelActive = true;

        /**** Now, poll the process a couple of times to ensure that
         **** it's actually running (paths are correct, etc...)
         ****/

    } else {
        int ret;

        /**** New Process ****/

        /**** Set all environment variables before calling exec() ****/

        /**** Dup stdout fd ****/
        close(1); dup(d_outFd[1]);

        if (d_mergeStd) {
            close(2); dup(d_outFd[1]);
        }

        /**** Now, call exec... ****/
        ret = execvp(d_procArgs.idx(0), d_procArgs.get_storage());

        perror("exec failed");
        _exit(ret);
    }

    return 0;
}
#else
int IviRemoteProc::invoke()
{
    STARTUPINFO      st_info;
    HANDLE           readEndTmp, readEnd, writeEnd;
    HANDLE           oldStdout, newStdout;
    HANDLE           oldStderr, newStderr;
    HANDLE           thisProc;
    int              error;


    d_procOutput = 0;


    /*** Create a pipe for reading process output ***/
    SECURITY_ATTRIBUTES     sa;

    sa.nLength = sizeof(SECURITY_ATTRIBUTES);
    sa.lpSecurityDescriptor = NULL;
    sa.bInheritHandle = TRUE;

    CreatePipe(&readEndTmp, &writeEnd, &sa, 0);

    

    /*** Now, duplicate the read handle ***/
    if (!DuplicateHandle(GetCurrentProcess(), readEndTmp,
            GetCurrentProcess(), &readEnd, 0, FALSE, /** not inherited **/
            DUPLICATE_SAME_ACCESS)) {
        fprintf(stderr, "DuplicateHandle failed: error=%d\n", GetLastError());
        fflush(stderr);
    }

    /*** Close the tmp (inheritable) end ***/
    CloseHandle(readEndTmp);

    /*** Need to assemble the arguments... ***/
    char      *exec_str;
    int        exec_len = 0;

    for (Uint32 i=0; i<(d_procArgs.length()-1); i++) {
        exec_len += (strlen(d_procArgs.idx(i))+4);
    } 

    exec_str = new char [exec_len];
    exec_str[0] = 0;

    for (Uint32 i=0; i<(d_procArgs.length()-1); i++) {
        strcat(exec_str, d_procArgs.idx(i));
        strcat(exec_str, " ");
    } 

    memset(&st_info, 0, sizeof(st_info));

    st_info.cb = sizeof(st_info);

#if 1
    st_info.dwFlags = STARTF_USESHOWWINDOW | STARTF_USESTDHANDLES;
    st_info.wShowWindow = SW_HIDE;

    st_info.hStdOutput = writeEnd;
    st_info.hStdError  = writeEnd;
#endif

    d_pid = d_procIdx;

    /*** Set an environment variable to represent the proc-id ***/
    char *pid_var = (char *)malloc(strlen("IVI_PROC_ID=XXXXXXXX"+4));
    sprintf(pid_var, "IVI_PROC_ID=%d", d_pid);
    putenv(pid_var);

    error = CreateProcess(NULL, exec_str, NULL, NULL, TRUE,
            CREATE_NEW_CONSOLE, NULL, NULL, &st_info, &d_procInfo);

    ivi_print("exec_str = %s\n", exec_str);

    if (error == 0) {
        DWORD errNum = GetLastError();
        LPVOID   lpMsgBuf;
  
        FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
                      FORMAT_MESSAGE_FROM_SYSTEM |
                      FORMAT_MESSAGE_IGNORE_INSERTS,
                      NULL, errNum, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                      (LPSTR)&lpMsgBuf, 0, NULL);

        Tcl_SetObjResult(d_interp, Tcl_NewStringObj((char *)lpMsgBuf, -1));

        fprintf(stderr, "\terror is: %s\n", (char *)lpMsgBuf);
        fflush(stderr);

        LocalFree(lpMsgBuf);

        CloseHandle(writeEnd);
        CloseHandle(readEnd);

        return TCL_ERROR;
    } else {

        d_running = true;

        d_procOutput = Tcl_MakeFileChannel((ClientData)readEnd, TCL_READABLE);
        Tcl_SetChannelOption(d_interp, d_procOutput, "-blocking", "off");
        Tcl_RegisterChannel(d_interp, d_procOutput);

        /*** Setup a timer to periodically poll for process status ***/
        d_waitTimer = Tcl_CreateTimerHandler(150, 
                &IviRemoteProc::WaitHandler, this);

        return TCL_OK;
    }
}
#endif

/********************************************************************
 * kill()
 ********************************************************************/
int IviRemoteProc::kill()
{
    fprintf(stderr, "IviRemoteProc::kill()\n");
#ifndef __MINGW32__
    ::kill(d_pid, 9);
#else
    fprintf(stderr, "TODO: Implement kill()\n");
#endif
}

/********************************************************************
 * wait()
 ********************************************************************/
int IviRemoteProc::wait(int *status)
{
#ifndef __MINGW32__
    pid_t pid = waitpid(d_pid, status, WNOHANG);

    if (pid > 0) {
        return 1;
    } else {
        return pid;
    }
#else
    fprintf(stderr, "TODO: Implement wait()\n");
    return 0;
#endif
}

/********************************************************************
 * isRunning()
 ********************************************************************/
bool IviRemoteProc::isRunning()
{
    return isRunning(0);
}

/********************************************************************
 * isRunning()
 ********************************************************************/
bool IviRemoteProc::isRunning(int ms_timeout)
{
#ifndef __MINGW32__
    if (d_running) {
        int status;
        int ret = waitpid(d_pid, &status, WNOHANG);

        if (ret > 0) {
            d_running = false;
            d_procExitCode = status;
        }
    }
#else
    DWORD waitResult = WaitForSingleObject(d_procInfo.hProcess, ms_timeout);

    if (waitResult == WAIT_OBJECT_0) {
        fprintf(stderr, "NOTE: Process exited\n");
        d_running = false;
    }
#endif

    return d_running;
}

/********************************************************************
 * findProc()
 ********************************************************************/
IviRemoteProc *IviRemoteProc::findProc(Uint32 pid)
{
    Uint32 i;
    IviRemoteProc *ret = 0;

    for (i=0; i<d_numProcs; i++) {
        if (!d_procList[i]) {
            fprintf(stderr, "ERROR: d_procList[%d] is NULL\n", i);
            continue;
        }
        if (d_procList[i]->d_pid == pid) {
            ret = d_procList[i];
            break;
        }
    }

    return ret;
}

/********************************************************************
 * deleteProc()
 ********************************************************************/
void IviRemoteProc::deleteProc(IviRemoteProc *proc)
{
    Uint32 i, x;

    for (i=0; i<d_numProcs; i++) {
        if (d_procList[i] == proc) {
            for (x=i; x<d_numProcs; x++) {
                d_procList[x] = d_procList[x+1];
            }
            break;
        }
    }

    if (i == d_numProcs) {
        fprintf(stderr, "IVI INTERNAL ERROR: Proc %x not on list\n");
    } else {
        d_numProcs--;
    }
}

/********************************************************************
 * getStdoutChannel()
 ********************************************************************/
const char *IviRemoteProc::getStdoutChannelName()
{
    if (d_procOutput) {
        return Tcl_GetChannelName(d_procOutput);
    } else {
        return 0;
    }
}

IviRemoteProc *IviRemoteProc::d_procList[128];
Uint32         IviRemoteProc::d_numProcs = 0;

/********************************************************************
 * IviRemoteProcCmd()
 ********************************************************************/
static int IviRemoteProcCmd(
        ClientData           clientData,
        Tcl_Interp          *interp,
        int                  argc,
        const char         **argv)
{
    IviRemoteProc *p = new IviRemoteProc(interp, argc, (char **)argv);

    return TCL_OK;
}

/********************************************************************
 * IviRemoteProc_Init()
 ********************************************************************/
extern "C" int IviRemoteProc_Init(Tcl_Interp *interp)
{
    Tcl_CreateCommand(interp, "ivi_proc", IviRemoteProcCmd, 0, 0);

    return TCL_OK;
}


