/*
 *  JLib - Jacob's Library.
 *  Copyright (C) 2003, 2004  Juan Carlos Seijo Prez
 * 
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 * 
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 * 
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 *  Juan Carlos Seijo Prez
 *  jacob@mainreactor.net
 */

/** Clase para manejo de ficheros de texto.
 * @file    JTextFile.h.
 * @author  Juan Carlos Seijo Prez
 * @date    15/06/2003
 * @version 0.0.1 - 15/06/2003 - Primera versin.
 */

#ifndef _JTEXTFILE_INCLUDED
#define _JTEXTFILE_INCLUDED

#include <JLib/Util/JTypes.h>
#include <JLib/Util/JFile.h>
#include <stdarg.h>
#include <ctype.h>
#include <stdlib.h>

/** Mxima longitud de la lnea.
 */
#define MAX_LINE_LENGTH 1024

/** Clase para manejo de ficheros de texto.
 */
class JTextFile : public JFile
{
protected:
  s8 *ptr;                              /**< Puntero de desplazamiento. */
  static s8 line;                       /**< Caracter de fin de lnea. */

public:

  /** Crea el fichero a partir del nombre.
   * @param  name Nombre del fichero.
   */
  JTextFile(const s8 *name = 0);

  /** Libera la memoria y cierra el fichero.
   */
  ~JTextFile();

  /** Avanza hasta el comienzo de la cadena dada. Comienza desde el principio si jump 
   * es true y se alcanza el final del fichero (hasta que se alcanza la posicin de inicio).
   * @param  str Cadena a buscar.
   * @param  jump Indica si debe volver al principio cuando la bsqueda llega al final.
   */
  bool FindNext(const s8 *str, bool jump = false);

  /** Lee el fichero como en la clase base pero asigna el puntero de desplazamiento.
   * @param  readSize Tamao a leer o cero si se debe leer todo el fichero.
   */
  virtual u32 Read(u32 readSize = 0);

  /** Carga todo el fichero como en la clase base pero asigna el puntero de desplazamiento.
   * @param  filename Nombre del fichero a cargar.
   * @param  mode Modo de apertura en la forma {w|r|a}[+][{b|t}].
   * por defecto el modo es "rb" (lectura, binario).
   * @return <b>true</b> si lo carg correctamente, <b>false</b> si no.
   */
  virtual bool Load(const char *filename, const char *mode = "rb");

	/** Salta la siguiente palabra y avanza el puntero
   * @return <b>true</b> si haba siguiente palabra, <b>false</b> si no.
   */
  bool SkipNextWord();

  /** Lee la palabra siguiente y avanza el puntero.
   * @param  str Buffer donde leer la palabra actual.
   * @return <b>true</b> si haba siguiente palabra, <b>false</b> si no.
   */
  bool ReadWord(s8 *str);

  /** Lee la palabra siguiente entrecomillada y avanza el puntero.
   * @param  str Buffer donde leer la palabra entrecomillada actual.
   * @return <b>true</b> si haba siguiente palabra entrecomillada, <b>false</b> si no.
   */
  bool ReadQuotedWord(s8 *str);

  /** Lee el prximo nmero decimal. Salta todos los caracteres que no sean nmeros.
   * @param  f Variable donde almacenar el nmero.
   * @return <b>true</b> si haba siguiente nmero, <b>false</b> si no.
   */
  bool ReadFloat(float *f);

  /** Lee el prximo nmero entero. Salta todos los caracteres que no sean nmeros.
   * @param  i Variable donde almacenar el nmero.
   * @return <b>true</b> si haba siguiente nmero, <b>false</b> si no.
   */
  bool ReadInteger(s32 *i);

  /** Mueve el puntero al comienzo de la siguiente lnea.
   * @return <b>true</b> si haba siguiente lnea, <b>false</b> si no.
   */
  bool NextLine();

  /** Mueve el puntero al comienzo del documento.
   */
  void StartOfDocument();

  /** Lee una lnea del fichero desde la posicin actual.
   * @param  str Variable donde guardar la siguiente lnea.
   * @return <b>true</b> si haba siguiente lnea, <b>false</b> si no.
   */
  bool ReadLine(s8 *str);

  /** Devuelve el puntero a la posicin actual.
   * @return Puntero a la posicin actual.
   */
  s8 * GetPos();

  /** Establece el puntero de posicin actual.
   * @param _ptr Nueva posicin actual.
   * @return <b>true</b> si exista la posicin, <b>false</b> si no.
   */
  bool SetPos(s8 *_ptr);

  /** Escribe en el fichero sin salto de lnea.
   * @param  str Cadena de caracteres a escribir.
   * @return Nmero de caracteres escritos.
   */
  u32 Print(const s8 *str);

  /** Escribe en el fichero con salto de lnea.
   * @param  str Cadena de caracteres a escribir.
   * @return Nmero de caracteres escritos.
   */
  u32 PrintLine(const s8 *str);

  /** Escribe en el fichero con formato y sin salto de lnea
   * @param  fmt Cadena de formato tipo printf.
   * @return Nmero de caracteres escritos.
   */
  u32 Printf(const s8 *fmt, ... );

  /** Cuenta el nmero de cadenas dadas existentes en el fichero.
   * @param  str Cadena de caracteres a buscar.
   * @param  init Posicin de inicio.
   * @param  end Posicin de fin.
   * @return Nmero de coincidencias.
   */
  u32 CountString(const s8 *str, s8* init = 0, s8* end = 0);

	/** Lectura de campos variables tipo scanf.
	 * @param  format Formato a leer de este fichero.
	 * @return Nmero de parmetros ledos correctamente.
	 */
	s32 Scanf(const char *format, ...);
};

#endif  // _JTEXTFILE_INCLUDED
