//
// $Id: statistics.cc,v 1.2 2001/08/29 15:20:47 dredd Exp $
//
// $Source: /cvsroot/hammerhead/hammerhead/src/statistics.cc,v $
// $Revision: 1.2 $
// $Date: 2001/08/29 15:20:47 $
// $State: Exp $
//
// Author: Geoff Wong
//

/*
 * Global counting variables
 * Author: Geoff Wong
 */

#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#ifdef FreeBSD
#include <machine/param.h>
#endif
#include <unistd.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <signal.h>
#include "statistics.h"
#include "config.h"
#include "spinlock.h"

struct stat_data *sdp = NULL;

void lock(void)
{
	s_lock(&(sdp->lock_data));
}

void unlock(void)
{
	s_unlock(&(sdp->lock_data));
}

time_t get_start_time(void)
{
	return sdp->StartTime;
}

void initialise_statistics()
{
	int sdh;

	sdh = shmget(SHM_KEY, sizeof(struct stat_data), 
		IPC_CREAT | SHM_R | SHM_W | 
		((SHM_R | SHM_W) >>3) | ((SHM_R | SHM_W) >>6));

	if (sdh == -1) 
	{
		/* get failed */
		fprintf(stderr, "shget failed - bailing...");
		exit(1);
	}

	sdp = (struct stat_data *)shmat(sdh, 0, 0);

	if (sdp == (struct stat_data *)-1) 
	{
		/* attach failed */
		fprintf(stderr, "shmat failed - bailing...");
		exit(1);
	}
	
	s_lock_init(&(sdp->lock_data));

	lock();
    sdp->Failures = 0;
    sdp->NoVerify = 0;
    sdp->TurnaroundTime = 0;
    sdp->Requests = 0;
    sdp->ResponseTime = 0;
    sdp->Responses = 0;
    sdp->_Scenarios = 0;
    sdp->_Sequences = 0;
    sdp->LastReport = 0;
    sdp->StartTime = time(0);
	sdp->ClaimedLength = 0;
	sdp->ReadLength = 0;
	unlock();
}

void inc_scenarios(unsigned int x)
{
	lock();
    sdp->_Scenarios += x;
	unlock();
}

void inc_sequences(unsigned int x)
{
	lock();
    sdp->_Sequences += x;
	unlock();
}

void inc_noverify(unsigned int x)
{
	lock();
    sdp->NoVerify += x;
	unlock();
}

void inc_failures(unsigned int x)
{
	lock();
    sdp->Failures += x;
	unlock();

	/* have we had too many failures - if so shutdown */	
	if ((sdp->Failures > (unsigned)MaxFailures) && (MaxFailures != 0))
	{
		/* signal my parent, who will signal all my children
	       to finish up and shutdown */
		 kill(getppid(), SIGHUP);
	}
}

void inc_requests(unsigned int x,unsigned  int tm)
{
	lock();
    sdp->Requests += x;
    sdp->TurnaroundTime += tm;
	unlock();
}

void inc_responses(unsigned int x, unsigned int tm)
{
	lock();
    sdp->Responses += x;
    sdp->ResponseTime += tm;
	unlock();
}

void inc_claimed(unsigned int x)
{
	lock();
	sdp->ClaimedLength += x;
	unlock();
}

void inc_read(unsigned int x)
{
	lock();
	sdp->ReadLength += x;
	unlock();
}

void Report(FILE * fp)
{
    if (fp == NULL) return;

    time_t tme = time(0);

    if (sdp->LastReport == 0)
    {
        sdp->LastReport = tme;
        return;
    }

	/* this is a bit dubious if ReportTime is 0  but Scenario::Report()
       is only called from a few tightly controlled places. (hammerhead.cc) */
    if (tme > (sdp->LastReport + ReportTime))
    {
        sdp->LastReport = tme;
        output_statistics(fp);
    }

}

#define MAX_DATE 128
static int in_output = 0;

void output_statistics(FILE * fp)
{
    int avtime, avrtime;
	float avstime, avqtime, avrstime;
	int cltime, rltime;
    time_t ctm, runtime;
    char buf[MAX_DATE];
	list<HammerTarget>::iterator targetit;
	list<String>::iterator aliasit;

	if (in_output) return;
    in_output = 1;

#ifdef FreeBSD
    ctm = time(NULL);
	struct tm tm;
	localtime_r(&ctm, &tm);

#define DAYSPERWEEK 7
#define MONSPERYEAR 12 
#define TM_YEAR_BASE 1900

    static const char       wday_name[DAYSPERWEEK][4] = {
            "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
    };
    static const char       mon_name[MONSPERYEAR][4] = {
            "Jan", "Feb", "Mar", "Apr", "May", "Jun",
            "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
    };
    (void) sprintf(buf, "%.3s %.3s%3d %02d:%02d:%02d %d\n",
            wday_name[tm.tm_wday],
            mon_name[tm.tm_mon],  
            tm.tm_mday, tm.tm_hour,
            tm.tm_min, tm.tm_sec,
            TM_YEAR_BASE + tm.tm_year);


#endif

#ifdef Linux
	ctm = time(NULL);
    ctime_r(&ctm, buf);
#endif

#ifdef SunOS
    ctm = time(0);
    ctime_r(&ctm, buf, MAX_DATE);
#endif

	lock();

    if (sdp->Requests == 0)  sdp->Requests = 1;
    avtime = sdp->TurnaroundTime / sdp->Requests;
    if (sdp->Responses == 0)  sdp->Responses = 1;
    avrtime = sdp->ResponseTime / sdp->Requests;
    if (sdp->Responses == 0)  sdp->Responses = 1;
    avrtime = sdp->ResponseTime / sdp->Requests;
    
    runtime = ctm - sdp->StartTime;
    if (runtime == 0) runtime = 1;
	avrstime = (float)sdp->Responses / (float)runtime;
    avstime = (float)sdp->_Scenarios / (float)runtime;
    avqtime = (float)sdp->_Sequences / (float)runtime;
	cltime = sdp->ClaimedLength / runtime;
	rltime = sdp->ReadLength / runtime;

    fprintf(fp, "\nConfig File                  : %s\n", ConfName.c_str());    
	for (targetit = HammerTargets.begin(); targetit != HammerTargets.end(); targetit++) 
	{
		HammerTarget target = *(targetit);

    	if (targetit == HammerTargets.begin()) 
			fprintf(fp, "Machine                      :");    
    	else 
			fprintf(fp, "                             :");    
    	fprintf(fp, " %s\n", target.HammerIP.c_str());    
	}

	if (IpAliases.size() > 0 )
	{
		for (aliasit = IpAliasPattern.begin(); aliasit != IpAliasPattern.end(); aliasit++)
		{
			String alias = *(aliasit);

    		if (aliasit == IpAliasPattern.begin()) 
				fprintf(fp, "IP aliases                   :");    
    		else 
				fprintf(fp, "                             :");    
    		fprintf(fp, " %s\n", alias.c_str());    
		}
	}

    fprintf(fp, "Time                         : %s", buf);    
    fprintf(fp, "SEED =                       : %d\n", Seed);    
    fprintf(fp, "Parent Process PID =         : %d\n", getpid());    
    fprintf(fp, "Total Run Time (sec)         : %d\n", RunTime);    
    fprintf(fp, "Sessions                     : %d\n", Sessions);    
    fprintf(fp, "Session Sleep Time (msec)    : %d\n", SleepTime/1000);    
    // output the StartLag time.
    // All threads are typically created created in 1 - 2 secs.
    fprintf(fp, "Startup Lag Time (seconds)   : %d\n", (StartLag * Sessions / 1000000)+1);
    fprintf(fp, "Failures                     : %d\n", sdp->Failures);    
    fprintf(fp, "NoVerify                     : %d\n", sdp->NoVerify);    
    fprintf(fp, "Total Requests Served        : %d\n", sdp->Requests);    
    fprintf(fp, "Total Turnaround Time (msec) : %lld\n", sdp->TurnaroundTime);    
    fprintf(fp, "Average Request Time  (msec) : %d\n", avtime);    
    fprintf(fp, "Total Responses              : %d\n", sdp->Responses);    
    fprintf(fp, "Average Responses / sec      : %f\n", avrstime);    
    fprintf(fp, "Total Response Time (msec)   : %lld\n", sdp->ResponseTime);    
    fprintf(fp, "Average Response Time (msec) : %d\n", avrtime);    
    fprintf(fp, "Scenario Throughput          : %d\n", sdp->_Scenarios);    
    fprintf(fp, "Sequence Throughput          : %d\n", sdp->_Sequences);    
    fprintf(fp, "Run Time (sec)               : %ld\n", runtime);    
    fprintf(fp, "Scenarios / sec              : %f\n", avstime);    
    fprintf(fp, "Sequences / sec              : %f\n", avqtime);    
	fprintf(fp, "Content Length bytes         : %d\n", sdp->ClaimedLength);
	fprintf(fp, "Content Length bytes / sec   : %d\n", cltime);
	fprintf(fp, "Read Length bytes            : %d\n", sdp->ReadLength);
	fprintf(fp, "Read Length bytes / sec      : %d\n", rltime);
    fflush(fp);

	unlock();

    in_output = 0;
}


