#ifndef NETBUFFER_H
#define NETBUFFER_H

/*
 * Variable-size, reference-counted memory buffer used to access network
 * packets
 *
 * Copyright (C) 2003,2004 Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include "Buffer.h"

class NetBuffer : public Buffer
{
public:
	unsigned int cursor;
	
public:
	NetBuffer() throw () : Buffer(), cursor(0) {}
	NetBuffer(unsigned int size) throw () : Buffer(size), cursor(0) {}
	NetBuffer(void* buf, unsigned int size, bool own = true) throw ()
		: Buffer(buf, size, own), cursor(0) {}
	NetBuffer(const void* buf, unsigned int size) throw ()
		: Buffer(buf, size), cursor(0) {}

	NetBuffer(const Buffer& buf) throw () : Buffer(buf), cursor(0) {}
	NetBuffer(const NetBuffer& buf) throw ()
		: Buffer(buf), cursor(buf.cursor) {}

	NetBuffer& operator=(const Buffer& buf) throw ()
	{
		Buffer::operator=(buf);
		cursor = 0;
		return *this;
	}

	NetBuffer& operator=(const NetBuffer& buf) throw ()
	{
		Buffer::operator=(buf);
		cursor = buf.cursor;
		return *this;
	}

	template<class T>
	bool fits(unsigned int ofs = 0) const throw ()
	{
		return cursor + ofs + sizeof(T) < size();
	}
	
	template<class T>
	const T* cast(unsigned int ofs = 0) const throw (ConsistencyCheckException)
	{
		if (cursor + ofs + sizeof(T) >= size())
			throw ConsistencyCheckException("Trying to read past the end of the buffer");
		return (const T*)((const char*)data() + cursor + ofs);
	}

	NetBuffer operator+(unsigned int ofs) throw (ConsistencyCheckException)
	{
		NetBuffer res(*this);
		res.skip(ofs);
		return res;
	}
	
	NetBuffer& operator+=(unsigned int ofs) throw (ConsistencyCheckException)
	{
		skip(ofs);
		return *this;
	}
	
	const NetBuffer after(unsigned int ofs) const throw (ConsistencyCheckException)
	{
		NetBuffer res(*this);
		res.skip(ofs);
		return res;
	}
	
	template<class T>
	const NetBuffer after() const throw (ConsistencyCheckException)
	{
		NetBuffer res(*this);
		res.skip(sizeof(T));
		return res;
	}

	template<class T>
	void skip() throw (ConsistencyCheckException)
	{
		skip(sizeof(T));
	}

	void skip(unsigned int t) throw (ConsistencyCheckException)
	{
		if (cursor + t >= size())
			throw ConsistencyCheckException("Trying to skip past the end of the buffer");
		cursor += t;
	}
};

// vim:set ts=4 sw=4:
#endif
