/* This is for emacs: -*-Mode: C++;-*- */
#if !defined(__INC_DATABASE_METADATA_H)
#define __INC_DATABASE_METADATA_H

#include "gql++/result-set.h"

namespace GQL
{

/** Information about the database.
 * This class provides information about the structure and
 * capabilities of a database. */
class DatabaseMetaData : public SigC::Object
{
  public:
    enum ImportedKeyUpdateRule
    {
      ImportedKeyNoAction,
      ImportedKeySetDefault,
      ImportedKeySetNull,
      ImportedKeyCascade,
      ImportedKeyRestrict
    };
    
    enum ImportedKeyDeferrability
    {
      ImportedKeyInitiallyDeferred,
      ImportedKeyInitiallyImmediate,
      ImportedKeyNotDeferrable
    };

    enum ColumnNullability
    {
      ColumnNoNulls, ///< might not allow NULL values
      ColumnNullable, ///< definitely allows NULL values
      ColumnNullableUnknown ///< nullability unknown
    };

    enum TableIndexType
    {
      TableIndexStatistic,
      TableIndexClustered,
      TableIndexHashed,
      TableIndexOther
    };

    /** Get information about tables.
     * Retrieves a description of the tables available in the 
     * \a catalog. Only table descriptions matching \a catalog, 
     * \a schema, \a table_name_pattern and one of \a types are 
     * returned. They are ordered by type, schema and name.
     * The table descriptions have the following columns:
     * <ul>
     *  <li>0. \b TABLE_CAT string => table catalog (may be null)
     *  <li>1. \b TABLE_SCHEM string => table schema (may be null)
     *  <li>2. \b TABLE_NAME String => table name
     *  <li>3. \b TABLE_TYPE String => table type. Typical types 
     *      are "TABLE", "VIEW", "SYSTEM TABLE", "GLOBAL TEMPORARY", 
     *      "LOCAL TEMPORARY", "ALIAS", "SYNONYM".
     *  <li>4. \b REMARKS String => explanatory comment on the table
     *  <li>5. \b TYPE_CAT String => the types catalog (may be null)
     *  <li>6. \b TYPE_SCHEM String => the types schema (may be null)
     *  <li>7. \b TYPE_NAME String => type name (may be null)
     *  <li>8. \b SELF_REFERENCING_COL_NAME String => name of the 
     *      designated "identifier" column of a typed table (may be null)
     *  <li>9. REF_GENERATION String => specifies how values in 
     *      SELF_REFERENCING_COL_NAME are created. Values are 
     *      "SYSTEM", "USER", "DERIVED". (may be null)
     * </ul> */
    virtual ResultSet *get_tables(const std::string& catalog,
                                  const std::string& schema,
                                  const std::string& table_name_pattern,
                                  const std::vector<std::string>& types) = 0;
    
    /* Get information about table columns. 
     * 
     * \return ResultSet of column descriptions, having the following
     *   columns:
     * <ul>
     *   <li>0. \b TABLE_CAT String => table catalog (may be null)
     *   <li>1. \b TABLE_SCHEM String => table schema (may be null)
     *   <li>2. \b TABLE_NAME String => table name
     *   <li>3. \b COLUMN_NAME String => column name
     *   <li>4. \b DATA_TYPE SQLType => data type
     *   <li>5. \b NULLABLE ColumnNullability => is NULL allowed.
     *   <li>6. REMARKS String => comment describing column (may be null)
     *   <li>7. COLUMN_DEF String => default value (may be null)
     *   <li>8. ORDINAL_POSITION int => index of column in table 
     *       (starting at 0)
     *   </ul> */
    virtual ResultSet *get_columns(const std::string& catalog,
                                   const std::string& schema,
                                   const std::string& table_name_pattern,
                                   const std::string& column_name_pattern) = 0;

    /** Get a description of a table's primary key columns.  
     * They are ordered by COLUMN_NAME.
     *
     * <P>Each column description has the following columns:
     *        <ol>
     *        <li>0. \b TABLE_CAT string => table catalog (may be null)
     *        <li>1. \b TABLE_SCHEM string => table schema (may be null)
     *        <li>2. \b TABLE_NAME string => table name
     *        <li>3. \b COLUMN_NAME string => column name
     *        <li>4. \b KEY_SEQ int => sequence number within primary key
     *        <li>5. \b PK_NAME string => primary key name (may be null)
     *        </li>
     *
     * \param catalog Catalog name. "" retrieves those without a catalog.
     * \param schema a Schema name pattern; "" retrieves those
     *   without a schema.
     * \param table Table name
     * \return ResultSet of primary key column descriptions. */
    virtual ResultSet *get_primary_keys(const std::string& catalog,
                                        const std::string& schema,
                                        const std::string& table) = 0;

    /** Retrieves a description of the foreign key columns in the given
     * foreign key table that reference the primary key columns of the
     * given primary key table (describe how one table imports
     * another's key).
     *
     * \return A result set with the following columns:
     *   <ul>
     *     <li>0: Primary key table catalog (may be \c NULL)</li>
     *     <li>1: Primary key table schema (may be \c NULL)</li>
     *     <li>2: Primary key table name</li>
     *     <li>3: primary key column name</li>
     *     <li>4: foreign key table catalog (may be null) being exported (may be null)</li>
     *     <li>5: foreign key table schema (may be null) being exported (may be null)</li>
     *     <li>6: Foreign key table name</li>
     *     <li>7: foreign key column name being exported</li>
     *     <li>8: sequence number within foreign key</li>
     *     <li>9: What happens to foreign key when primary is updated:</li>
     *     <li>10: What happens to the foreign key when primary is deleted.</li>
     *     <li>11: foreign key name (may be null)</li>
     *     <li>12: primary key name (may be null)</li>
     *     <li>13: can the evaluation of foreign key constraints be deferred until commit</li>
     *   </ul> */
    virtual ResultSet *get_cross_reference(
            const std::string& primary_catalog,
            const std::string& primary_schema,
            const std::string& primary_table,
            const std::string& foreign_catalog,
            const std::string& foreign_schema,
            const std::string& foreign_table) = 0;

    virtual ResultSet *get_imported_keys(const std::string& catalog,
                                         const std::string& schema,
                                         const std::string& table) = 0;
    
    virtual ResultSet *get_exported_keys(const std::string& catalog,
                                         const std::string& schema,
                                         const std::string& table) = 0;

    /** Get a description of a table's indices and statistics. They are
     * ordered by NON_UNIQUE, TYPE, INDEX_NAME, and ORDINAL_POSITION.
     *
     * Each index column description has the following columns:
     *	<ul>
     *	<li>0: \b TABLE_CAT String => table catalog (may be null)
     *	<li>1: \b TABLE_SCHEM String => table schema (may be null)
     *	<li>2: \b TABLE_NAME String => table name
     *	<li>3: \b NON_UNIQUE boolean => Can index values be non-unique?
     *		false when TYPE is tableIndexStatistic
     *	<li>4: \b INDEX_QUALIFIER String => index catalog (may be null);
     *		null when TYPE is tableIndexStatistic
     *	<li>5: \b INDEX_NAME String => index name; null when TYPE is
     *		tableIndexStatistic
     *	<li>6: \b TYPE short => index type:
     *		<UL>
     *		<li> tableIndexStatistic - this identifies table statistics that are
     *			 returned in conjuction with a table's index descriptions
     *		<li> tableIndexClustered - this is a clustered index
     *		<li> tableIndexHashed - this is a hashed index
     *		<li> tableIndexOther - this is some other style of index
     *		</UL>
     *	<li>7: \b ORDINAL_POSITION short => column sequence number
     *		within index; zero when TYPE is tableIndexStatistic
     *	<li>8: \b COLUMN_NAME String => column name; null when TYPE is
     *		tableIndexStatistic
     *	<li>\b ASC_OR_DESC String => column sort sequence, "A" => ascending
     *		"D" => descending, may be null if sort sequence is not supported;
     *		null when TYPE is tableIndexStatistic
     *	<li>\b CARDINALITY int => When TYPE is tableIndexStatisic then
     *		this is the number of rows in the table; otherwise it is the
     *		number of unique values in the index.
     *	<li>\b PAGES int => When TYPE is  tableIndexStatisic then
     *		this is the number of pages used for the table, otherwise it
     *		is the number of pages used for the current index.
     *	<li>\b FILTER_CONDITION String => Filter condition, if any.
     *		(may be null)
     *	</ul>
     *
     * \param catalog a catalog name; "" retrieves those without a catalog
     * \param schema a schema name pattern; "" retrieves those without a schema
     * \param table a table name
     * \param unique when true, return only indices for unique values;
     *	   when false, return indices regardless of whether unique or not
     * \param approximate when true, result is allowed to reflect approximate
     *	   or out of data values; when false, results are requested to be
     *	   accurate
     * \return ResultSet each row is an index column description
     */
    virtual ResultSet *get_index_info(const std::string& catalog,
                                      const std::string& schema,
                                      const std::string& table,
                                      bool unique,
                                      bool approximate) = 0;
};

}

#endif
