# GNU Enterprise Common Library - i18n support
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: i18n.py 6976 2005-02-08 12:54:14Z reinhard $
"""
Internationalization support
"""
from types import *

import exceptions
import gettext
import locale
import os.path
import string
import sys

from gnue import paths

# -----------------------------------------------------------------------------
# Global variables
# -----------------------------------------------------------------------------

__modules = {}                          # Modules by filename
__catalogs = {}                         # Message catalogs by domain
__userlanguage = None

language = None
encoding = None
enc_policy = "replace"                  # policy to use if an unicode character


# -----------------------------------------------------------------------------
# Find a module from filename
# -----------------------------------------------------------------------------

def __find_module (filename):
  if __modules.has_key (filename):
    return __modules [filename]
  for mod in sys.modules.values ():
    if hasattr (mod, '__file__'):
      # mod.__file__ can be .pyc if the module is already compiled
      f = mod.__file__
      if f [-1:] == 'c':
        f = f [:-1]
      if os.path.normcase (f) == os.path.normcase (filename):
        __modules [filename] = mod
        return mod

# -----------------------------------------------------------------------------
# Find the correct translation catalog
# -----------------------------------------------------------------------------

def __find_catalog ():

  # find out the filename of the calling function
  caller_file = (sys._getframe (2)).f_code.co_filename

  # find out the module name
  caller_module = __find_module (caller_file)

  if caller_module is None:
    return None

  # make 'gnue-common' from 'gnue.common.foo.bar'
  x = string.replace (caller_module.__name__, '.', '-', 1)
  i = string.find (x, '.')
  domain = x [:i]

  if __catalogs.has_key (domain):
    return __catalogs [domain]
  else:
    try:
      if os.name == 'posix':
        catalog = gettext.translation (domain, paths.data + '/share/locale')
      else:
        catalog = gettext.translation (domain, paths.data + '/share/locale',
                                       [language])
    except:
      catalog = None

    __catalogs [domain] = catalog

    return catalog

# -----------------------------------------------------------------------------
# Translate a message and return unicode
# -----------------------------------------------------------------------------

def utranslate (message):
  """
  Translates a message and returns a unicode string.  This function is
  available as the builtin function "u_()".
  """
  catalog = __find_catalog ()

  if catalog is None:
    return message

  return catalog.ugettext (message)

# -----------------------------------------------------------------------------
# Translate a message and return local encoding
# -----------------------------------------------------------------------------

def translate (message):
  """
  Translates a message and returns an 8 bit string, encoded with @encoding (the
  current locale's encoding).  This function is available as the builtin
  function "_()".
  """
  catalog = __find_catalog ()

  if catalog is None:
    return message

  return (catalog.ugettext (message)).encode (encoding, enc_policy)

# -----------------------------------------------------------------------------
# Convert Unicode to String, let everything else untouched. This is o().
# -----------------------------------------------------------------------------

def outconv (message):
  """
  Encodes a message to @encoding (the current locale's encoding).  This
  function is available as the builtin function "o()".
  """
  if isinstance (message, UnicodeType):
    return message.encode (encoding, enc_policy)
  else:
    return message


# -----------------------------------------------------------------------------
# Get the current encoding
# -----------------------------------------------------------------------------

def getencoding ():
  """
  This function returns the encoding of the currently active locale
  @returns: encoding of the current locale
  """
  # For Windows, getlocale () is broken - it returns e.g. ('Hungarian_Hungary',
  # '1250') instead of ('hu_HU', 'cp1250'). So we have to use getdefaultlocale.
  # However, this means that under Windows, no locales but the default locale
  # can be used.
  if sys.platform == 'win32':
    return locale.getdefaultlocale () [1] or 'ascii'
  else:
    return locale.getlocale () [1] or 'ascii'


# -----------------------------------------------------------------------------
# Get the current language
# -----------------------------------------------------------------------------

def getlanguage ():
  """
  This function return the language of the currently acitve locale
  @returns: language of the current locale
  """
  return __userlanguage or 'C'


# ---------------------------------------------------------------------------
# Get the locale string of the current user
# ---------------------------------------------------------------------------

def getuserlocale ():
  """
  This function tries to find out which locale the user is using.
  @returns: localestring of the user's locale, i.e. de_AT@euro
  """

  # *Actually* the following is very much what locale.getdefaultlocale should
  # do anyway.  However, that function is broken for $LANGUAGE containing a
  # list of locales, separated with ":". So we have to manually rebuild it...
  items = ['LANGUAGE', 'LC_ALL', 'LC_MESSAGES', 'LANG']
  for key in items:
    if os.environ.has_key (key):
      # Some systems (most notably Debian...) set $LANGUAGE to a *list* of
      # locales, separated by a ":".
      envLang = (os.environ [key]).split (':') [0]
      return locale.locale_alias.get (envLang, envLang)

  # Now this should only happen on Windows, where getdefaultlocale seems to
  # work ok.
  try:
    result = locale.getdefaultlocale () [0]
  except locale.Error:
    result = ''

  return result


# ---------------------------------------------------------------------------
# Change the current locale
# ---------------------------------------------------------------------------

def setcurrentlocale (newLocale):
  """
  This function changes the current locale. If it fails it tries to succeed
  using a more general form of the requested locale, i.e. if 'de_AT@euro'
  fails, 'de_AT' will be tried next. If that fails too 'de' will be tried.
  @param newLocale: string of the locale to be set, e.g. de_AT.ISO88591@euro
      (full blown) or 'de_AT' or 'en_AU'
  """
  if newLocale is None:
    newLocale = ''

  parts  = []
  add    = []
  normal = locale.normalize (newLocale)
  next   = normal.split ('@') [0]
  __userlanguage = next.split ('.') [0]

  # Setting a locale different than the current locale doesn't work on Windows.
  if sys.platform == 'win32':
    return

  alias  = locale.locale_alias.get (__userlanguage.split ('_') [0].lower ())
  if alias:
    add = [alias.split ('@') [0]]
    add.append (add [-1].split ('.') [0])
    add.append (locale.locale_alias.get (add [-1].split ('_') [0].lower ()))

  for item in [normal, next, __userlanguage] + add:
    if item is not None and item not in parts:
      parts.append (item)

  for item in parts:
    try:
      locale.setlocale (locale.LC_ALL, item)

    except locale.Error:
      pass

    else:
      break


# -----------------------------------------------------------------------------
# Module initialization
# -----------------------------------------------------------------------------

# Initialize locale. This has been reported to fail on Mac.
try:
  locale.setlocale (locale.LC_ALL, '')
except:
  pass

# On win32, getlocale is broken - it returns Hungarian_Hungary instead of
# hu_HU. On the other hand, getdefaultlocale is broken when $LANGUAGE contains
# a ":" separated list of locales, like it does for Debian...
if sys.platform == 'win32':
  (language, encoding) = locale.getdefaultlocale ()
else:
  (language, encoding) = locale.getlocale ()

# Make sure encoding is not None
if not encoding:
  encoding = 'ascii'

# Now define the new builtin stuff
import __builtin__
__builtin__.__dict__['u_'] = utranslate
__builtin__.__dict__['_'] = translate
__builtin__.__dict__['o'] = outconv
