/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: gnoise.c,v 1.19 2002/01/13 02:51:16 dengen Exp $
 *
 */


#include <fcntl.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <pthread.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include "display_cache.h"
#include "edit.h"
#include "gnoise.h"
#include "gtkwavedisplay.h"
#include "gtkwaveset.h"
#include "oss.h"
#include "snd_buf.h"

static cmd_t		cmd[MAX_CMD_QUEUE+1];	/* ring buffer of cmds for gui to do */
static gint		cmd_next = 0;		/* next cmd in ring to be done */
static gint		cmd_last = 0;		/* where to write next cmd to be done */
static gboolean		cmd_q_empty = TRUE;	/* to tell the difference between queue empty and full */
static pthread_mutex_t	cmd_q_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_t	load_thread;
static pthread_t	play_thread;
static dcache_t		disp_cache;
static snd_buf_t	snd_buffer;
static GtkWidget	*win_modal_ask	= NULL;
static GtkWidget	*win_main	= NULL;
static GtkWidget	*win_properties	= NULL;


/* globals... */
GtkWaveSet		*ws;
gboolean		progress_cancel	= FALSE;
loop_t			playing_looped	= LOOP_NONE;
gboolean		playing		= FALSE;
gboolean		playing_stop	= FALSE;
gboolean		recording	= FALSE;
gboolean		recording_stop	= FALSE;
gboolean		file_open	= FALSE;
gboolean		scrolling	= FALSE;
GtkWidget		*win_levels	= NULL;
pthread_t		edit_thread;
pthread_t		record_thread;
guint8			lpeak;
guint8			rpeak;
gboolean		lclip;
gboolean		rclip;
prefs_t			prefs;


/* local module prototypes */
static int	idle_func(gpointer data);
static void	load_file(void);
static gboolean modal_ask(gboolean can_cancel, const char *title,
			  const char *ok, const char *cancel,
			  const char *fmt,...);
static void	reap_play_thread(void);
static void	reap_record_thread(void);
static void	window_title(const char *fmt,...);
static gboolean warn_undo_loss(void);
static void	get_opendir(char *path);
static void	put_opendir(const char *path);

#ifdef ENABLE_LADSPA
extern void ladspa_enumerate_menu(GtkWidget *menu);
#endif


int main (int argc, char *argv[])
{
    gchar	*filename;
    gchar	*cwd;
    gint	idle_tag;
    Display	*Xdisplay;
#   ifdef ENABLE_LADSPA
    GtkWidget  *ladspa_menu;
#   endif

#   ifdef ENABLE_GNOME
    gnome_init ("gnoise", VERSION, argc, argv);
#   else
    gtk_set_locale();
    gtk_init(&argc, &argv);
#   endif

    /* make ~/.gnoise if it doesn't exist */
    filename = g_strconcat(g_get_home_dir(), "/.gnoise", NULL);
    mkdir(filename, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
    g_free(filename);

    /* parse rc file */
    filename = g_strconcat(g_get_home_dir(), "/.gnoise/gtkrc", NULL);
    gtk_rc_init();
    gtk_rc_parse(filename);
    g_free(filename);

    prefs_load();

    win_main = create_MainWindow();
    gtk_signal_connect(GTK_OBJECT(win_main), "destroy", GTK_SIGNAL_FUNC(on_quit), NULL);
    gtk_widget_show(win_main);
    window_title("%s v%s", PACKAGE, VERSION);
    idle_tag = gtk_timeout_add(10, idle_func, NULL);
    queue_cmd(CMD_WIDGET_HIDE, "Progressbar");
    queue_cmd(CMD_WIDGET_HIDE, "Cancel");
    queue_cmd(CMD_WIDGET_HIDE, "Timebar");
    if (!prefs.tb_markers)
	queue_cmd(CMD_WIDGET_HIDE, "MarkersTB");
    if (!prefs.tb_transport)
	queue_cmd(CMD_WIDGET_HIDE, "TransportTB");

    status("Welcome to gnoise");

    /* check if backing store is supported by this X server.
     * XFree86 4.x by default doesn't support backing stores.
     *
     * If the X server doesn't support it, we'll instead have exposures
     * generated so we can redraw destination areas that wern't copied
     * from the obscured XCopyArea's region (see gtk_wave_display_expose &
     * gtk_wave_display_scroll).
     */

    Xdisplay = GDK_WINDOW_XDISPLAY(win_main->window);
    if (DoesBackingStore(DefaultScreenOfDisplay(Xdisplay)) == NotUseful)
    {
	modal_ask(FALSE, "Warning", "Ok", NULL,
		"Since your X server doesn't support backing store, smooth\n"
		"scrolling will be slower. If you are using XFree86 4.x you\n"
		"can turn on backing store by starting the server with the\n"
		"+bs argument or put the line: Option \"backingstore\" in\n"
		"the device section of your XF86Config-4 file.");
    }

    /* attach ladspa plugins */
#   ifdef ENABLE_LADSPA
#   ifdef ENABLE_GNOME
    ladspa_menu = GTK_WIDGET(lookup_widget(win_main, "About_LADSPA"))->parent;

#   else
    ladspa_menu = GTK_WIDGET(lookup_widget(win_main, "Plugins_menu"));
#   endif
    ladspa_enumerate_menu(ladspa_menu);
#   endif

    memset(&snd_buffer, 0, sizeof(snd_buffer));
    memset(&disp_cache, 0, sizeof(disp_cache));
    snd_buffer.dcache = &disp_cache;
    ws = GTK_WAVE_SET(lookup_widget(win_main, "WaveSet"));

    /* the selection and markers dialogs need to know which wave set they're
     * operating on so override the signal handler for the menu item that
     * calls them to make sure it gets passed the waveset
     */
    gtk_signal_connect(GTK_OBJECT(lookup_widget(win_main, "menuitem_selection")),
		       "activate", GTK_SIGNAL_FUNC (on_dlgselection), ws);
    gtk_signal_connect(GTK_OBJECT(lookup_widget(win_main, "menuitem_markers")),
		       "activate", GTK_SIGNAL_FUNC (on_dlgmarkers), ws);
                                                
    
    /* open file specified on cmd line */
    if (argc > 1)
    {
	if (argv[1][0] == '/')
	{
	    strcpy(snd_buffer.file, argv[1]);
	}
	else
	{
	    cwd = g_get_current_dir();
	    filename = g_strconcat(cwd, "/", argv[1], NULL);
	    strcpy(snd_buffer.file, filename);
	    g_free(cwd);
	    g_free(filename);
	}
	load_file();
    } else {
	char path[PATH_MAX+1];
	
	get_opendir(path);
	chdir(path);
    }


    gtk_main();
    gtk_timeout_remove(idle_tag);
    return 0;
}



void log(const char *head, const char *fmt,...)
{
#   ifdef ENABLE_LOG
    va_list v_lst;

    va_start(v_lst, fmt);
    fprintf(stderr, "%-15.15s: ", head);
    vfprintf(stderr, fmt, v_lst);
    va_end(v_lst);
#   endif
}



void sample_fmt(smpl_fmt_enum_t fmt, smpl_indx sample, snd_info_t *si,
		char *buf, int bufsize)
{
    if (sample == -1)
    {
	buf[0] = '\0';
	return;
    }

    if (fmt == SMPL_FMT_PREFS)
	fmt = prefs.display_fmt;

    switch(fmt)
    {
	case SMPL_FMT_SAMPLES:
	    snprintf(buf, bufsize, "%d", sample);
	    break;
	case SMPL_FMT_BYTES:
	    snprintf(buf, bufsize, "%d", sample * si->sample_bits/8 * si->channels);
	    break;
	case SMPL_FMT_SMART_SIZE:
	    {
		gfloat bytes = sample * si->sample_bits/8 * si->channels;
		if (bytes > 1<<30)
		    snprintf(buf, bufsize, "%3.3f GB", bytes / (1024 * 1024 * 1024));
		else if (bytes > 1<<20)
		    snprintf(buf, bufsize, "%3.3f MB", bytes / (1024 * 1024));
		else if (bytes > 1<<10)
		    snprintf(buf, bufsize, "%3.3f KB", bytes / 1024);
		else
		    snprintf(buf, bufsize, "%d", (int)bytes);
	    }
	    break;
	case SMPL_FMT_CDSECTORS:
	    {
		guint bytes = sample * si->sample_bits/8 * si->channels;
		guint sectors = bytes / 2352;
		guint remain = (((bytes % 2352) / 2352.0) * 1000.0);
		snprintf(buf, bufsize, "%d.%4.4d", sectors, remain);
	    }
	    break;
	case SMPL_FMT_HHMMSS_SSS:
	case SMPL_FMT_HHMMSS:
	    {
		gfloat seconds;
		guint32 hours;
		guint32 minutes;

		seconds  = (gfloat)sample / (gfloat)si->sample_rate;
		hours    = seconds/3600.0;
		seconds -= hours*3600;
		minutes  = seconds/60.0;
		seconds -= minutes*60;

		if (fmt == SMPL_FMT_HHMMSS_SSS)
		{
		    gfloat thousandths = (seconds - ((gfloat)((int)seconds))) * 1000;
		    snprintf(buf, bufsize, "%2.2d:%2.2d:%2.2d.%3.3d", hours, minutes, (int)seconds, (int)thousandths);
		}
		else
		    snprintf(buf, bufsize, "%2.2d:%2.2d:%2.2d", hours, minutes, (int)seconds);
	    }
	    break;
	case SMPL_FMT_PREFS:
	default:
	    log("ERROR", "Unknown fmt %d in sample_fmt\n", fmt);
	    break;
    }
}


void set_time(smpl_indx time)
{
    static char old_play_str[30] = "";
    static char new_play_str[30] = "";
    GtkWidget *st_bar;

    if (!(win_main && (st_bar = lookup_widget(win_main, "Timebar"))))
	return;

    sample_fmt(SMPL_FMT_PREFS, time,
	&ws->sb->info, new_play_str, sizeof(new_play_str));

    /* try to prevent flicker by drawing string only if it's changed */
    if (strcmp(old_play_str, new_play_str))
    {
	gtk_statusbar_pop(GTK_STATUSBAR(st_bar), 1);
	gtk_statusbar_push(GTK_STATUSBAR(st_bar), 1, new_play_str);
	strcpy(old_play_str, new_play_str);
    }
}

void status(const char *fmt,...)
{
    static gchar status_text[256];
    va_list v_lst;

    va_start(v_lst, fmt);
    vsnprintf(status_text, sizeof(status_text)-1, fmt, v_lst);
    va_end(v_lst);
    queue_cmd(CMD_STATUS_UPDATE, &status_text);
}



static void window_title(const char *fmt,...)
{
    gchar title_text[256];
    va_list v_lst;

    if (!win_main)
	return;

    va_start(v_lst, fmt);
    vsnprintf(title_text, sizeof(title_text)-1, fmt, v_lst);
    va_end(v_lst);
    gtk_window_set_title(GTK_WINDOW(win_main), title_text);
    gtk_widget_queue_draw(win_main);
}



void put_opendir(const char *path)
{
    int		fd;
    gchar	*name;

    if (prefs.remember_open_dir)
    {
	name = g_strconcat(g_get_home_dir(), "/.gnoise/opendir", NULL);
	fd = open(name, O_RDWR | O_CREAT | O_TRUNC, 0644);
	if (fd >= 0)
	{
	    write(fd, path, strlen(path));
	    close(fd);
	}
	g_free(name);
    }
}



void get_opendir(char *path)
{
    int		fd;
    gchar	*name;

    name = g_strconcat(g_get_home_dir(), "/.gnoise/opendir", NULL);
    fd = open(name, O_RDONLY | O_NONBLOCK);
    if (fd >= 0)
    {
	int rc;

	rc = read(fd, path, PATH_MAX);
	path[rc] = '\0';
	close(fd);
    }
    else
    {
	strcpy(path, g_get_home_dir());
    }
    g_free(name);
}



static void on_modal_ask_ok(GtkButton *button, gpointer user_data)
{
    *(gboolean *)user_data = TRUE;
    gtk_widget_destroy(win_modal_ask);
    gtk_main_quit();
}

static void on_modal_ask_cancel(GtkButton *button, gpointer user_data)
{
    *(gboolean *)user_data = FALSE;
    gtk_widget_destroy(win_modal_ask);
    gtk_main_quit();
}

static gboolean on_modal_ask_destroy(GtkWidget *widget, GdkEvent *event,
			      gpointer user_data)
{
    *(gboolean *)user_data = FALSE;
    gtk_main_quit();
    return FALSE;
}

static gboolean modal_ask(gboolean can_cancel, const char *title,
			  const char *ok, const char *cancel,
			  const char *fmt,...)
{
    GtkWidget	*MDLGVBox;
    GtkWidget	*MDLGLabel;
    GtkWidget	*separator;
    GtkWidget	*HButtonBox;
    GtkWidget	*Ok;
    GtkWidget	*Cancel;
    char	msg_buf[512];
    va_list	v_lst;
    gboolean	rc;

    win_modal_ask = gtk_window_new (GTK_WINDOW_DIALOG);
    gtk_window_set_title (GTK_WINDOW (win_modal_ask), title);
    gtk_window_set_modal (GTK_WINDOW (win_modal_ask), TRUE);
    gtk_window_set_policy (GTK_WINDOW (win_modal_ask), FALSE, TRUE, TRUE);

    MDLGVBox = gtk_vbox_new (FALSE, 0);
    gtk_widget_show (MDLGVBox);
    gtk_container_add (GTK_CONTAINER (win_modal_ask), MDLGVBox);
    gtk_container_set_border_width (GTK_CONTAINER (MDLGVBox), 2);

    va_start(v_lst, fmt);
    vsnprintf(msg_buf, sizeof(msg_buf)-1, fmt, v_lst);
    va_end(v_lst);

    MDLGLabel = gtk_label_new (msg_buf);
    gtk_widget_show (MDLGLabel);
    gtk_box_pack_start (GTK_BOX (MDLGVBox), MDLGLabel, TRUE, TRUE, 0);
    gtk_label_set_justify (GTK_LABEL (MDLGLabel), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment (GTK_MISC (MDLGLabel), 0.5, 0.10);

    separator = gtk_hseparator_new ();
    gtk_widget_show (separator);
    gtk_box_pack_start (GTK_BOX (MDLGVBox), separator, FALSE, FALSE, 4);

    HButtonBox = gtk_hbutton_box_new ();
    gtk_widget_show (HButtonBox);
    gtk_box_pack_end (GTK_BOX (MDLGVBox), HButtonBox, FALSE, FALSE, 0);
    gtk_button_box_set_layout (GTK_BUTTON_BOX (HButtonBox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing (GTK_BUTTON_BOX (HButtonBox), 5);

    Ok = gtk_button_new_with_label (ok);
    gtk_widget_show (Ok);
    gtk_container_add (GTK_CONTAINER (HButtonBox), Ok);
    GTK_WIDGET_SET_FLAGS (Ok, GTK_CAN_DEFAULT);

    if (can_cancel)
    {
	Cancel = gtk_button_new_with_label (cancel);
	gtk_widget_show (Cancel);
	gtk_container_add (GTK_CONTAINER (HButtonBox), Cancel);
	GTK_WIDGET_SET_FLAGS (Cancel, GTK_CAN_DEFAULT);

	gtk_signal_connect (GTK_OBJECT (Cancel), "clicked",
			    GTK_SIGNAL_FUNC (on_modal_ask_cancel),
			    &rc);
    }

    gtk_signal_connect (GTK_OBJECT (win_modal_ask), "destroy_event",
                        GTK_SIGNAL_FUNC (on_modal_ask_destroy),
                        &rc);
    gtk_signal_connect (GTK_OBJECT (win_modal_ask), "delete_event",
                        GTK_SIGNAL_FUNC (on_modal_ask_destroy),
                        &rc);
    gtk_signal_connect (GTK_OBJECT (Ok), "clicked",
                        GTK_SIGNAL_FUNC (on_modal_ask_ok),
                        &rc);

    gtk_widget_grab_default (Ok);
    gtk_widget_show(win_modal_ask);
    gtk_main();

    return rc;
}



void queue_cmd(cmd_enum_t cmd_arg, void *data)
{
    pthread_mutex_lock(&cmd_q_mutex);
    if ((cmd_next == cmd_last) && !cmd_q_empty)
    {
	int cmdnr;
	
	log("CMD_Q", "cmd queue full!, unable to queue cmd %d next %d last %d\n",
	    cmd_arg, cmd_next, cmd_last);
	log("CMD_Q", "    queue dump follows\n");
	for(cmdnr = 0; cmdnr < MAX_CMD_QUEUE; cmdnr++)
	{
	    log("CMD_Q", "    %3.3d %d %p\n", cmdnr, cmd[cmdnr].cmd, cmd[cmdnr].data);
	}
    }

    cmd[cmd_last].cmd  = cmd_arg;
    cmd[cmd_last].data = data;
    cmd_last = (cmd_last != MAX_CMD_QUEUE) ? cmd_last+1 : 0;
    cmd_q_empty = FALSE;
    pthread_mutex_unlock(&cmd_q_mutex);
}



static int idle_func(gpointer data)
{
    GtkProgress		*pgrs;
    GtkWidget		*widget;
    gfloat		 progress;
    gchar		 buf[10];
    gboolean		 thread_return;
    selection_t		*sel;

    while(cmd_next != cmd_last)
    {
	switch(cmd[cmd_next].cmd)
	{
	case CMD_WIDGET_SHOW:
	    if (win_main)
	    {
		widget = GTK_WIDGET(lookup_widget(win_main, cmd[cmd_next].data));
		gtk_widget_show(widget);
	    }
	    break;

	case CMD_WIDGET_HIDE:
	    if (win_main)
	    {
		widget = GTK_WIDGET(lookup_widget(win_main, cmd[cmd_next].data));
		gtk_widget_hide(widget);
	    }
	    break;

#if 0
	case CMD_PROGRESS_CANCEL_SHOW:
	    if (win_main)
	    {
		widget = GTK_WIDGET(lookup_widget(win_main, "Cancel"));
		gtk_widget_show(widget);
	    }
	    break;

	case CMD_PROGRESS_CANCEL_HIDE:
	    if (win_main)
	    {
		widget = GTK_WIDGET(lookup_widget(win_main, "Cancel"));
		gtk_widget_hide(widget);
	    }
	    break;
	    
	case CMD_PROGRESS_BAR_SHOW:
	    if (win_main)
	    {
		pgrs = GTK_PROGRESS(lookup_widget(win_main, "Progressbar"));
		gtk_progress_set_show_text(pgrs, TRUE);
		gtk_widget_show(GTK_WIDGET(pgrs));
	    }
	    break;

	case CMD_PROGRESS_BAR_HIDE:
	    if (win_main)
	    {
		pgrs = GTK_PROGRESS(lookup_widget(win_main, "Progressbar"));
		gtk_progress_set_show_text(pgrs, FALSE);
		gtk_widget_hide(GTK_WIDGET(pgrs));
	    }
	    break;

	case CMD_TIME_SHOW:
	    if (win_main)
	    {
		widget = GTK_WIDGET(lookup_widget(win_main, "Timebar"));
		gtk_widget_show(GTK_WIDGET(widget));
	    }
	    break;

	case CMD_TIME_HIDE:
	    if (win_main)
	    {
		widget = GTK_WIDGET(lookup_widget(win_main, "Timebar"));
		gtk_widget_hide(GTK_WIDGET(widget));
	    }
	    break;
#endif

	case CMD_PROGRESS_BAR_UPDATE:
	    if (win_main)
	    {
		pgrs = GTK_PROGRESS(lookup_widget(win_main, "Progressbar"));
		progress = *(gfloat *)cmd[cmd_next].data;
		gtk_progress_set_value(pgrs, progress);
	    }
	    break;


	case CMD_SELECTION_DIALOG_UPDATE:
	    dlgselection_update(ws);
	    break;

	case CMD_ADJUST_ZOOM:
	    if (win_main)
	    {
		sprintf(buf, "1:%d", 1 << ws->zoom);
		gtk_label_set_text(GTK_LABEL(lookup_widget(win_main,
						"ZoomLevel")), buf);
	    }
	    break;

	case CMD_REAP_PLAY_THREAD:
	    reap_play_thread();
	    break;


	case CMD_REAP_RECORD_THREAD:
	    reap_record_thread();
	    break;

	case CMD_REAP_LOAD_THREAD:
	    /*
	     * if loading succeded, create wave display's and tell them
	     * to draw.
	     */

	    pthread_join(load_thread, (void *)&thread_return);
	    if (thread_return)
	    {
		gchar *cd_dir;

		/* cd to dir file is in */
		cd_dir = g_dirname(snd_buffer.file);
		chdir(cd_dir);
		put_opendir(cd_dir);
		g_free(cd_dir);
		

		/* could create instead... */

		gtk_wave_set_snd_attach(ws, &snd_buffer);

		file_open = TRUE;
		status("Editing %s", g_basename(ws->sb->file));
		window_title("%s v%s - %s", PACKAGE, VERSION, g_basename(ws->sb->file));
		queue_cmd(CMD_ADJUST_ZOOM, NULL);
		if (prefs.show_time)
		{
		    set_time(0);
		    queue_cmd(CMD_WIDGET_SHOW, "Timebar");
		}
	    }
	    else
		status("Open canceled");
	    break;


	case CMD_REAP_EDIT_THREAD:
	    pthread_join(edit_thread, (void *)&thread_return);
	    if (thread_return)
		status("Edit completed");
	    else
		status("Edit failed");
	    break;

	case CMD_STATUS_UPDATE:
	    if (win_main)
	    {
		widget = lookup_widget(win_main, "Statusbar");
		gtk_statusbar_pop(GTK_STATUSBAR(widget) , 1);
		gtk_statusbar_push(GTK_STATUSBAR(widget), 1, cmd[cmd_next].data);
	    }
	    break;

	case CMD_DISPLAY_UPDATE:
	    sel = (selection_t *)cmd[cmd_next].data;
	    gtk_wave_set_redraw_area(ws, sel->start, sel->end, CHANNEL_ALL);
	    break;

	default:
	    log("CMD_Q ERROR", "Unknown cmd %d\n", cmd[cmd_next].cmd);
	    break;
	}


	cmd_next = (cmd_next != MAX_CMD_QUEUE) ? cmd_next+1 : 0;
    }
    
    pthread_mutex_lock(&cmd_q_mutex);
    cmd_q_empty = TRUE;
    pthread_mutex_unlock(&cmd_q_mutex);

    if (playing)
    {
	dlglevels_update();

	/* see if we should scroll display */
	if (prefs.playback_tracked)
	{
	    /* make window at correct offset */
	    if (play_sample < ws->win_frst) /* window beyond play sample */
	    {
		gtk_wave_set_win_set(ws, play_sample, TRUE);
		scrolling = FALSE;
	    } else if (play_sample > ws->win_frst + ws->win_size)
		/* if we get so far behind, just redraw entire window */
		gtk_wave_set_win_set(ws, play_sample, TRUE);


	    if (prefs.playback_scrolled)
	    {
		if (!scrolling)
		{
		    scrolling = gtk_wave_set_scroll(ws, play_sample);
		    gtk_wave_set_play_set(ws, play_sample);
		} else {
		    gtk_wave_set_scroll(ws, play_sample);
		}
	    } else {
		gtk_wave_set_play_set(ws, play_sample);
	    }
	}

	/* see if we should update current time being played */
	if (prefs.show_time)
	    set_time(play_sample);
    }

    if (recording)
	dlgrecord_update_levels();

    return(TRUE);
}

static void reap_play_thread(void)
{
    gboolean	thread_return;

    if (play_thread == 0)
	return;

    pthread_join(play_thread, (void *)&thread_return);
    play_thread = 0;
    scrolling = FALSE;
    if (win_main)
    {
	gtk_wave_set_play_set(ws, play_sample);
	if (!thread_return)
	    modal_ask(FALSE, "OSS Error", "Ok", NULL,
			"Unable to open the /dev/dsp device and set\n"
			"the requested modes. Perhaps another program\n"
			"has the device open?");
	else
	    status("Editing %s", g_basename(ws->sb->file));
	gtk_widget_set_sensitive(GTK_WIDGET(ws), TRUE);
	lpeak = 0;
	rpeak = 0;
	dlglevels_update();
    }
}

static void reap_record_thread(void)
{
    gboolean	thread_return;

    if (record_thread == 0)
	return;

    pthread_join(record_thread, (void *)&thread_return);
    record_thread = 0;
    scrolling = FALSE;
    if (win_main)
    {
	if (!thread_return)
	    modal_ask(FALSE, "OSS Error", "Ok", NULL,
			"Unable to open the /dev/dsp device and set\n"
			"the requested modes. Perhaps another program\n"
			"has the device open?");
	gtk_widget_set_sensitive(GTK_WIDGET(ws), TRUE);
	lpeak = 0;
	rpeak = 0;
    }
}

static void load_file(void)
{
    if (snd_file_open(&snd_buffer))
    {
	pthread_create(&load_thread, NULL, dcache_attach, (void *)&snd_buffer);
    } else
	snd_file_close(&snd_buffer);
}


static void filesel_ok(GtkWidget *w, GtkFileSelection *fs)
{
    strcpy(snd_buffer.file, gtk_file_selection_get_filename(fs));
    gtk_widget_destroy(GTK_WIDGET(fs));
    load_file();
}



static gboolean warn_undo_loss(void)
{
    if (snd_buffer.undo_cnt > 0)
    {
	if (!modal_ask(TRUE, "Warning", "Ok", "Cancel",
		"There are currently %d outstanding edits\n"
		"which you will lose the ability to undo\n"
		"if you proceed.", snd_buffer.undo_cnt))
	    return FALSE;
    }
    return TRUE;
}

/*
 * Menu handlers
 */


/* File menu */

void on_new_file(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;
    if (warn_undo_loss())
	on_close(NULL, NULL);
}


void on_open(GtkMenuItem *menuitem, gpointer user_data)
{
    static GtkWidget	*wnd = NULL;

    if (file_open)
    {
	if (!warn_undo_loss())
	    return;
	on_close(NULL, NULL);
    }

    if (!wnd)
    {
	wnd = gtk_file_selection_new("Open wave file");
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(wnd));
	gtk_signal_connect(GTK_OBJECT(wnd), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &wnd);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(wnd)->ok_button), "clicked", GTK_SIGNAL_FUNC(filesel_ok), wnd);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(wnd)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(wnd));
    }

    if (!GTK_WIDGET_VISIBLE(wnd))
	gtk_widget_show(wnd);
}

void on_save(GtkMenuItem *menuitem, gpointer user_data)
{
    status("Saving...");
    snd_file_save(&snd_buffer);
}

void on_close(GtkMenuItem *menuitem, gpointer user_data)
{
    if (file_open)
    {
	on_stop(NULL, NULL);
	reap_play_thread();		/* wait for playing to stop */
	if (!warn_undo_loss())
	    return;
	edit_undos_free(ws);
	status("Saving...");
	snd_file_save(&snd_buffer);
	if (snd_buffer.dcache->dirty)
	    dcache_save(&snd_buffer);
	if (win_properties)
	    gtk_widget_destroy(win_properties);
	snd_file_close(&snd_buffer);
	gtk_wave_set_snd_detach(ws);
	file_open = FALSE;
	queue_cmd(CMD_WIDGET_HIDE, "Timebar");
	status("File closed");
    }

    window_title("%s v%s", PACKAGE, VERSION);
}

gboolean on_dlgproperties_destroy(GtkWidget *widget, GdkEvent *event,
					 gpointer user_data)
{
    win_properties = NULL;
    return FALSE;
}

void on_properties(GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *cl;
    char item[11][40] = {
	"Filename",
	"Channels",
	"Sample size",
	"Sample rate",
	"Samples",
	"Sample data",
	"Sample data mod 2352",
	"Time",
	"First cached zoom level",
	"Undos",
	"Markers" };
#   define PROPERTIES (sizeof(item)/sizeof(item[0]))
    char text[PROPERTIES][PATH_MAX];
    char *text_column[2];
    int i;

    if (win_properties != NULL)
	return;

    if (!file_open)
	return;

    win_properties = create_PropertiesDialog();

    i = 0;
    sprintf(text[i++], "%s", g_basename(ws->sb->file));
    sprintf(text[i++], "%d (%s)", ws->sb->info.channels, ws->sb->info.channels == 1 ? "Mono" : "Stereo");
    sprintf(text[i++], "%d", ws->sb->info.sample_bits);
    sprintf(text[i++], "%d Hz", ws->sb->info.sample_rate);
    sprintf(text[i++], "%d", ws->sb->info.samples);
    sprintf(text[i++], "%d", ws->sb->data_size);
    sprintf(text[i++], "%s", (ws->sb->data_size % 2352) ? "No" : "Yes");
    sample_fmt(SMPL_FMT_HHMMSS, ws->sb->info.samples,
	       &ws->sb->info, text[i++], PATH_MAX);
    sprintf(text[i++], "%d", 1<<ws->sb->dcache->zoom_level);
    sprintf(text[i++], "%d", ws->sb->undo_cnt);
    sprintf(text[i++], "%d", ws->sb->markers);

    cl = GTK_WIDGET(lookup_widget(win_properties, "CList"));
    for(i = 0; i < PROPERTIES; i++)
    {
	text_column[0] = item[i];
	text_column[1] = text[i];
	gtk_clist_append(GTK_CLIST(cl), text_column);
    };

    gtk_widget_show(win_properties);
#   undef PROPERTIES
}

void on_quit(GtkMenuItem *menuitem, gpointer user_data)
{
    win_main = NULL;
    on_close(NULL, NULL);
    gtk_main_quit();
}


/* Edit menu */

void on_cut(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_CUT, NULL, TRUE, &edit_thread);
}

void on_copy(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_COPY, NULL, TRUE, &edit_thread);
}

void on_paste(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_PASTE, NULL, TRUE, &edit_thread);
}

void on_select_all(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    gtk_wave_set_selection_set(ws, 0, ws->sb->info.samples, CHANNEL_ALL);
    queue_cmd(CMD_SELECTION_DIALOG_UPDATE, NULL);
}

void on_undo(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_UNDO, NULL, TRUE, &edit_thread);
}

void on_save_selection_as_ok(GtkWidget *w, GtkFileSelection *fs)
{
    static char filename[PATH_MAX];
    struct stat st;

    strcpy(filename, gtk_file_selection_get_filename(fs));
    gtk_widget_destroy(GTK_WIDGET(fs));

    /* prevent the user from saving over the current file */
    if (stat(filename, &st) == 0)
    {
	if (st.st_dev == ws->sb->st_dev && st.st_ino == ws->sb->st_ino)
	{
	    if (!modal_ask(TRUE, "Overwrite", "Ok", "Cancel",
			   "The file exists and is the one you are editing "
			   "do you really want to overwrite it?"))
		return;
	}
    }
    edit_cmd(ws, EDIT_SELECTION_SAVE, &filename, FALSE, NULL);
}

void on_save_selection_as(GtkMenuItem *menuitem, gpointer user_data)
{
    static GtkWidget *wnd = NULL;

    if (!file_open)
	return;

    if (!wnd)
    {
	wnd = gtk_file_selection_new("Save to WAV file");
	gtk_signal_connect(GTK_OBJECT(wnd), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &wnd);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(wnd)->ok_button), "clicked", GTK_SIGNAL_FUNC(on_save_selection_as_ok), wnd);
	gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(wnd)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(wnd));
    }

    if (!GTK_WIDGET_VISIBLE(wnd))
	gtk_widget_show(wnd);
}


/* Process menu */

void on_mute(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_MUTE, NULL, TRUE, &edit_thread);
}

void on_fade_in(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_FADE_LINEAR_IN, NULL, TRUE, &edit_thread);
}

void on_fade_out(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_FADE_LINEAR_OUT, NULL, TRUE, &edit_thread);
}

void on_interpolate(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_INTERP_LINEAR, NULL, TRUE, &edit_thread);
}

void on_insert_silence(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_INSERT_SILENCE, NULL, TRUE, &edit_thread);
}

void on_reverse(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_REVERSE, NULL, TRUE, &edit_thread);
}

void on_normalize(GtkMenuItem *menuitem, gpointer user_data)
{
    if (!file_open)
	return;

    edit_cmd(ws, EDIT_NORMALIZE, NULL, TRUE, &edit_thread);
}



/* Plugins menu */
#ifdef ENABLE_LADSPA
void on_aboutladspa(GtkMenuItem *menuitem, gpointer user_data)
{
    modal_ask(FALSE, "About LADSPA", "Ok", NULL,
	"LADSPA stands for Linux Audio Developers Simple Plugin Architecture\n"
	"which is a specification written by the members of the Linux Audio\n"
	"Developers mailing list\n");
}

void on_ladsparescan(GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *ladspa_menu;

    ladspa_menu = GTK_WIDGET(lookup_widget(win_main, "About_LADSPA"))->parent;
    ladspa_enumerate_menu(ladspa_menu);
}
#else
static char no_ladspa_msg[] =
    "This version of GNoise was compiled without LADSPA support.\n\n"
    "LADSPA support in GNoise is at a very alpha stage, but if\n"
    "would like to try it, obtain the sources and run configure\n"
    "with --enable-ladspa";

void on_aboutladspa(GtkMenuItem *menuitem, gpointer user_data)
{
    modal_ask(FALSE, "No LADSPA", "Ok", NULL, no_ladspa_msg);
}

void on_ladsparescan(GtkMenuItem *menuitem, gpointer user_data)
{
    modal_ask(FALSE, "No LADSPA", "Ok", NULL, no_ladspa_msg);
}
#endif /* ENABLE_LADSPA */


/* Help menu */

void on_about(GtkMenuItem *menuitem, gpointer user_data)
{
    const char gpl_terms[] = 
	"This program is free software; you can redistribute it and/or modify\n"
	"it under the terms of the GNU General Public License version 2 as \n"
	"published by the Free Software Foundation.\n \n"
	"This program is distributed in the hope that it will be useful, but\n"
	"WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
	"See the GNU General Public License for more details.";

#   ifndef ENABLE_GNOME
    char buf[1024];
    sprintf(buf, 
	"%s v%s\n"
	"Copyright (C) 1999-2001, Dwight Engen <dengen@users.sourceforge.net>\n\n"
	"%s",
	PACKAGE, VERSION, gpl_terms);
    modal_ask(FALSE, "About", "Ok", NULL, buf);
#   else
    const char *authors[] = {"Dwight Engen <dengen@users.sourceforge.net>", NULL};
    GtkWidget  *about = gnome_about_new(PACKAGE, VERSION,
				"Copyright (C) 1999-2001 Dwight Engen",
				authors,
				gpl_terms, "gnoise/gnoise-logo.png");
    gtk_widget_show(about);
#   endif
}




/*
 * Handle status bar gui items
 */

void on_cancel(GtkButton *button, gpointer user_data)
{
    progress_cancel = TRUE;
}


/*
 * Handle transport toolbar buttons
 */

void on_playlinebegin(GtkButton *button, gpointer user_data)
{
    if (!playing && file_open)
    {
	if ((prefs.selection_bond) &&
	    (ws->wd[0]->selection_start != ws->wd[0]->selection_end))
	{
	    gtk_wave_set_play_set(ws, ws->wd[0]->selection_start);
	} else {
	    gtk_wave_set_play_set(ws, 0);
	}
    }
}

void on_playlineend(GtkButton *button, gpointer user_data)
{
    if (!playing && file_open)
    {
	if ((prefs.selection_bond) &&
	    (ws->wd[0]->selection_start != ws->wd[0]->selection_end))
	{
	    gtk_wave_set_play_set(ws, ws->wd[0]->selection_end);
	} else {
	    gtk_wave_set_play_set(ws, ws->sb->info.samples);
	}
    }
}

void on_stop(GtkButton *button, gpointer user_data)
{
    if (playing && file_open)
    {
	playing_stop = TRUE;
	scrolling = FALSE;
	gtk_wave_set_play_set(ws, play_sample);
	gtk_widget_set_sensitive(GTK_WIDGET(ws), TRUE);
	lpeak = 0;
	rpeak = 0;
	dlglevels_update();
    }
}

void on_play(GtkButton *button, gpointer user_data)
{
    char start[SMPL_FMT_MAX];
    char end[SMPL_FMT_MAX];

    if (!playing && file_open)
    {
	/* If there's a channel bonded selection, we'll play it */
	if ((prefs.selection_bond) &&
	    (ws->wd[0]->selection_start != ws->wd[0]->selection_end))
	{
	    play_start = ws->wd[0]->selection_start;
	    play_end   = ws->wd[0]->selection_end;
	    if (playing_looped == LOOP_FILE)
		playing_looped = LOOP_SELECTION;
	} else {
	    play_start = ws->wd[0]->play_smpl;
	    play_end   = ws->sb->info.samples;
	}
	play_sample = play_start;

	gtk_widget_set_sensitive(GTK_WIDGET(ws), FALSE);
	sample_fmt(SMPL_FMT_PREFS, play_start, &ws->sb->info, start, sizeof(start));
	sample_fmt(SMPL_FMT_PREFS, play_end, &ws->sb->info, end, sizeof(end));
	status("Playing %s to %s", start, end);
	pthread_create(&play_thread, NULL, oss_play, (void *)ws);
    }
}

void on_playlooped(GtkButton *button, gpointer user_data)
{
    playing_looped = LOOP_FILE;
    on_play(button, user_data);
}


void on_record(GtkButton *button, gpointer user_data)
{
    gboolean record_file_valid;

    if (file_open)
    {
	status("Please close the current file before recording");
	return;
    }

    record_file_valid = dlgrecord(snd_buffer.file);

    /* wait till recorded file is flushed out */
    while(recording)
	usleep(100000);
    if (record_file_valid)
	load_file();
}

void on_toolbars(GtkMenuItem *menuitem, gpointer user_data)
{
    /* bring up settings dialog with toolbars tab showing */
    on_dlgprefs(NULL, (gpointer)3);
}

void on_zoom(GtkButton *button, gpointer user_data)
{
    scrolling = FALSE;
    if (file_open)
	gtk_wave_set_zoom(ws, (gboolean)user_data);

    /* if we're playing and zooming would make the play position be to the
     * left of the visible region, move the visible region so the playline
     * will be visible. (we don't have to worry about it being to the right
     * since during playback the idle func keeps the playline visible by
     * scrolling)
     */
    if (playing && play_sample < ws->win_frst)
	gtk_wave_set_win_set(ws, play_sample, TRUE);
}

void on_zoom_in(GtkMenuItem *menuitem, gpointer user_data)
{
    on_zoom(NULL, (gpointer)TRUE);
}

void on_zoom_out(GtkMenuItem *menuitem, gpointer user_data)
{
    on_zoom(NULL, (gpointer)FALSE);
}

void on_zoom_out_full(GtkMenuItem *menuitem, gpointer user_data)
{
    if (file_open)
	gtk_wave_set_zoom_out_full(ws, TRUE);
}

void on_zoom_selection(GtkButton *button, gpointer user_data)
{
    if (file_open)
	gtk_wave_set_zoom_selection(ws);
}

void on_tb_markers(GtkMenuItem *menuitem, gpointer user_data)
{
    GtkWidget *tb_markers;

    if (win_main == NULL)
	return;

#   if 0 //def ENABLE_GNOME
    tb_markers = GTK_WIDGET(gnome_app_get_dock_item_by_name(GNOME_APP(win_main),
						 "MarkersHBox"));
#   else
    tb_markers = lookup_widget(win_main, "MarkersTB");
#   endif

    if (GTK_CHECK_MENU_ITEM(menuitem)->active)
	gtk_widget_show(GTK_WIDGET(tb_markers));
    else
	gtk_widget_hide(GTK_WIDGET(tb_markers));
}


void
on_tb_transport                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
    GtkWidget *tb_transport;

    if (win_main == NULL)
	return;

#   if 0 //def ENABLE_GNOME
    tb_transport = GTK_WIDGET(gnome_app_get_dock_item_by_name(GNOME_APP(win_main),
						 "MarkersHBox"));
#   else
    tb_transport = lookup_widget(win_main, "TransportTB");
#   endif

    if (GTK_CHECK_MENU_ITEM(menuitem)->active)
	gtk_widget_show(GTK_WIDGET(tb_transport));
    else
	gtk_widget_hide(GTK_WIDGET(tb_transport));
}

/*
 * Handle markers toolbar buttons
 */

void on_markernew(GtkButton *button, gpointer user_data)
{
    gtk_wave_set_marker_add(ws, ws->wd[0]->play_smpl);
}


void on_markerdel(GtkButton *button, gpointer user_data)
{
    gtk_wave_set_marker_del_cur(ws);
}

void on_markernext(GtkButton *button, gpointer user_data)
{
    gtk_wave_set_marker_set_cur(ws, TRUE);
}

void on_markerprev(GtkButton *button, gpointer user_data)
{
    gtk_wave_set_marker_set_cur(ws, FALSE);
}
