/******************************************************************************\
 gnofin/data-if.c   $Revision: 1.16 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gtk/gtksignal.h>
#include "data-if.h"
#include "bankbook.h"
#include "account.h"
#include "record.h"
#include "record-type.h"
#include "data-events.h"


/******************************************************************************
 * Bankbook
 */

Bankbook *
if_bankbook_new ()
{
  trace ("");
  return BANKBOOK (bankbook_new ());
}

void
if_bankbook_destroy (Bankbook *book)
{
  trace ("");

  if (book)
    gtk_object_destroy (GTK_OBJECT (book));
}

const gchar *
if_bankbook_get_error (Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  return bankbook_get_error (book);
}

void
if_bankbook_dump (Bankbook *book)
{
  g_print ("\n");

  g_print ("D - %p {dirty=%u,err=%s,hist=%u}\n",
           book,
	   book->dirty,
	   book->lasterr,
	   book->remember_events);

  g_list_foreach (book->record_types, (GFunc) record_type_dump, NULL);

  string_cache_dump (&book->payee_cache, "P");
  string_cache_dump (&book->memo_cache, "M");
  string_cache_dump (&book->category_cache, "C");
  string_cache_dump (&book->link_cache, "L");

  g_list_foreach (book->accounts, (GFunc) account_dump, NULL);

  history_dump (&book->history);
  g_print ("\n");
}


/******************************************************************************
 * History
 */

void
if_bankbook_enable_history (Bankbook * book, gboolean enabled)
{
  trace ("");
  g_return_if_fail (book);

  book->remember_events = (enabled ? 1 : 0);

  /* It might lead to certain inconsistencies in the data structure
   * when history is again enabled if the history is not first cleared. 
   * So, we clear it whenever it is disabled. */
  if (!enabled)
    history_clear (&book->history);
}

gboolean
if_bankbook_can_undo (const Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, FALSE);

  return BANKBOOK_CAN_UNDO (book);
}

gboolean
if_bankbook_can_redo (const Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, FALSE);

  return BANKBOOK_CAN_REDO (book);
}

void
if_bankbook_undo (Bankbook *book)
{
  trace ("");
  g_return_if_fail (book);

  if (book->remember_events)
    history_undo (&book->history); 

  book->dirty--;
  
  gtk_signal_emit_by_name (GTK_OBJECT (book), "history_changed");
  gtk_signal_emit_by_name (GTK_OBJECT (book), "dirty_flag_changed");
}

void
if_bankbook_redo (Bankbook *book)
{
  trace ("");
  g_return_if_fail (book);

  if (book->remember_events)
    history_redo (&book->history);
  
  book->dirty++;

  gtk_signal_emit_by_name (GTK_OBJECT (book), "history_changed");
  gtk_signal_emit_by_name (GTK_OBJECT (book), "dirty_flag_changed");
}

void
if_bankbook_clear_history (Bankbook *book)
{
  trace ("");
  g_return_if_fail (book);

  if (book->remember_events)
    history_clear (&book->history);

  gtk_signal_emit_by_name (GTK_OBJECT (book), "history_changed");
}

gboolean
if_bankbook_is_dirty (const Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, FALSE);

  return BANKBOOK_IS_DIRTY (book);
}

void
if_bankbook_reset_dirty (Bankbook *book)
{
  trace ("");
  g_return_if_fail (book);

  book->dirty = 0;
  gtk_signal_emit_by_name (GTK_OBJECT (book), "dirty_flag_changed");
}

void
if_bankbook_enter_batch_mode (Bankbook *book)
{
  trace ("");
  g_return_if_fail (book);

  if (book->remember_events)
    history_enter_batch_mode (&book->history);
}

void
if_bankbook_leave_batch_mode (Bankbook *book)
{
  trace ("");
  g_return_if_fail (book);

  if (book->remember_events)
  {
    history_leave_batch_mode (&book->history);

    if (book->history.batch_mode == 0)
    {
      book->dirty++;

      gtk_signal_emit_by_name (GTK_OBJECT (book), "history_changed");
      gtk_signal_emit_by_name (GTK_OBJECT (book), "dirty_flag_changed");
    }
  }
}


/******************************************************************************
 * Insertions
 */

Account *
if_bankbook_insert_account (Bankbook *book, const AccountInfo *info)
{
  Account *account;

  trace ("");
  g_return_val_if_fail (book, NULL);
  g_return_val_if_fail (info, NULL);
  g_return_val_if_fail (info->name, NULL);

  /* Perform some checks to see if the account info is valid.
   * Namely, we make sure that the account name exists and is unique. */
  if (info->name[0] == '\0')
  {
    bankbook_set_error (book, _("The account name must not be empty"));
    return NULL;
  }
  account = get_account_by_name (book->accounts, info->name);
  if (account)
  {
    bankbook_set_error (book, _("The account name \"%s\" is already in use"), info->name);
    return NULL;
  }

  /* Create new account */
  account = account_new ();
  account_set_info (account, 0, info);

  /* Realize account insertion */
  {
    EventAccountInsert event = { account };
    realize_data_event (book, EVENT_ACCOUNT_INSERT, &event);
  }

  g_assert (account->ref_count > 1);

  /* We release our reference to the account */
  account_unref (account);
  return account;
}

RecordType *
if_bankbook_insert_record_type (Bankbook *book, const RecordTypeInfo *info)
{
  RecordType *type;
  
  trace ("");
  g_return_val_if_fail (book, NULL);
  g_return_val_if_fail (info, NULL);

  /* Verify that info->name is unique */
  type = get_record_type_by_name (book->record_types, info->name);
  if (type)
  {
    bankbook_set_error (book, _("The type name \"%s\" is already in use"), info->name);
    return NULL;
  }

  /* Verify that the sign makes sense... could help detect subtle bugs. */
  g_assert (info->sign != 3);

  /* Create new record type */
  type = record_type_new ();
  record_type_set_info (type, 0, info);

  /* Realize record type insertion */
  {
    EventRecordTypeInsert event = { type };
    realize_data_event (book, EVENT_RECORD_TYPE_INSERT, &event);
  }
  record_type_unref (type);
  return type;
}

void
if_bankbook_insert_category (Bankbook *book, const gchar *category_name)
{
  trace ("");
  g_return_if_fail (book);
  g_return_if_fail (category_name);

  /* The operation of adding a category is not saved in the history */
  cached_string_new_attached (category_name, &book->category_cache);
}


/******************************************************************************
 * Listings
 */

const GList *
if_bankbook_get_accounts (const Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  return book->accounts;
}

const GList *
if_bankbook_get_record_types (const Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  return book->record_types;
}

GList *
if_bankbook_get_payee_strings (const Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  return string_cache_get_strings (&book->payee_cache);
}

GList *
if_bankbook_get_memo_strings (const Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  return string_cache_get_strings (&book->memo_cache);
}

GList *
if_bankbook_get_category_strings (const Bankbook *book)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  return string_cache_get_strings (&book->category_cache);
}

GList *
if_bankbook_get_linked_acc_strings (const Bankbook *book, const Account *skip)
{
  GList *node, *strings=NULL;

  trace ("");
  g_return_val_if_fail (book, NULL);

  for (node=book->accounts; node; node=node->next)
  {
    Account *acc = LIST_DEREF (Account, node);
    if (skip != acc)
      strings = g_list_prepend (strings, acc->name);
  }

  for (node=book->link_cache.cache; node; node=node->next)
  {
    CachedString *cs = LIST_DEREF (CachedString, node);
    if (!skip || strcmp (skip->name, cs->string))
    {
      if (!g_list_find_custom (strings, cs->string, (GCompareFunc) strcmp))
	strings = g_list_prepend (strings, cs->string);
    }
  }
  return g_list_reverse (strings);
}


/******************************************************************************
 * Lookups
 */

Account *
if_bankbook_get_account_by_name (const Bankbook *book, const gchar *name)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  return get_account_by_name (book->accounts, name);
}

Account *
if_bankbook_get_account_by_index (const Bankbook *book, guint index)
{
  trace ("");
  g_return_val_if_fail (book, NULL);
  
  return (Account *) g_list_nth_data (book->accounts, index);
}

RecordType *
if_bankbook_get_record_type_by_name (const Bankbook *book, const gchar *name)
{
  trace ("");
  g_return_val_if_fail (book, NULL);

  return get_record_type_by_name (book->record_types, name);
}

RecordType *
if_bankbook_get_record_type_by_index (const Bankbook *book, guint index)
{
  trace ("");
  g_return_val_if_fail (book, NULL);
  
  return (RecordType *) g_list_nth_data (book->record_types, index);
}


/******************************************************************************
 * Account
 */

void
if_account_delete (Account *account)
{
  trace ("");
  g_return_if_fail (account);
  g_return_if_fail (account->parent);

  /* Realize account deletion */
  {
    EventAccountDelete event = { account };
    realize_data_event (account->parent, EVENT_ACCOUNT_DELETE, &event);
  }
}

gboolean
if_account_set_info (Account *account, guint mask, const AccountInfo *info)
{
  Bankbook *book;

  trace ("");
  g_return_val_if_fail (account, FALSE);
  g_return_val_if_fail (account->parent, FALSE);

  if (mask == 0)
    mask = ACCOUNT_ALL_WRITABLE_FIELDS;

  /* Check if anything is actually being changed, if not just return */
  {
    AccountInfo acc = {0};

    account_get_info (account, mask, &acc);

    if (!account_info_diff (&acc, info, mask))
      return TRUE;
  }

  book = account->parent;

  /* Perform some checks to see if the new account info is valid.
   * Namely, we make sure that, if a new account name is specified,
   * it is unique. */
  if (mask & ACCOUNT_FIELD_NAME)
  {
    if (strcmp (info->name, account->name) == 0)  /* no reason to set this field */
      mask &= ~ACCOUNT_FIELD_NAME;

    else if (get_account_by_name (book->accounts, info->name) != NULL)
    {
      bankbook_set_error (book, _("The account name \"%s\" is already in use"), info->name);
      return FALSE;
    }
  }

  /* Realize modification to account info */
  if (mask)
  {
    EventAccountSetInfo event = { account, mask, {0} };
    memcpy (&event.info, info, sizeof (*info));
    realize_data_event (book, EVENT_ACCOUNT_SET_INFO, &event);
  }
  return TRUE;
}

void
if_account_get_info (const Account *account, guint mask, AccountInfo *info)
{
  trace ("");
  account_get_info (account, mask, info);
}

Record *
if_account_insert_record (Account *account, const RecordInfo *info)
{
  Record *record;
  Bankbook *book;

  trace ("");
  g_return_val_if_fail (account, NULL);
  g_return_val_if_fail (info, NULL);
  g_return_val_if_fail (account->parent, NULL);

  book = account->parent;

  /* Verify that the date specified is valid. */
  g_return_val_if_fail (g_date_valid ((GDate *) &info->date), NULL);

  /* Verify that a pre-defined type has been specified. */
  g_return_val_if_fail (g_list_find (book->record_types, info->type), NULL);

  /* Create new record */
  record = record_new ();
  record_set_info (record, 0, info);

  /* Realize record insertion */
  {
    EventRecordInsert event = { account, record };
    realize_data_event (book, EVENT_RECORD_INSERT, &event);
  }
  record_unref (record);
  return record;
}

const GList *
if_account_get_records (const Account *account)
{
  trace ("");
  g_return_val_if_fail (account, NULL);

  return account->records;
}

gboolean
if_account_sort_records (Account *account, guint sort_field, gboolean sort_rev)
{
  Bankbook *book;

  trace ("");
  g_return_val_if_fail (account, FALSE);
  g_return_val_if_fail (account->parent, FALSE);

  book = account->parent;

  if (sort_field == RECORD_FIELD_CLEARED_BAL ||
      sort_field == RECORD_FIELD_OVERALL_BAL ||
      sort_field == RECORD_FIELD_LINKED_REC ||
      sort_field == RECORD_FIELD_LINKED_ACC)
  {
    bankbook_set_error (book, _("Sorting by the specified field is undefined."));
    return FALSE;
  }

  account_sort_records (account, sort_field, sort_rev);
  gtk_signal_emit_by_name (GTK_OBJECT (book), "record_list_invalidated", account);

  /* The action of sorting is not queued on the event history.
   * Its not something the user would want to undo/redo. */

  return TRUE;
}

Record *
if_account_get_record_by_index (const Account * account, guint index)
{
  trace ("");
  g_return_val_if_fail (account, NULL);

  return (Record *) g_list_nth_data (account->records, index);
}

gint
if_account_get_index (const Account * account)
{
  trace ("");
  return account_index (account);
}

Bankbook *
if_account_get_parent (const Account * account)
{
  trace ("");
  g_return_val_if_fail (account, NULL);

  return account->parent;
}

const gchar *
if_account_get_name (const Account * account)
{
  trace ("");
  g_return_val_if_fail (account, NULL);

  return account->name;
}

guint
if_account_get_last_number (const Account *account, const RecordType *type)
{
  GList *it;
  guint n = 0;

  trace ("");
  g_return_val_if_fail (account, 0);
  g_return_val_if_fail (type, 0);
  g_return_val_if_fail (type->numbered, 0);

  for (it=account->records; it; it=it->next)
  {
    Record *record = LIST_DEREF (Record, it);

    if (record->type == type)
      n = MAX (n, record->number);
  }
  return n;
}

gboolean
if_account_is_foreign (const Account *account)
{
  trace ("");
  g_return_val_if_fail (account, FALSE);

  return (account->foreign == 1);
}


/******************************************************************************
 * Record interface
 */

void
if_record_delete (Record *record)
{
  trace ("");
  g_return_if_fail (record);
  g_return_if_fail (record->parent);
  g_return_if_fail (record->parent->parent);

  /* Realize record deletion */
  {
    EventRecordDelete event = { record->parent, record };
    realize_data_event (record->parent->parent, EVENT_RECORD_DELETE, &event);
  }
}

gboolean
if_record_set_info (Record *record, guint mask, const RecordInfo *info)
{
  Bankbook *book;
  gboolean breaks_link = FALSE;

  trace ("");
  g_return_val_if_fail (record, FALSE);
  g_return_val_if_fail (record->parent, FALSE);
  g_return_val_if_fail (record->parent->parent, FALSE);
  g_return_val_if_fail (info, FALSE);

  book = record->parent->parent;

  if (mask == 0)
    mask = RECORD_ALL_WRITABLE_FIELDS;

  /* Check if anything is actually being changed, if not just return */
  {
    RecordInfo rec = {0};

    record_get_info (record, mask, &rec);

    if (!record_info_diff (&rec, info, mask))
      return TRUE;
  }

  /* Check to see if link would be broken
   */
  if (mask & RECORD_FIELD_LINKED_ACC_NAME &&
      record->type->linked && !record->link_broken)
  {
    Account *acc = get_account_by_name (book->accounts, info->linked_acc_name);
    if (acc == NULL || (acc == record->parent))
      breaks_link = TRUE;
  }
  trace ("breaks_link = %d", breaks_link);

  /* Realize modification to record info */
  {
    EventRecordSetInfo event = { record, mask, {0}, breaks_link };
    memcpy (&event.info, info, sizeof (*info));
    realize_data_event (book, EVENT_RECORD_SET_INFO, &event);
  }
  return TRUE;
}

void
if_record_get_info (const Record *record, guint mask, RecordInfo *info)
{
  trace ("");
  record_get_info (record, mask, info);
}

RecordType *
if_record_get_type (const Record *record)
{
  trace ("");
  g_return_val_if_fail (record, NULL);
  g_return_val_if_fail (record->type, NULL);

  return record->type;
}

gint
if_record_get_index (const Record *record)
{
  trace ("");
  return record_index (record);
}

Account *
if_record_get_parent (const Record *record)
{
  trace ("");
  g_return_val_if_fail (record, NULL);
  g_return_val_if_fail (record->parent, NULL);

  return record->parent;
}


/******************************************************************************
 * RecordType interface
 */

void
if_record_type_delete (RecordType *type)
{
  Bankbook *book;

  trace ("");
  g_return_if_fail (type);
  g_return_if_fail (type->parent);

  book = type->parent;

  /* Verify that the record type is not in use */
  if (type->usage_count > 0)
  {
    bankbook_set_error (book, _("The record type \"%s\" cannot be deleted while it is in use"), type->name);
    return;
  }

  /* Realize type deletion */
  {
    EventRecordTypeDelete event = { type };
    realize_data_event (book, EVENT_RECORD_TYPE_DELETE, &event);
  }
}

gboolean
if_record_type_set_info (RecordType *type, guint mask, const RecordTypeInfo *info)
{
  trace ("");
  g_return_val_if_fail (type, FALSE);
  g_return_val_if_fail (info, FALSE);

  /* Realize modification to record type info */
  {
    EventRecordTypeSetInfo event = { type, mask, {0} };
    event.mask = record_type_info_copy (&event.info, info, mask);
    realize_data_event (type->parent, EVENT_RECORD_TYPE_SET_INFO, &event);
  }
  return TRUE;
}

void
if_record_type_get_info (const RecordType *type, guint mask, RecordTypeInfo *info)
{
  trace ("");
  record_type_get_info (type, mask, info);
}

gint
if_record_type_get_index (const RecordType *type)
{
  trace ("");
  return record_type_index (type);
}

Bankbook *
if_record_type_get_parent (const RecordType *type)
{
  g_return_val_if_fail (type, NULL);
  g_return_val_if_fail (type->parent, NULL);

  return type->parent;
}

const gchar *
if_record_type_get_name (const RecordType *type)
{
  trace ("");
  g_return_val_if_fail (type, NULL);

  return type->name;
}

gboolean
if_record_type_is_numbered (const RecordType *type)
{
  trace ("");
  g_return_val_if_fail (type, FALSE);

  return (type->numbered == 1);
}

gboolean
if_record_type_is_linked (const RecordType *type)
{
  trace ("");
  g_return_val_if_fail (type, FALSE);

  return (type->linked == 1);
}

guint
if_account_info_copy (AccountInfo *dest, const AccountInfo *src, guint mask)
{
  return account_info_copy (dest, src, mask);
}

void
if_account_info_clear (AccountInfo *info, guint mask)
{
  account_info_clear (info, mask);
}

gboolean
if_account_info_diff (const AccountInfo *a, const AccountInfo *b, guint mask)
{
  return account_info_diff (a, b, mask);
}

guint
if_record_info_copy (RecordInfo *dest, const RecordInfo *src, guint mask)
{
  return record_info_copy (dest, src, mask);
}

void
if_record_info_clear (RecordInfo *info, guint mask)
{
  record_info_clear (info, mask);
}

gboolean
if_record_info_diff (const RecordInfo *a, const RecordInfo *b, guint mask)
{
  return record_info_diff (a, b, mask);
}

guint
if_record_type_info_copy (RecordTypeInfo *dest, const RecordTypeInfo *src, guint mask)
{
  return record_type_info_copy (dest, src, mask);
}

void
if_record_type_info_clear (RecordTypeInfo *info, guint mask)
{
  record_type_info_clear (info, mask);
}

gboolean
if_record_type_info_diff (const RecordTypeInfo *a, const RecordTypeInfo *b, guint mask)
{
  return record_type_info_diff (a, b, mask);
}

// vim: ts=8 sw=2
