/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 *  Authors: Jeffrey Stedfast <fejj@ximian.com>
 *
 *  Copyright 2002 Ximian, Inc. (www.ximian.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gmime/gmime.h>

#define ENABLE_ZENTIMER
#include "zentimer.h"

#if 1
static char *charsets[] = {
	"iso-8859-1",
	"iso-8859-2",
	"iso-8859-4",
	"iso-8859-5",
	"iso-8859-7",
	"iso-8859-8",
	"iso-8859-9",
	"iso-8859-13",
	"iso-8859-15",
	"koi8-r",
	"koi8-u",
	"windows-1250",
	"windows-1251",
	"windows-1252",
	"windows-1253",
	"windows-1254",
	"windows-1255",
	"windows-1256",
	"windows-1257",
	"euc-kr",
	"euc-jp",
	"iso-2022-kr",
	"iso-2022-jp",
	"utf-8",
};

static int num_charsets = sizeof (charsets) / sizeof (charsets[0]);
#endif


#if 1
static void
test_cache (void)
{
	GSList *node, *next, *open_cds = NULL;
	iconv_t cd;
	int i;
	
	srand (time (NULL));
	
	for (i = 0; i < 5000; i++) {
		const char *from, *to;
		int which;
		
		which = rand () % num_charsets;
		from = charsets[which];
		which = rand () % num_charsets;
		to = charsets[which];
		
		cd = g_mime_iconv_open (from, to);
		if (cd == (iconv_t) -1) {
			g_warning ("%d: failed to open converter for %s to %s",
				   i, from, to);
			continue;
		}
		
		which = rand () % 3;
		if (!which) {
			g_mime_iconv_close (cd);
		} else {
			open_cds = g_slist_prepend (open_cds, cd);
		}
	}
	
	node = open_cds;
	while (node) {
		next = node->next;
		cd = node->data;
		g_mime_iconv_close (cd);
		g_slist_free_1 (node);
		node = next;
	}
}
#endif


struct {
	const char *text;
	const char *charset;
} tests[] = {
	{ "Ətraflı", "utf-8" },                /* az */
	{ " ", "windows-cp1251" },      /* bg */
	{ "Cnjuge", "iso-8859-1" },                /* ca */
	{ "Avanceret sgning", "iso-8859-1" },      /* da */
	{ "Lschen", "iso-8859-1" },                /* de */
	{ "some text", "iso-8859-1" },              /* en */
	{ "piv", "iso-8859-15" },                /* fi */
	{ "Modifi", "iso-8859-1" },                /* fr */
	{ "Tidal", "iso-8859-1" },                 /* ga */
	{ "Fbrica", "iso-8859-1" },                /* gl */
	{ "Szem-Blyhv-A ", "iso-8859-2" },            /* hu */
	{ "Non c' corrispondenza", "iso-8859-1" }, /* it */
	{ "$(B>e5i8!:w(B", "euc-jp" },                   /* ja */
	{ "$(C0m1^(B $(C0K;v(B", "euc-kr" },                  /* ko */
	{ "Isami paieka", "iso-8859-13" },        /* lt */
	{ "Paplaint Meklana", "iso-8859-13" }, /* lv */
	{ "Kopiren", "iso-8859-15" },              /* nl */
	{ "ydelagd Sk", "iso-8859-1" },           /* nn */
	{ "Avansert sk", "iso-8859-1" },           /* no */
	{ "-Brda-A ksi-Bki-A adresowej", "iso-8859-2" }, /* pl */
	{ "C-Butare-A avansat-B-A ", "iso-8859-2" },      /* ro */
	{ "-L-A -L-A ", "koi8-r" },         /* ru */
	{ "Pokročilé hľadanie", "utf-8" },    /* sk */
	{ "Ga želite", "utf-8" },                  /* sl */
	{ "den nd?", "iso-8859-1" },              /* sv */
	{ "Geli-Mmi-A Arama", "iso-8859-9" },         /* tr */
	{ " ", "koi8-u" },        /* uk */
	{ "ɚ尋(I>(B", "utf-8" },         /* zh_TW */
	{ NULL, NULL }
};


static void
test_utils (void)
{
	char *utf8, *native;
	iconv_t cd;
	int i;
	
	for (i = 0; tests[i].text; i++) {
		cd = g_mime_iconv_open ("UTF-8", tests[i].charset);
		if (cd != (iconv_t) -1) {
			utf8 = g_mime_iconv_strdup (cd, tests[i].text);
			g_mime_iconv_close (cd);
			if (utf8 == NULL) {
				g_warning ("tests[%d]: failed to convert \"%s\" to UTF-8 from %s",
					   i, tests[i].text, tests[i].charset);
				continue;
			}
			
			cd = g_mime_iconv_open (tests[i].charset, "UTF-8");
			if (cd != (iconv_t) -1) {
				native = g_mime_iconv_strdup (cd, utf8);
				g_mime_iconv_close (cd);
				if (!native) {
					g_warning ("tests[%d]: failed to convert \"%s\" to %s from UTF-8",
						   i, tests[i].text, tests[i].charset);
				} else if (strcmp (tests[i].text, native)) {
					g_warning ("tests[%d]: there seems to have been some lossage\n"
						   "in the conversion back to the native charset:\n"
						   "\"%s\" != \"%s\"", i, tests[i].text, native);
				}
				
				g_free (native);
			} else {
				g_warning ("tests[%d]: failed to open conversion descriptor for UTF-8 to %s",
					   i, tests[i].charset);
			}
			
			g_free (utf8);
		} else {
			g_warning ("tests[%d]: failed to open conversion descriptor for %s to UTF-8",
				   i, tests[i].charset);
		}
	}
}

int main (int argc, char **argv)
{
	g_mime_iconv_init ();
	
	ZenTimerStart();
	test_cache ();
	ZenTimerStop();
	ZenTimerReport("test_cache()");
	
	test_utils ();
	
	return 0;
}
